﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs/fs_SaveDataTransfer.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fssystem/dbm/fs_DbmParameters.h>

#include "testFs_Integration_Api.h"

namespace nnt { namespace fs { namespace api {

    struct SaveDataTestParameters
    {
        int64_t storageSize;
        char rootPath[256];
        const char* mountName;
        int64_t saveDataId;
    };

    inline void SetupSaveData(SaveDataTestParameters* outParameters, nn::fs::SaveDataSpaceId spaceId) NN_NOEXCEPT
    {
        static const int64_t MarginSize = 512 * 1024 * 1024;

        static const int64_t StorageSizeMax = 2020ULL * 1024 * 1024;
        static const int64_t JournalSizeMax = 2020ULL * 1024 * 1024;
        NN_STATIC_ASSERT(StorageSizeMax + JournalSizeMax < static_cast<int64_t>(4) * 1024 * 1024 * 1024);

        static const char MountName[] = "TestMount";
        static const int64_t SystemSaveDataIdForTest = 0x8000000000004000;
        static const int64_t OwnerIdForTest = 0x0005000c10000000;

        nn::fs::DeleteSaveData(nn::fs::SaveDataSpaceId::User, SystemSaveDataIdForTest);
        nn::fs::DeleteSaveData(nn::fs::SaveDataSpaceId::System, SystemSaveDataIdForTest);
        nn::fs::DeleteSaveData(nn::fs::SaveDataSpaceId::SdUser, SystemSaveDataIdForTest);
        nn::fs::DeleteSaveData(nn::fs::SaveDataSpaceId::SdSystem, SystemSaveDataIdForTest);

        int64_t storageSize = StorageSizeMax;
        int64_t journalSize = JournalSizeMax;

        int64_t mediaSize = nn::fs::SaveDataTransferSizeCalculator::GetFreeSpaceSize(spaceId);
        NN_ABORT_UNLESS_GREATER(mediaSize, MarginSize);

        if (storageSize + journalSize > mediaSize - MarginSize)
        {
            storageSize = (mediaSize - MarginSize) / 2;
            journalSize = storageSize;

            NN_LOG("StorageSize is changed from %lld to %lld\n", StorageSizeMax, storageSize);
        }

        outParameters->mountName = MountName;
        outParameters->saveDataId = SystemSaveDataIdForTest;
        outParameters->storageSize = storageSize;

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateSystemSaveData(spaceId, SystemSaveDataIdForTest, OwnerIdForTest, storageSize, journalSize, 0));

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSystemSaveData(MountName, spaceId, SystemSaveDataIdForTest));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(MountName);
        };

        nn::util::SNPrintf(outParameters->rootPath, sizeof(outParameters->rootPath), "%s:/API_TEST", MountName);
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(outParameters->rootPath));
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CommitSaveData(MountName));
    }

    inline void CleanupSaveData(nn::fs::SaveDataSpaceId spaceId, const SaveDataTestParameters& parameters) NN_NOEXCEPT
    {
        nn::fs::DeleteSaveData(spaceId, parameters.saveDataId);
        if (spaceId == nn::fs::SaveDataSpaceId::SdUser)
        {
            nn::fs::DeleteSaveData(nn::fs::SaveDataSpaceId::SdSystem, parameters.saveDataId);
        }
    }

    inline FileSystemAttribute GetSaveDataFileSystemAttribute(const SaveDataTestParameters& parameters) NN_NOEXCEPT
    {
        FileSystemAttribute attribute(FsApiTestType::Integration);

        attribute.directoryNameLengthMax = nn::fssystem::dbm::MaxDirectoryNameLength;
        attribute.fileNameLengthMax      = nn::fssystem::dbm::MaxFileNameLength;
        // EntryNameLengthMax 以上のパスが shim で弾かれるので設定できるのは EntryNameLengthMax まで
        attribute.directoryPathLengthMax = nn::fs::EntryNameLengthMax;
        attribute.filePathLengthMax      = nn::fs::EntryNameLengthMax;

        attribute.fileSizeMax = parameters.storageSize - 16 * 16 * 1024; // 16 ブロックほどマージンを見る
        attribute.storageSize = parameters.storageSize;

        attribute.fileOpenMax      = 30;
        attribute.directoryOpenMax = 20;

        attribute.speedEmulationTypeFlag = static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::MmcStorageContextEnabled);

        attribute.isSupportedMultiBytePath = false;
        attribute.isSupportedGetFreeSpaceSize = true;
        attribute.isConcatenationFileSystem = false;
        attribute.isSaveFileSystem = true;
        attribute.isSupportedQueryRange = true;
        attribute.isSupportedFileOpenCheckOnCommit = true;

        attribute.fileSizeAlignment = 1;

        return attribute;
    }

}}}
