﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_FileSystemPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_StringUtil.h>

#include <fsa/fs_FileAccessor.h>

#include <nnt/fsApi/testFs_ApiCommon.h>

namespace nnt { namespace fs { namespace api {

    class FsApiIntegrationTestFile : public ITestFile, public nn::fs::detail::Newable
    {
    public:
        explicit FsApiIntegrationTestFile(nn::fs::FileHandle file) NN_NOEXCEPT
        {
            m_File.handle = file.handle;
            m_IsClosed = false;
            m_IsDirty = false;
        }
        virtual ~FsApiIntegrationTestFile() NN_NOEXCEPT NN_OVERRIDE
        {
            if( !m_IsClosed )
            {
                Close();
            }
        }
        virtual void Close() NN_NOEXCEPT NN_OVERRIDE
        {
            // Close を多重に呼ぶと死亡するのが正しい挙動
            // (後続のテストを実行したいので、とりあえず無視する)
            EXPECT_FALSE(m_IsClosed);

            if( !m_IsClosed )
            {
                // Close 前に Flush が呼ばれていないと死亡するのが正しい挙動
                // (後続のテストを実行したいので、とりあえず Flush しておく)
                EXPECT_FALSE(m_IsDirty);

                if( m_IsDirty )
                {
                    nn::fs::FlushFile(m_File);
                    m_IsDirty = false;
                }

                nn::fs::CloseFile(m_File);
                m_IsClosed = true;
            }
        }
        virtual Result Read(int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(nn::fs::ReadFile(m_File, offset, buffer, size, option));
            NN_RESULT_SUCCESS;
        }
        virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(nn::fs::ReadFile(m_File, offset, buffer, size));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result Read(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::ReadFile(outValue, m_File, offset, buffer, size, option);
        }
        virtual Result Read(size_t* outValue, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_RESULT_DO(nn::fs::ReadFile(outValue, m_File, offset, buffer, size));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result Write(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE
        {
            if( (option.flags & nn::fs::WriteOptionFlag_Flush) != 0 )
            {
                m_IsDirty = false;
            }

            NN_RESULT_DO(nn::fs::WriteFile(m_File, offset, buffer, size, option));

            if( (option.flags & nn::fs::WriteOptionFlag_Flush) == 0 )
            {
                m_IsDirty = true;
            }

            NN_RESULT_SUCCESS;
        }
        virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
        {
            m_IsDirty = false;
            NN_RESULT_DO(nn::fs::FlushFile(m_File));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::SetFileSize(m_File, size);
        }
        virtual nn::Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::GetFileSize(outValue, m_File);
        }
        virtual nn::Result OperateRange(void* outBuffer, size_t outBufferSize, nn::fs::OperationId operationId, int64_t offset, int64_t size, const void* inBuffer, size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE
        {
            // NOTE: shim 層で OperateRange が実装されるまでは、ここで実装
            NN_RESULT_DO(GetAccessor().OperateRange(
                outBuffer, outBufferSize, operationId, offset, size, inBuffer, inBufferSize
            ));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result OperateRange(
            nn::fs::OperationId operationId,
            int64_t offset,
            int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            // NOTE: shim 層で OperateRange が実装されるまでは、ここで実装
            NN_RESULT_DO(GetAccessor().OperateRange(nullptr, 0, operationId, offset, size, nullptr, 0));
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result QueryRange(nn::fs::QueryRangeInfo* outValue, int64_t offset, int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            // FIXME : FileServiceObjectAdapter::DoOperateRange でチェックしてほしい
            NN_FSP_REQUIRES(outValue != nullptr, nn::fs::ResultNullptrArgument());

            NN_RESULT_DO(nn::fs::QueryRange(outValue, m_File, offset, size));
            NN_RESULT_SUCCESS;
        }

    private:
        nn::fs::detail::FileAccessor& GetAccessor() NN_NOEXCEPT
        {
            return *reinterpret_cast<nn::fs::detail::FileAccessor*>(m_File.handle);
        }

    private:
        nn::fs::FileHandle m_File;
        bool m_IsDirty;
        bool m_IsClosed;
    };

    class FsApiIntegrationTestDirectory : public ITestDirectory, public nn::fs::detail::Newable
    {
    public:
        explicit FsApiIntegrationTestDirectory(nn::fs::DirectoryHandle dir) NN_NOEXCEPT
        {
            m_Dir.handle = dir.handle;
            m_IsClosed = false;
        }
        virtual ~FsApiIntegrationTestDirectory() NN_NOEXCEPT NN_OVERRIDE
        {
            if( !m_IsClosed )
            {
                nn::fs::CloseDirectory(m_Dir);
            }
        }
        virtual void Close() NN_NOEXCEPT NN_OVERRIDE
        {
            if( !m_IsClosed )
            {
                nn::fs::CloseDirectory(m_Dir);
                m_IsClosed = true;
            }
        }
        virtual nn::Result Read(int64_t* outValue, nn::fs::DirectoryEntry* outEntries, int64_t count) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::ReadDirectory(outValue, outEntries, m_Dir, count);
        }
        virtual nn::Result GetEntryCount(int64_t* outValue) NN_NOEXCEPT  NN_OVERRIDE
        {
            return nn::fs::GetDirectoryEntryCount(outValue, m_Dir);
        }
    private:
        nn::fs::DirectoryHandle m_Dir;
        bool m_IsClosed;
    };

    class FsApiIntegrationTestFileSystem : public ITestFileSystem, public nn::fs::detail::Newable
    {
    public:
        FsApiIntegrationTestFileSystem() NN_NOEXCEPT
        {
            m_MountName[0] = '\0';
        }
        explicit FsApiIntegrationTestFileSystem(const char* mountName) NN_NOEXCEPT
        {
            nn::util::Strlcpy(m_MountName, mountName, NN_ARRAY_SIZE(m_MountName));
        }
        virtual ~FsApiIntegrationTestFileSystem() NN_NOEXCEPT NN_OVERRIDE {}
        virtual nn::Result CreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::CreateFile(path, size, option);
        }
        virtual nn::Result CreateFile(const char* path, int64_t size) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::CreateFile(path, size);
        }
        virtual nn::Result DeleteFile(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::DeleteFile(path);
        }
        virtual nn::Result CreateDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::CreateDirectory(path);
        }
        virtual nn::Result DeleteDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::DeleteDirectory(path);
        }
        virtual nn::Result DeleteDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::DeleteDirectoryRecursively(path);
        }
        virtual nn::Result CleanDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::CleanDirectoryRecursively(path);
        }
        virtual nn::Result RenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::RenameFile(currentPath, newPath);
        }
        virtual nn::Result RenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::RenameDirectory(currentPath, newPath);
        }
        virtual nn::Result GetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::GetEntryType(outValue, path);
        }
        virtual nn::Result GetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::GetFreeSpaceSize(outValue, path);
        }
        virtual nn::Result GetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::GetTotalSpaceSize(outValue, path);
        }
        virtual nn::Result GetFileTimeStampRaw(nn::fs::FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::GetFileTimeStampRawForDebug(outTimeStamp, path);
        }
        virtual nn::Result OpenFile(std::unique_ptr<ITestFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT NN_OVERRIDE
        {
            if( outValue == nullptr )
            {
                NN_RESULT_DO(nn::fs::OpenFile(nullptr, path, mode));
            }
            else
            {
                nn::fs::FileHandle file;
                NN_RESULT_DO(nn::fs::OpenFile(&file, path, mode));
                std::unique_ptr<FsApiIntegrationTestFile> pFile(new FsApiIntegrationTestFile(file));
                *outValue = std::move(pFile);
            }
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result OpenDirectory(std::unique_ptr<ITestDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT NN_OVERRIDE
        {
            if( outValue == nullptr )
            {
                NN_RESULT_DO(nn::fs::OpenDirectory(nullptr, path, mode));
            }
            else
            {
                nn::fs::DirectoryHandle dir;
                NN_RESULT_DO(nn::fs::OpenDirectory(&dir, path, mode));
                std::unique_ptr<FsApiIntegrationTestDirectory> pDir(new FsApiIntegrationTestDirectory(dir));
                *outValue = std::move(pDir);
            }
            NN_RESULT_SUCCESS;
        }
        virtual nn::Result Commit() NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::Commit(m_MountName);
        }
        virtual nn::Result CommitSaveData() NN_NOEXCEPT NN_OVERRIDE
        {
            return nn::fs::CommitSaveData(m_MountName);
        }
        virtual nn::Result Flush() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT();
        }
        virtual nn::Result QueryEntry(char* outBuffer, size_t outBufferSize, const char* inBuffer, size_t inBufferSize, nn::fs::fsa::QueryId queryId, const char* path) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(outBuffer);
            NN_UNUSED(outBufferSize);
            NN_UNUSED(inBuffer);
            NN_UNUSED(inBufferSize);
            NN_UNUSED(queryId);
            NN_UNUSED(path);
            NN_ABORT();
        }

    private:
        char m_MountName[nn::fs::MountNameLengthMax + 1];
    };

    inline int64_t GetBisFreeSize(nn::fs::BisPartitionId partitionId) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountBis(nn::fs::GetBisMountName(partitionId), partitionId));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(nn::fs::GetBisMountName(partitionId));
        };

        char path[128];
        nn::util::SNPrintf(path, sizeof(path), "%s:%s", nn::fs::GetBisMountName(partitionId), "/");

        int64_t size;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFreeSpaceSize(&size, path));
        return size;
    }

}}}
