﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstddef> // nullptr, std::size_t
#include <cstdlib> // std::malloc, std::free

#include <nnt/codecUtil/testCodec_TestBase.h>
#include <nnt/codecUtil/testCodec_ObjectBase.h>

namespace nnt {
namespace codec {

// Derive and use
class PcmDataBufferBase
    : protected ObjectBase
{

protected:
    PcmDataBufferBase() NN_NOEXCEPT
        : m_PcmBufferAddress(nullptr)
        , pcmDataBuffer(nullptr)
        , pcmDataBufferSize(0)
        , pcmDataBufferSampleCount(0)
    {}

    virtual ~PcmDataBufferBase() NN_NOEXCEPT
    {
        if (nullptr != m_PcmBufferAddress)
        {
            std::free(m_PcmBufferAddress);
        }
        m_PcmBufferAddress = nullptr;
        pcmDataBuffer = nullptr;
    }

    virtual void Initialize(const TestParameterSet& parameterSet) NN_NOEXCEPT
    {
        // Define some aliases
        const int& sampleRate = parameterSet.sampleRate;
        const int& channelCount = parameterSet.channelCount;
        const int& frame = parameterSet.frame;
        NN_ASSERT(sampleRate != 0);
        NN_ASSERT(channelCount != 0);
        NN_ASSERT(frame != 0);
        // Initializing
        pcmDataBufferSampleCount = static_cast<std::size_t>(frame) * sampleRate / 1000 / 1000;
        pcmDataBufferSize = pcmDataBufferSampleCount * channelCount * sizeof(int16_t);
        m_PcmBufferAddress = std::malloc(pcmDataBufferSize);
        ASSERT_NE(nullptr, m_PcmBufferAddress);
        pcmDataBuffer = static_cast<int16_t*>(m_PcmBufferAddress);

        // Done
        m_IsInitialized = true;
    }

    virtual void Finalize() NN_NOEXCEPT
    {
        m_IsInitialized = false;
        if (nullptr != m_PcmBufferAddress)
        {
            std::free(m_PcmBufferAddress);
        }
        m_PcmBufferAddress = nullptr;
        pcmDataBuffer = nullptr;
        pcmDataBufferSize = 0;
        pcmDataBufferSampleCount = 0;
    }

private:
    void* m_PcmBufferAddress;

protected:
    int16_t* pcmDataBuffer;
    std::size_t pcmDataBufferSize;
    int pcmDataBufferSampleCount;
};

}} // nnt::codec
