﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cinttypes> // int32_t
#include <cstddef> // nullptr, std::size_t
#include <cstdlib> // std::malloc, std::free

#include <nnt/codecUtil/testCodec_TestBase.h>
#include <nnt/codecUtil/testCodec_ObjectBase.h>

namespace nnt {
namespace codec {

template <class T>
T* CreateAacDecoder(int32_t option) NN_NOEXCEPT;

template <>
NN_FORCEINLINE nn::codec::AacDecoder* CreateAacDecoder<nn::codec::AacDecoder>(int32_t option) NN_NOEXCEPT
{
    NN_UNUSED(option);
    return new nn::codec::AacDecoder;
}

template <class T>
NN_FORCEINLINE void DeleteAacDecoder(T* pDecoder) NN_NOEXCEPT
{
    if (pDecoder != nullptr)
    {
        delete pDecoder;
    }
}

template <typename DecoderType>
class AacDecoderBase
    : protected ObjectBase
{

protected:
    AacDecoderBase() NN_NOEXCEPT
        : decoder()
        , decoderWorkBufferAddress(nullptr)
        , decoderWorkBufferSize(0)
    {}

    virtual ~AacDecoderBase() NN_NOEXCEPT
    {
        if (nullptr != decoderWorkBufferAddress)
        {
            std::free(decoderWorkBufferAddress);
        }
        decoderWorkBufferAddress = nullptr;
    }

    virtual void Initialize(const TestParameterSet& parameterSet) NN_NOEXCEPT
    {
        // Define some aliases
        const int& sampleRate = parameterSet.sampleRate;
        const int& channelCount = parameterSet.channelCount;
        const nn::codec::Mpeg4AudioObjectType& type = parameterSet.type;
        NN_ASSERT(sampleRate != 0);
        NN_ASSERT(channelCount != 0);
        NN_ASSERT(type != nn::codec::Mpeg4AudioObjectType_Invalid);
        // Initializing
        decoderWorkBufferSize = decoder.GetWorkBufferSize(sampleRate, channelCount, type);
        decoderWorkBufferAddress = std::malloc(decoderWorkBufferSize);
        ASSERT_NE(nullptr, decoderWorkBufferAddress);
        // decoder.Initialize() を呼ぶかどうかはテスト次第なので、ここではしない。
        m_IsInitialized = true;
    }

    virtual void Finalize() NN_NOEXCEPT
    {
        m_IsInitialized = false;
        if (decoder.IsInitialized())
        {
            decoder.Finalize();
        }
        if (nullptr != decoderWorkBufferAddress)
        {
            std::free(decoderWorkBufferAddress);
        }
        decoderWorkBufferAddress = nullptr;
        decoderWorkBufferSize = 0;
    }

    DecoderType decoder;
    void* decoderWorkBufferAddress;
    std::size_t decoderWorkBufferSize;
};

}} // nnt::codec
