﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/atk.h>
#include <nn/mem.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_FormatString.h>
#include <nn/nn_Log.h>

namespace nnt{ namespace atk { namespace util {
////////////////////////////////////////////////////////////////////////////////////
// Path Utility
////////////////////////////////////////////////////////////////////////////////////

//---------------------------------------------------------------------------
//! @brief ターゲット名を取得します
//!
//! @param[out]  path      戻り値が true の場合、取得したパスを格納します。
//! @param[in]   pathSize  path のサイズです。
//!
//! @return      ターゲット名が取得出来れば true を、そうでない場合は false を返します。
//---------------------------------------------------------------------------
bool GetTargetName(char *path, size_t pathSize) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief ターゲット名を取得します
//!
//! @return ターゲット名の文字列を返します。
//---------------------------------------------------------------------------
const char* const GetTargetName() NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief 各テストの Contents ディレクトリを取得します
//!
//! @param[out]  path      戻り値が true の場合、取得したパスを格納します。
//! @param[in]   pathSize  path のサイズです。
//!
//! @return      テストのルートディレクトリが取得出来れば true を、そうでない場合は false を返します。
//---------------------------------------------------------------------------
bool GetContentsDirectoryPath(char *path, size_t pathSize) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief common サウンドアーカイブの Rom 上の位置を取得します
//!
//! @param[out]  path      戻り値が true の場合、取得したパスを格納します。
//! @param[in]   pathSize  path のサイズです。
//! @param[in]   mountName マウント名です
//!
//! @return      取得に成功すれば true を、そうでない場合は false を返します。
//---------------------------------------------------------------------------
bool GetCommonSoundArchivePath(char *path, size_t pathSize, const char* mountName) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief TestResults ディレクトリの位置を取得します
//!
//! @param[out]  path      戻り値が true の場合、取得したパスを格納します。
//! @param[in]   pathSize  path のサイズです。
//!
//! @return      取得に成功すれば true を、そうでない場合は false を返します。
//---------------------------------------------------------------------------
bool GetTestResultsDirectoryPath(char *path, size_t pathSize) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief Atk の TestBinaries ディレクトリの位置を取得します
//!
//! @param[out]  path      戻り値が true の場合、取得したパスを格納します。
//! @param[in]   pathSize  path のサイズです。
//!
//! @return      取得に成功すれば true を、そうでない場合は false を返します。
//---------------------------------------------------------------------------
bool GetAtkTestBinariesDirectoryPath(char *path, size_t pathSize) NN_NOEXCEPT;

////////////////////////////////////////////////////////////////////////////////////
// Fs Setup Utility
////////////////////////////////////////////////////////////////////////////////////
void UnmountFs(const char* mountName) NN_NOEXCEPT;

////////////////////////////////////////////////////////////////////////////////////
// HostPcFs Setup Utility
////////////////////////////////////////////////////////////////////////////////////
void MountHostPcFs(const char* mountName, const char* mountPath) NN_NOEXCEPT;

////////////////////////////////////////////////////////////////////////////////////
// RomFs Setup Utility
////////////////////////////////////////////////////////////////////////////////////
void MountRomFs(const char* mountName) NN_NOEXCEPT;

////////////////////////////////////////////////////////////////////////////////////
// File Operation Utility
////////////////////////////////////////////////////////////////////////////////////

//---------------------------------------------------------------------------
//! @brief filePath からファイルの存在するディレクトリを取得します。
//!
//! @param[out]  directoryName      取得されるディレクトリです。
//! @param[in]   directoryNameSize  directoryName のバッファサイズです。
//! @param[in]   filePath           ディレクトリ名を取得する対象のパスです。
//!
//! @return      取得に成功すれば true を、そうでない場合は false を返します
//---------------------------------------------------------------------------
bool GetFileDirectory(char* directoryName, const size_t directoryNameSize, const char* filePath) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief filePath からファイル名を取得します。
//!
//! @param[out]  fileName      取得されるファイル名です。
//! @param[in]   fileNameSize  fileName のバッファサイズです。
//! @param[in]   filePath      ファイル名を取得する対象のパスです。
//!
//! @return      取得に成功すれば true を、そうでない場合は false を返します
//---------------------------------------------------------------------------
bool GetFileName(char* fileName, const size_t fileNameSize, const char* filePath) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief path のディレクトリを作成します。
//!        既に path にディレクトリが存在していた場合、そのまま戻ります。
//!
//! @param[in]   path  作成するディレクトリの path です。
//---------------------------------------------------------------------------
void CreateDirectory(const char* path) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief path にサイズが size のファイルを作成します。
//!        既に path にファイルが存在していた場合、そのファイルを上書きして新たなファイルを作成します。
//!
//! @param[in]   path  作成するファイルの path です。
//! @param[in]   size  作成するファイルの size です。
//---------------------------------------------------------------------------
void CreateNewFile(const char* path, size_t size) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief sourcePath から destinationPath に対象のファイルをコピーします
//!        既に destinationPath にファイルが存在していた場合、そのファイルを上書きしてコピーします。
//!
//! @param[in]   sourcePath  コピー元のファイルの path です。
//! @param[in]   destinationPath  コピ－先のファイルの path です。
//---------------------------------------------------------------------------
void CopyFile(const char* sourcePath, const char* destinationPath) NN_NOEXCEPT;

////////////////////////////////////////////////////////////////////////////////////
// Command Utility
////////////////////////////////////////////////////////////////////////////////////

//---------------------------------------------------------------------------
//! @brief DriverCommand に対してダミーコマンドを送信します
//!
//! @param[out]  tag コマンドの tag を格納します。返り値が false の場合は不定値です。
//!
//! @return      コマンドの送信が成功した場合は true を、そうでない場合は false を返します。
//---------------------------------------------------------------------------
bool SendDummyCommand(uint32_t* tag) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief tag のコマンドの処理が終了しているかを調べます。
//!
//! @param[in]   tag コマンドの tag です。
//!
//! @return      処理が終わっていれば true を、そうでない場合は false を返します。
//---------------------------------------------------------------------------
bool IsCommandFinished(uint32_t tag) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief DriverCommand に対してリプライコマンドを送信します
//!
//! @param[out]  pIsCommandProcessed コマンドのリプライを示す bool 値のポインタです。コマンド処理が終わると、 true が代入されます。
//!
//! @return      コマンドの送信が成功した場合は true を、そうでない場合は false を返します。
//---------------------------------------------------------------------------
bool SendReplyCommand(bool* pIsCommandProcessed) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief コマンドを flush します。
//!
//! @param[in]   isForceFlush true の場合、コマンドの flush を強制的に行います。
//---------------------------------------------------------------------------
void FlushCommand(bool isForceFlush) NN_NOEXCEPT;

//---------------------------------------------------------------------------
//! @brief LowLevelVoiceCommandに積まれたコマンド数を返します。
//!
//! @return      コマンド数を返します。
//---------------------------------------------------------------------------
uint32_t GetVoiceCommandListCount() NN_NOEXCEPT;

////////////////////////////////////////////////////////////////////////////////////
// Other Utility
////////////////////////////////////////////////////////////////////////////////////
//! @brief 未初期化状態のバッファを確保します。
//! @param[in] allocator バッファの確保に使用するアロケータです。
//! @param[in] size 確保するバッファのサイズです。
//! @return 確保したバッファのアドレスです。
void* AllocateUninitializedMemory(nn::mem::StandardAllocator& allocator, size_t size);

//! @brief 未初期化状態のバッファを確保します。
//! @param[in] allocator バッファの確保に使用するアロケータです。
//! @param[in] size 確保するバッファのサイズです。
//! @param[in] alignment 確保するバッファのアライメントです。
//! @return 確保したバッファのアドレスです。
void* AllocateUninitializedMemory(nn::mem::StandardAllocator& allocator, size_t size, size_t alignment);

//! @brief ダミーコマンドを送信し、ドライバーコマンドが処理されるまで待ちます。
void WaitForProcessCommand() NN_NOEXCEPT;

//! @brief アーカイブプレイヤーを更新し、１ゲームフレーム相当(16ms) の SleepThread を行います。
void UpdateAndWait(nn::atk::SoundArchivePlayer& archivePlayer) NN_NOEXCEPT;

//! @brief アーカイブプレイヤーを更新し、指定の時間の SleepThread を行います。
void UpdateAndWait(nn::atk::SoundArchivePlayer& archivePlayer, const nn::TimeSpan& waitTime) NN_NOEXCEPT;

struct PreAtkTestArg
{
    bool isResourceExclusionCheckEnabled; //!< レンダラ更新との排他チェックの有効化フラグです。初期値は false です。
    PreAtkTestArg()
    {
        isResourceExclusionCheckEnabled = false;
    }
};

//! @brief Atk のテスト開始時に呼ばれる関数です。PreAtkTestArg は初期値で使用します。
void OnPreAtkTest() NN_NOEXCEPT;
//! @brief Atk のテスト開始時に呼ばれる関数です。
void OnPreAtkTest(PreAtkTestArg& arg) NN_NOEXCEPT;

////////////////////////////////////////////////////////////////////////////////////
// Math Utility
////////////////////////////////////////////////////////////////////////////////////
template<class T> class Statistics
{
public:
    Statistics() NN_NOEXCEPT
        : m_DataCount(0)
        , m_Sum(0)
        , m_Max(0)
        , m_Min(0)
    {
    }
    ~Statistics() NN_NOEXCEPT
    {
    }

    void AddData(T data) NN_NOEXCEPT
    {
        m_Sum = m_DataCount == 0 ? data : m_Sum + data;
        m_Max = m_DataCount == 0 ? data : std::max(m_Max, data);
        m_Min = m_DataCount == 0 ? data : std::min(m_Min, data);
        ++m_DataCount;
    }

    void ResetData() NN_NOEXCEPT
    {
        m_Sum = 0;
        m_Max = 0;
        m_Min = 0;
        m_DataCount = 0;
    }

    float GetMean() NN_NOEXCEPT
    {
        return m_DataCount == 0 ? 0 : static_cast<float>(m_Sum) / m_DataCount;
    }

    T GetMax() NN_NOEXCEPT
    {
        return m_Max;
    }

    T GetMin() NN_NOEXCEPT
    {
        return m_Min;
    }

    T GetSum() NN_NOEXCEPT
    {
        return m_Sum;
    }

private:
    int m_DataCount;
    T m_Sum;
    T m_Max;
    T m_Min;
};

}}}


