﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/teamcity/testTeamcity_Logger.h>
#include <nn/util/util_FormatString.h>
#include <nn/init.h>
#include <nn/nn_Assert.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nnt_Argument.h>

#include <nn/nn_Common.h>

#include <jerror.h>
#include <image_LibjpegHelper.h>

#include "testImageJpeg_Io.h"
#include "testImageJpeg_Tga.h"
#include "testImageJpeg_LibjpegApi.h"

/**
 @file libjpeg で使用するメモリ量の計算機能をテストします。

 @detail
 nn::image::detail::jpeg の提供する、「libjpeg が使用するワーク
 メモリ量を推定する機能」をテストします。

 入力は、ピクセルデータおよび、正常/ 異常な JPEG データと、
 規定されたパラメータをテスト対象とし、未定義のパラメータ
 などの入力はテストしません。
 これは、本機能 が SDK 利用者から直接に使用されないことに
 起因しています。

 テストに使用するデータについて、詳細は「JPEGライブラリ／テスト計画書」
 を参照してください。
 */

#if 1

namespace
{
struct DecodeArg
{
    nnt::image::jpeg::TestArgDecode decodeArg;
    nn::image::JpegStatus status;
    size_t workBufSizeExpected;
    bool hasExifExpected;
};

struct EncodeArg
{
    nnt::image::jpeg::TestArgEncode encodeArg;
    nn::image::JpegStatus status;
    size_t workBufSizeExpected;
};

/**
    @brief 常に成功するであろうデータでエラーが起きたときは必ずfalse(期待値ではない)を返す。
 */
bool ErrorOk(int errorCode, int reason)
{
    NN_LOG(">>> Checking error code %d, %d is 'OK 0'\n", errorCode, reason);
    return errorCode == 0 && reason == 0;
}
/**
    @brief メモリ不足検出用エラーハンドラ
 */
bool ErrorOutOfMemory(int errorCode, int reason)
{
    NN_LOG(">>> Checking error code %d, %d is 'JERR_OUT_OF_MEMORY %d' (without reason %d)\n",
        errorCode, reason,
        nn::image::detail::jpeg::JERR_OUT_OF_MEMORY, OUT_OF_MEM_REASON_OUTPUT);
    return (errorCode == nn::image::detail::jpeg::JERR_OUT_OF_MEMORY &&
        reason != OUT_OF_MEM_REASON_OUTPUT);
}
/**
    @brief DHT以上検出
 */
bool ErrorNoHuffTable(int errorCode, int reason)
{
    NN_UNUSED(reason);
    NN_LOG(">>> Checking error code %d is 'JERR_NO_HUFF_TABLE %d'\n",
        errorCode, nn::image::detail::jpeg::JERR_NO_HUFF_TABLE);
    return (errorCode == nn::image::detail::jpeg::JERR_NO_HUFF_TABLE);
}

static const char g_kExifPattern[] = {'E', 'x', 'i', 'f', '\0', '\0'};
static const uint8_t g_kExifBigEndian[] = {0x4D, 0x4D, 0x00, 0x2A};
static const uint8_t g_kExifLittleEndian[] = {0x49, 0x49, 0x2A, 0x00};
}

#define IABS(a) ((a) < 0? -(a): (a))

TEST(ImageLibjpegHelper, DecodeTest)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing helper API for JPEG decoder\n");

    static const char *files[] = {
        "./B-D-01_Baseline_420_1080P_16.jpg",
        "./B-D-02_Baseline_444_Odd_1_Optimised.jpg",
        "./B-D-03_Progressive_420_VGA_4.jpg",
        "./B-D-04_Progressive_422_Odd_8.jpg",
        "./B-D-05_Baseline_400_QVGA_1.jpg",
        "./B-D-06_Progressive_411_QVGA_2.jpg",
    };
    static const DecodeArg args[] = {
        {
            {ErrorOk, 16, {240, 135}},  // デコードパラメータ: 期待値, スケーリング値, スケーリング後のサイズ
            nn::image::JpegStatus_Ok, // メモリ量推定の結果
            21000, // メモリ量期待値 (参考情報)
            false // Exif の有無
        },
        {
            {ErrorOk, 1, {639, 479}},
            nn::image::JpegStatus_Ok,
            35000,
            true
        },
        {
            {ErrorOk, 4, {160, 120}},
            nn::image::JpegStatus_Ok,
            943000,
            false
        },
        {
            {ErrorOk, 8, {80, 60}},
            nn::image::JpegStatus_Ok,
            1250000,
            false
        },
        {
            {ErrorOk, 1, {320, 240}},
            nn::image::JpegStatus_Ok,
            22000,
            false
        },
        {
            {ErrorOk, 2, {160, 120}},
            nn::image::JpegStatus_Ok,
            250000,
            false
        },
    };

    void *mem = nullptr;

    for (int i = 0; i < static_cast<int>(sizeof(files) / sizeof(files[0])); i++)
    {
        nn::image::detail::JpegHeader header = {};

        NN_LOG("- Image[%d]: %s\n", i, files[i]);
        const char *file = files[i];

        NN_LOG(
            "  - {scale,width,height}={%d,%u,%u}\n",
            args[i].decodeArg.resolutionDenom,
            args[i].decodeArg.expectedDim.width,
            args[i].decodeArg.expectedDim.height);

        /* ----------------------------------------------
         * JPEGデータの入力
         */
        size_t size;
        ASSERT_TRUE(nnt::image::jpeg::io::GetAssetSize(&size, file));
        NN_LOG("  - Size: %u\n", size);

        uint8_t *jpeg = new uint8_t[size];
        ASSERT_TRUE(nnt::image::jpeg::io::ReadAsset(jpeg, size, file));

        /* ----------------------------------------------
         * ヘッダ解析
         */
        ASSERT_EQ(args[i].status, nn::image::detail::ExtractJpegHeader(&header, jpeg, size, args[i].decodeArg.resolutionDenom));

        /* ----------------------------------------------
         * ヘッダ内容の検査
         */
        // サイズ
        NN_LOG("  - Estimated dimension: {%d, %d}\n", header.dimension.width, header.dimension.height);
        ASSERT_EQ(args[i].decodeArg.expectedDim.width, header.dimension.width);
        ASSERT_EQ(args[i].decodeArg.expectedDim.height, header.dimension.height);

        // Exif
        EXPECT_EQ(args[i].hasExifExpected, header.hasExif);
        if (header.hasExif)
        {
            NN_LOG("  - Comparing Exif magic number\n");
            EXPECT_EQ(0, memcmp((uint8_t*)header.exifData - sizeof(g_kExifPattern), g_kExifPattern, sizeof(g_kExifPattern)));
            EXPECT_TRUE(memcmp(header.exifData, g_kExifBigEndian, sizeof(g_kExifBigEndian)) == 0 || memcmp(header.exifData, g_kExifLittleEndian, sizeof(g_kExifLittleEndian)) == 0);
        }

        NN_LOG("  - Estimated work buffer size: %d (expected=%d)\n", header.bufSizeForDecomp, args[i].workBufSizeExpected);
        mem = malloc(header.bufSizeForDecomp);

        /* ----------------------------------------------
         * JPEGデータのデコード (やらなくてもいいが、ついでにデコード機能をテストする)
         */
        unsigned int pixelSize = args[i].decodeArg.expectedDim.width * args[i].decodeArg.expectedDim.height;
        uint32_t *pixel = new uint32_t[pixelSize];

        {
            // ワークメモリ量 95% でエラーになるか？
            double ratio = 0.95;
            size_t memSizeSmall = (size_t)(header.bufSizeForDecomp * ratio);
            NN_LOG("  - Ensmallen size: %d (%.0f%%)\n", memSizeSmall, ratio * 100.0);
            nnt::image::jpeg::TestArgDecode arg = args[i].decodeArg;
            arg.errorHandler = ErrorOutOfMemory;

            NN_LOG("  - Test decoding with small work buffer\n");
            EXPECT_TRUE(nnt::image::jpeg::DecompressBinary(
                pixel, pixelSize * sizeof(uint32_t),
                jpeg, size, arg,
                mem, memSizeSmall));
        }

        // ワークメモリ量 100% でデコードできるか？
        NN_LOG("  - Test decoding with full work buffer\n");
        EXPECT_TRUE(nnt::image::jpeg::DecompressBinary(
            pixel, pixelSize * sizeof(uint32_t),
            jpeg, size, args[i].decodeArg,
            mem, header.bufSizeForDecomp));

        free(mem);
        mem = nullptr;

        /* ----------------------------------------------
            * ピクセルデータを正解データと比較
            */
        char truthPath[256] = {};
        nn::util::SNPrintf(truthPath, sizeof(truthPath) / sizeof(truthPath[0]), "%s.bin", file);
        NN_LOG("  - Comparing to truth: %s\n", truthPath);

        ASSERT_TRUE(nnt::image::jpeg::io::GetAssetSize(&size, truthPath));
        NN_LOG("  - Size of truth: %u\n", size);

        uint8_t *truth = new uint8_t[size];
        ASSERT_TRUE(nnt::image::jpeg::io::ReadAsset(truth, size, truthPath));

        uint64_t error = 0;
        for (unsigned int j = 0; j < size / sizeof(uint32_t) && j < 256; j++)
        {
            uint32_t color = pixel[j];
            uint8_t r = static_cast<uint8_t>((color & 0xFF000000) >> 24);
            uint8_t g = static_cast<uint8_t>((color & 0x00FF0000) >> 16);
            uint8_t b = static_cast<uint8_t>((color & 0x0000FF00) >> 8);
            error += (IABS(r - truth[j * 4]) + IABS(g - truth[j * 4 + 1]) + IABS(b - truth[j * 4 + 2]));
        }
        delete[] truth;
        NN_LOG("  - Comparison error: %d\n", error);
        EXPECT_EQ(0U, error);

        delete[] pixel;
        delete[] jpeg;
    }

    if (mem != nullptr)
    {
        free(mem);
    }
} // NOLINT(readability/fn_size)

TEST(ImageLibjpegHelper, DecodeErrorTest)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing helper API for JPEG decoder with BAD input\n");

    static const char *files[] = {
        "./z02_BadSos_UnknownColor(A-D-02).jpg",
        "./z04_BadSof_2color(A-D-02).jpg",
        "./z08_BadDht_NoDC0(A-D-02).jpg",
        "./z10_BadDqt_NoDqt1(A-D-02).jpg",
        "./z12_Terminated_InMarker(A-D-02).jpg",
    };
    static const DecodeArg args[] = {
        {
            {ErrorOk, 1, {640, 480}},
            nn::image::JpegStatus_WrongFormat,
            0, false
        },
        {
            {ErrorOk, 1, {640, 480}},
            nn::image::JpegStatus_WrongFormat,
            0, false
        },
        {
            {ErrorNoHuffTable, 1, {640, 480}},
            nn::image::JpegStatus_Ok,
            0, false
        },
        {
            {ErrorOk, 1, {640, 480}},
            nn::image::JpegStatus_WrongFormat,
            0, false
        },
        {
            {ErrorOk, 1, {640, 480}},
            nn::image::JpegStatus_WrongFormat,
            0, false
        },
    };

    void *mem = nullptr;

    for (int i = 0; i < static_cast<int>(sizeof(files) / sizeof(files[0])); i++)
    {
        nn::image::detail::JpegHeader header = {};

        NN_LOG("- Image[%d]: %s\n", i, files[i]);
        const char *file = files[i];

        NN_LOG(
            "  - {scale,width,height}={%d,%u,%u}\n",
            args[i].decodeArg.resolutionDenom,
            args[i].decodeArg.expectedDim.width,
            args[i].decodeArg.expectedDim.height);

        /* ----------------------------------------------
         * JPEGデータの入力
         */
        size_t size;
        ASSERT_TRUE(nnt::image::jpeg::io::GetAssetSize(&size, file));
        NN_LOG("  - Size: %u\n", size);

        uint8_t *jpeg = new uint8_t[size];
        ASSERT_TRUE(nnt::image::jpeg::io::ReadAsset(jpeg, size, file));

        /* ----------------------------------------------
         * ヘッダ解析
         */
        nn::image::JpegStatus extractResult = nn::image::detail::ExtractJpegHeader(&header, jpeg, size, args[i].decodeArg.resolutionDenom);
        NN_LOG("  - Returned value from parser: %d\n", extractResult);
        EXPECT_EQ(args[i].status, extractResult);

        if (extractResult == nn::image::JpegStatus_Ok)
        {
            /* ----------------------------------------------
             * ヘッダ内容の検査
             */
            // サイズ
            NN_LOG("  - Estimated dimension: {%d, %d}\n", header.dimension.width, header.dimension.height);
            ASSERT_EQ(args[i].decodeArg.expectedDim.width, header.dimension.width);
            ASSERT_EQ(args[i].decodeArg.expectedDim.height, header.dimension.height);

            // Exif
            EXPECT_EQ(args[i].hasExifExpected, header.hasExif);

            NN_LOG("  - Estimated work buffer size: %d (expected=%d)\n", header.bufSizeForDecomp, args[i].workBufSizeExpected);

            /* ----------------------------------------------
             * JPEGデータのデコード (やらなくてもいいが、ついでにデコード機能をテストする)
             */
            unsigned int pixelSize = args[i].decodeArg.expectedDim.width * args[i].decodeArg.expectedDim.height;
            uint32_t *pixel = new uint32_t[pixelSize];

            mem = malloc(header.bufSizeForDecomp);

            NN_LOG("  - Test decoding with full work buffer\n");
            EXPECT_TRUE(nnt::image::jpeg::DecompressBinary(
                pixel, pixelSize * sizeof(uint32_t),
                jpeg, size, args[i].decodeArg,
                mem, header.bufSizeForDecomp));

            free(mem);
            mem = nullptr;

            delete[] pixel;
        }

        delete[] jpeg;
    }

    if (mem != nullptr)
    {
        free(mem);
    }
} // NOLINT(readability/fn_size)

TEST(ImageLibjpegHelper, EncodeTest)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing helper API for JPEG encoder\n");

    static const char *files[] = {
        "./Noise_VGA.tga",
        "./Nature_Odd.tga",
        "./Noise_1080P.tga",
    };
    static const char *truths[] = {
        "./B-E-01_Noise_VGA_100_422.jpg",
        "./B-E-02_Nature_Odd_100_444.jpg",
        "./B-E-03_Noise_1080P_100_420.jpg"
    };
    static const EncodeArg args[] = {
        {
            {ErrorOk, 10, 2, 1},        // エンコードパラメータ: 期待値, 画質設定, 水平サンプリング比, 垂直サンプリング比
            nn::image::JpegStatus_Ok, // メモリ量推定の結果
            33000 // メモリ量期待値 (参考情報)
        },
        {
            {ErrorOk, 10, 1, 1},
            nn::image::JpegStatus_Ok,
            38000
        },
        {
            {ErrorOk, 10, 2, 2},
            nn::image::JpegStatus_Ok,
            78000
        },
    };

    void *mem = nullptr;

    for (int i = 0; i < static_cast<int>(sizeof(files) / sizeof(files[0])); i++)
    {
        NN_LOG("- Image[%d]: %s\n", i, files[i]);
        const char *file = files[i];

        NN_LOG(
            "  - {quality,Y H-sample,Y V-sample}={%d,%u,%u}\n",
            args[i].encodeArg.quality, args[i].encodeArg.yHSample, args[i].encodeArg.yVSample);

        /* ----------------------------------------------
         * ピクセルデータの入力
         */
        size_t size;
        ASSERT_TRUE(nnt::image::jpeg::io::GetAssetSize(&size, file));
        NN_LOG("  - Size: %u\n", size);

        uint8_t *tga = new uint8_t[size];
        ASSERT_TRUE(nnt::image::jpeg::io::ReadAsset(tga, size, file));

        nnt::image::jpeg::Dimension dim;
        ASSERT_TRUE(nnt::image::jpeg::Tga::GetDimension(&dim, tga, size));
        NN_LOG("  - Dimension: {%d, %d}\n", dim.width, dim.height);

        unsigned int pixelSize = dim.width * dim.height;
        uint32_t *pixel = new uint32_t[pixelSize];
        nnt::image::jpeg::Tga::GetPixels(
            pixel, pixelSize * sizeof(uint32_t),
            1, tga, size);
        delete[] tga;

        /* ----------------------------------------------
         * エンコード用ワークメモリの計算
         */
        nn::image::JpegSamplingRatio sampling;
        if (args[i].encodeArg.yHSample == args[i].encodeArg.yVSample)
        {
            sampling = (args[i].encodeArg.yHSample == 2)?
                nn::image::JpegSamplingRatio_420: nn::image::JpegSamplingRatio_444;
        }
        else
        {
            sampling = nn::image::JpegSamplingRatio_422;
        }
        switch (sampling)
        {
        case nn::image::JpegSamplingRatio_420:
            NN_LOG("  - Sampling ratio: 4:2:0\n");
            break;
        case nn::image::JpegSamplingRatio_422:
            NN_LOG("  - Sampling ratio: 4:2:2\n");
            break;
        case nn::image::JpegSamplingRatio_444:
            NN_LOG("  - Sampling ratio: 4:4:4\n");
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        nn::image::Dimension dimension = {dim.width, dim.height};

        size_t memSize;
        EXPECT_EQ(args[i].status, nn::image::detail::GetBufferSizeForJpegCompression(&memSize, dimension, sampling));

        NN_LOG("  - Estimated work buffer size: %d (expected=%d)\n", memSize, args[i].workBufSizeExpected);
        mem = malloc(memSize);

        /* ----------------------------------------------
         * JPEGデータへのエンコード (やらなくてもいいが、ついでにエンコード機能をテストする)
         */
        static const size_t kJpegSize = 10 * 1024 * 1024;
        uint8_t *jpeg = new uint8_t[kJpegSize];

        {
            // ワークメモリ量を 95% で試す。
            double ratio = 0.95;
            size_t memSizeSmall = (size_t)(memSize * ratio);
            NN_LOG("  - Ensmallen size: %d (%.0f%%)\n", memSizeSmall, ratio * 100.0);

            // 95% ではエラーになる (ことが多い)
            nnt::image::jpeg::TestArgEncode arg = args[i].encodeArg;
            arg.errorHandler = ErrorOutOfMemory;

            NN_LOG("  - Test encoding with small work buffer\n");
            size_t dummyOut;
            ASSERT_TRUE(
                nnt::image::jpeg::CompressBinary(
                &dummyOut, jpeg, kJpegSize, pixel, dim, arg, mem, memSizeSmall));
        }

        // ワークメモリ量 100% でエンコードできるか？
        NN_LOG("  - Test encoding with full work buffer\n");
        size_t actualOut;
        ASSERT_TRUE(
            nnt::image::jpeg::CompressBinary(
                &actualOut, jpeg, kJpegSize, pixel, dim, args[i].encodeArg, mem, memSize));
        delete[] pixel;
        free(mem);
        mem = nullptr;

        /* ----------------------------------------------
         * 正解データとの比較
         */
        NN_LOG("  - Comparing to truth: %s\n", truths[i]);

        ASSERT_TRUE(nnt::image::jpeg::io::GetAssetSize(&size, truths[i]));
        NN_LOG("  - Size of truth: %u\n", size);

        uint8_t *truth = new uint8_t[size];
        ASSERT_TRUE(nnt::image::jpeg::io::ReadAsset(truth, size, truths[i]));

        uint64_t error = 0;
        for (unsigned int j = 0; j < size; j++)
        {
            error += IABS(truth[j] - (reinterpret_cast<uint8_t*>(jpeg)[j]));
        }
        delete[] truth;
        NN_LOG("  - Comparison error %d\n", error);
        EXPECT_EQ(error, 0U);

        delete[] jpeg;
    }

    if (mem != nullptr)
    {
        free(mem);
    }
} // NOLINT(readability/fn_size)

#endif

#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1)
extern "C" void nninitStartup()
{
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    nn::os::SetMemoryHeapSize( MemoryHeapSize );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address;
    auto result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    NN_ASSERT( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}
#endif

/**
    @brief FS用アロケータ
 */
void* Allocate(size_t size)
{
    return malloc(size);
}

/**
    @brief FS用デアロケータ
 */
void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    std::free(p);
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    // TeamCity の表示を適切にするため、イベントリスナの登録を一旦すべて解除し、
    // ServiceMessageLogger -> デフォルトのイベントリスナ の順で登録し直す。
    ::testing::TestEventListeners& listeners = ::testing::UnitTest::GetInstance()->listeners();
    ::testing::TestEventListener* defaultResultPrinter = listeners.Release(listeners.default_result_printer());
    listeners.Append(new nnt::teamcity::ServiceMessageLogger());
    listeners.Append(defaultResultPrinter);

    nn::fs::SetAllocator(Allocate, Deallocate);
    nn::fs::MountHostRoot();

    if (nnt::image::jpeg::io::Initialize())
    {
        int testResult = RUN_ALL_TESTS();
        nnt::image::jpeg::io::Cleanup();
        nnt::Exit(testResult);
    }
    nnt::Exit(-1);
}
