﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/teamcity/testTeamcity_Logger.h>
#include <nn/util/util_FormatString.h>
#include <nn/init.h>
#include <nn/nn_Assert.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nnt_Argument.h>

// import from External/libjpeg/include
#include <jerror.h>

#include "testImageJpeg_Io.h"
#include "testImageJpeg_Tga.h"
#include "testImageJpeg_LibjpegApi.h"

/**
 @file libjpeg のエンコード/ デコード機能をテストします。

 @detail
 libjpeg の「ピクセルデータを JPEG エンコードする能力」と
 「 JPEG データをデコードする能力」をテストします。
 従って入力は、ピクセルデータおよび、正常/ 異常な JPEG データ
 と、規定されたパラメータをテスト対象とし、未定義のパラメータ
 などの入力はテストしません。
 これは、libjpeg が SDK 利用者から直接に使用されないことに
 起因しています。

 テストに使用するデータについて、詳細は「JPEGライブラリ／テスト計画書」
 を参照してください。
 */

namespace
{
/**
    @brief エンコード時の出力バッファ不足エラーになることを確認する。
 */
bool ErrorShortOutput(int errorCode, int reason)
{
    return (errorCode == nn::image::detail::jpeg::JERR_OUT_OF_MEMORY &&
        reason == OUT_OF_MEM_REASON_OUTPUT);
}

/**
    @brief JPEG ヘッダのセグメント長が想定外の場合に返るエラーになることを確認する。
 */
bool ErrorBadLength(int errorCode, int reason)
{
    NN_UNUSED(reason);
    return errorCode == nn::image::detail::jpeg::JERR_BAD_LENGTH;
}

/**
    @brief 常に成功するであろうデータでエラーが起きたときは必ずfalse(期待値ではない)を返す。
 */
bool ErrorOk(int errorCode, int reason)
{
    NN_LOG(">>> Checking returned value %d, %d is 'OK 0'\n", errorCode, reason);
    return (errorCode == 0 && reason == 0);
}

/**
    @brief FS用アロケータ
 */
void* Allocate(size_t size)
{
    return malloc(size);
}

/**
    @brief FS用デアロケータ
 */
void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    std::free(p);
}

#define IABS(a) ((a) < 0? -(a): (a))
}

/**
    @brief libjpeg のエンコード機能のテスト (正常系データ)
 */
TEST(ImageLibjpeg, EncodeTest)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing JPEG encode API of libjpeg\n");

    static const char *files[] = {
        "./White_VGA.tga",
        "./Black_720P.tga",
        "./Noise_1080P.tga",
        "./Nature_Odd.tga",
        "./Noise_1080P.tga",
    };
    static const char *truths[] = {
        "./A-E-01_White_VGA_10_420.jpg",
        "./A-E-02_Black_720P_50_422.jpg",
        "./A-E-03_Noise_1080P_100_444.jpg",
        "./A-E-04_Nature_Odd_100_420.jpg",
        "",
    };
    static const nnt::image::jpeg::TestArgEncode args[] = {
        {ErrorOk, 10, 2, 2},
        {ErrorOk, 50, 2, 1},
        {ErrorOk, 100, 1, 1},
        {ErrorOk, 100, 2, 2},
        {ErrorShortOutput, 100, 2, 2},
    };

    static const size_t kMemSize = 20 * 1024 * 1024;
    void *mem = malloc(kMemSize);

    for (int i = 0; i < static_cast<int>(sizeof(files) / sizeof(files[0])); i++)
    {
        NN_LOG("- Image[%d]: %s\n", i, files[i]);
        const char *file = files[i];

        NN_LOG("  - {quality,Y H-sample,Y V-sample}={%d,%u,%u}\n", args[i].quality, args[i].yHSample, args[i].yVSample);

        /* ----------------------------------------------
         * ピクセルデータの入力
         */
        size_t size;
        ASSERT_TRUE(nnt::image::jpeg::io::GetAssetSize(&size, file));
        NN_LOG("  - Size: %u\n", size);

        uint8_t *tga = new uint8_t[size];
        ASSERT_TRUE(nnt::image::jpeg::io::ReadAsset(tga, size, file));

        nnt::image::jpeg::Dimension dim;
        ASSERT_TRUE(nnt::image::jpeg::Tga::GetDimension(&dim, tga, size));
        NN_LOG("  - Dimension: {%d, %d}\n", dim.width, dim.height);

        unsigned int pixelSize = dim.width * dim.height;
        uint32_t *pixel = new uint32_t[pixelSize];
        nnt::image::jpeg::Tga::GetPixels(
            pixel, pixelSize * sizeof(uint32_t),
            1, tga, size);
        delete[] tga;

        /* ----------------------------------------------
         * JPEGデータへのエンコード
         */
        static const size_t kJpegSize = 10 * 1024 * 1024;
        uint8_t *jpeg = new uint8_t[kJpegSize];

        size_t actualOut;
        ASSERT_TRUE(
            nnt::image::jpeg::CompressBinary(
                &actualOut, jpeg,
                (args[i].errorHandler == ErrorShortOutput)?
                    1024: kJpegSize, // TORIAEZU 出力バッファ不足のテストケースでは出力 １KiB にする
                pixel, dim,
                args[i],
                mem, kMemSize));
        delete[] pixel;

        /* ----------------------------------------------
         * 正常時には正解データとの比較
         */
        if (args[i].errorHandler == ErrorOk)
        {
            NN_LOG("  - Comparing to truth: %s\n", truths[i]);

            ASSERT_TRUE(nnt::image::jpeg::io::GetAssetSize(&size, truths[i]));
            NN_LOG("  - Size: %u\n", size);

            uint8_t *truth = new uint8_t[size];
            ASSERT_TRUE(nnt::image::jpeg::io::ReadAsset(truth, size, truths[i]));

            uint64_t error = 0;
            for (unsigned int j = 0; j < size; j++)
            {
                error += IABS(truth[j] - (reinterpret_cast<uint8_t*>(jpeg)[j]));
            }
            delete[] truth;
            NN_LOG("  - Comparison error %d\n", error);
            EXPECT_EQ(error, 0U);
        }

        delete[] jpeg;
    }

    free(mem);
}

/**
    @brief libjpeg のデコード機能のテスト (正常系データ)
 */
TEST(ImageLibjpeg, DecodeTest)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing JPEG decode API of libjpeg\n");

    static const char *files[] = {
        "./A-D-01_Baseline_420_Odd_1.jpg",
        "./A-D-02_Baseline_422_VGA_8_JFIF.jpg",
        "./A-D-03_Progressive_422_Odd_16_JFIF.jpg",
        "./A-D-04_Progressive_444_1080P_1_APP1.jpg",
        "./A-D-05_Baseline_411_QVGA_1.jpg",
        "./A-D-06_Progressive_400_QVGA_2.jpg",
    };
    static const nnt::image::jpeg::TestArgDecode args[] = {
        {ErrorOk, 1, {639, 479}},
        {ErrorOk, 8, {80, 60}},
        {ErrorOk, 16, {80, 60}},
        {ErrorOk, 1, {1920, 1080}},
        {ErrorOk, 1, {320, 240}},
        {ErrorOk, 2, {160, 120}},
    };

    static const size_t kMemSize = 20 * 1024 * 1024;
    void *mem = malloc(kMemSize);

    for (int i = 0; i < static_cast<int>(sizeof(files) / sizeof(files[0])); i++)
    {
        NN_LOG("- Image[%d]: %s\n", i, files[i]);
        const char *file = files[i];

        NN_LOG("  - {scale,width,height}={%d,%u,%u}\n", args[i].resolutionDenom, args[i].expectedDim.width, args[i].expectedDim.height);

        /* ----------------------------------------------
         * JPEGデータの入力
         */
        size_t size;
        ASSERT_TRUE(nnt::image::jpeg::io::GetAssetSize(&size, file));
        NN_LOG("  - Size: %u\n", size);

        uint8_t *jpeg = new uint8_t[size];
        ASSERT_TRUE(nnt::image::jpeg::io::ReadAsset(jpeg, size, file));

        /* ----------------------------------------------
         * JPEGデータのデコード
         */
        unsigned int pixelSize = args[i].expectedDim.width * args[i].expectedDim.height;
        uint32_t *pixel = new uint32_t[pixelSize];
        ASSERT_TRUE(nnt::image::jpeg::DecompressBinary(
            pixel, pixelSize * sizeof(uint32_t),
            jpeg, size, args[i],
            mem, kMemSize));
        delete[] jpeg;

        /* ----------------------------------------------
         * ピクセルデータを正解データと比較
         */
        char truthPath[256] = {};
        nn::util::SNPrintf(truthPath, sizeof(truthPath) / sizeof(truthPath[0]), "%s.bin", file);
        NN_LOG("  - Comparing to truth: %s\n", truthPath);

        ASSERT_TRUE(nnt::image::jpeg::io::GetAssetSize(&size, truthPath));
        NN_LOG("  - Size: %u\n", size);

        uint8_t *truth = new uint8_t[size];
        ASSERT_TRUE(nnt::image::jpeg::io::ReadAsset(truth, size, truthPath));

        uint64_t error = 0;
        for (int j = 0;  j < static_cast<int>(size / sizeof(uint32_t)); j++)
        {
            uint32_t color = pixel[j];
            uint8_t r = static_cast<uint8_t>((color & 0xFF000000) >> 24);
            uint8_t g = static_cast<uint8_t>((color & 0x00FF0000) >> 16);
            uint8_t b = static_cast<uint8_t>((color & 0x0000FF00) >> 8);
            error += (IABS(r - truth[j * 4]) + IABS(g - truth[j * 4 + 1]) + IABS(b - truth[j * 4 + 2]));
        }
        delete[] truth;
        NN_LOG("  - Comparison error %d\n", error);
        EXPECT_EQ(error, 0U);

        delete[] pixel;
    }

    free(mem);
}

/**
    @brief libjpeg のデコード機能のテスト (異常系データ)
 */
TEST(ImageLibjpeg, DecodeErrorTest)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing JPEG decode API of libjpeg with BAD input\n");

    static const char *files[] = {
        "./z01_BadSos_4color(A-D-02).jpg",
        "./z06_BadSof_Small(A-D-02).jpg",
        "./z07_BadDht_DC0asAC0(A-D-02).jpg",
        "./z09_BadDqt_Force16bitDqt1(A-D-02).jpg",
        "./z13_Terminated_NoPix(A-D-02).jpg",
        "./z15_BadData_ScanLack(A-D-04).jpg",
    };
    static const nnt::image::jpeg::TestArgDecode args[] = {
        {ErrorBadLength, 1, {640, 480}},
        {ErrorOk, 1, {448, 240}},
        {ErrorOk, 1, {640, 480}},
        {ErrorOk, 1, {640, 480}},
        {ErrorOk, 1, {640, 480}},
        {ErrorBadLength, 1, {1920, 1080}},
    };

    static const size_t kMemSize = 20 * 1024 * 1024;
    void *mem = malloc(kMemSize);

    for (int i = 0; i < static_cast<int>(sizeof(files) / sizeof(files[0])); i++)
    {
        NN_LOG("- Image[%d]: %s\n", i, files[i]);
        const char *file = files[i];

        NN_LOG("  - {scale,width,height}={%d,%u,%u}\n", args[i].resolutionDenom, args[i].expectedDim.width, args[i].expectedDim.height);

        /* ----------------------------------------------
         * JPEGデータの入力
         */
        size_t size;
        ASSERT_TRUE(nnt::image::jpeg::io::GetAssetSize(&size, file));
        NN_LOG("  - Size: %u\n", size);

        uint8_t *jpeg = new uint8_t[size];
        ASSERT_TRUE(nnt::image::jpeg::io::ReadAsset(jpeg, size, file));

        /* ----------------------------------------------
         * JPEGデータのデコード
         */
        unsigned int pixelSize = args[i].expectedDim.width * args[i].expectedDim.height;
        uint32_t *pixel = new uint32_t[pixelSize];
        EXPECT_TRUE(nnt::image::jpeg::DecompressBinary(
            pixel, pixelSize * sizeof(uint32_t),
            jpeg, size, args[i],
            mem, kMemSize));
        delete[] jpeg;

        delete[] pixel;
    }

    free(mem);
}

#if defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1)
extern "C" void nninitStartup()
{
    // メモリヒープの全体サイズを設定する
    const size_t MemoryHeapSize = 128 * 1024 * 1024;
    nn::os::SetMemoryHeapSize( MemoryHeapSize );

    // メモリヒープから malloc で使用するメモリ領域を確保
    uintptr_t address;
    auto result = nn::os::AllocateMemoryBlock( &address, MemoryHeapSize );
    NN_ASSERT( result.IsSuccess() );

    // malloc 用のメモリ領域を設定する
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}
#endif

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    // TeamCity の表示を適切にするため、イベントリスナの登録を一旦すべて解除し、
    // ServiceMessageLogger -> デフォルトのイベントリスナ の順で登録し直す。
    ::testing::TestEventListeners& listeners = ::testing::UnitTest::GetInstance()->listeners();
    ::testing::TestEventListener* defaultResultPrinter = listeners.Release(listeners.default_result_printer());
    listeners.Append(new nnt::teamcity::ServiceMessageLogger());
    listeners.Append(defaultResultPrinter);

    nn::fs::SetAllocator(Allocate, Deallocate);
    nn::fs::MountHostRoot();

    if (nnt::image::jpeg::io::Initialize())
    {
        int testResult = RUN_ALL_TESTS();
        nnt::image::jpeg::io::Cleanup();
        nnt::Exit(testResult);
    }

    nnt::Exit(-1);
}
