﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/nn_Assert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Log.h>
#include <nn/nn_BitTypes.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/teamcity/testTeamcity_Logger.h>
#include <nnt/nnt_Argument.h>

#include <image_ExifParser.h>

namespace e = nn::image::detail;

namespace
{

/**
    @brief TIFF ヘッダのテストケース定義
 */
struct TiffTestCase
{
    const char *label;          ///< テスト名
    const nn::Bit8 *data;       ///< 解析対象のバイナリデータ
    uint16_t size;              ///< data の大きさ
    uint16_t expectedOffset;    ///< 解析された TIFF ヘッダ中の、IFD オフセットの期待値
};

/**
    @brief IFD ヘッダの解析のテストケース定義
 */
struct IfdHeaderTestCase
{
    const char *label;          ///< テスト名
    uint16_t total;             ///< Exif 情報全体の大きさ
    uint16_t offset;            ///< IFD へのオフセット
    const nn::Bit8 *data;       ///< IFD バイナリ
    uint16_t size;              ///< data の大きさ
    uint16_t expectedCount;     ///< 解析された IFD ヘッダ中の、タグ数の期待値
    uint16_t expectedOffset;    ///< 解析された IFD ヘッダ中の、次の IFD へのオフセットの期待値
};

/**
    @brief IFD 検索のテストケース定義
  */
struct IfdSearchTestCase
{
    const char *label;          ///< テスト名
    uint8_t offset;             ///< IFD へのオフセット
    const nn::Bit8 *ifd;        ///< IFD バイナリ
    uint16_t ifdSize;           ///< ifd の大きさ
    bool maskValueCount;        ///< 値の個数を"任意"にして検索するかのフラグ (クエリ全体に影響する)
    const e::ValuedIfdTag *tags;   ///< 検索クエリ + 結果の格納先
    uint16_t tagNum;            ///< クエリ数
};

/**
    @brief IFD 構築のテストケース定義
 */
struct IfdBuildTestCase
{
    const char *label;          ///< テスト名
    uint16_t offset;            ///< IFD へのオフセット
    const e::ValuedIfdTag *tags;   ///< IFD を構築するタグ
    uint16_t tagNum;            ///< タグ数
    const uint8_t *expected;    ///< 出力される IFD バイナリの期待値
    size_t expectedSize;        ///< expected の大きさ
};

/**
    @brief IFD 中のタグを検索するテスト
 */
void TestSearchingIfdTags(const e::BinaryIo &io, const IfdSearchTestCase *testCases, const int caseNum)
{
    // Exif バイナリ領域
    nn::Bit8 *exifData = (nn::Bit8*)malloc(0xFFFF);

    for (int testIndex = 0; testIndex < caseNum; testIndex++)
    {
        auto &test = testCases[testIndex];

        NN_ASSERT(test.offset + test.ifdSize <= 0xFFFF);

        NN_LOG("   - Test ID: %s\n", test.label);

        // 与えられた IFD を持つ Exif バイナリを生成 (IFD 以外の領域はダミー)
        e::ExifBinary bin = {exifData, static_cast<uint16_t>(test.offset + test.ifdSize), io};
        memcpy(exifData + test.offset, test.ifd, test.ifdSize);

        // IFD ヘッダの取得
        e::IfdHeader header = {};
        ASSERT_TRUE(e::ReadIfdHeader(&header, test.offset, bin));

        // クエリの構築
        auto *query = new e::IfdTag[test.tagNum];
        for (int i = 0; i < test.tagNum; i++)
        {
            query[i] = test.tags[i].tag;
            if (test.maskValueCount)
            {
                query[i].valueCount = 0;
            }
        }

        // タグの検索
        ASSERT_TRUE(e::SearchIfdTags(query, test.tagNum, header, bin));

        // 検索結果について、各タグの評価
        //  - 値の個数
        //  - 値の内容
        for (int i = 0; i < test.tagNum; i++)
        {
            auto &q = query[i];
            auto &exp = test.tags[i];

            if (exp.value.vUndefined == nullptr)
            {
                // タグが見つからない場合は、「見つからなかったときの値」になるかを検査
                EXPECT_EQ(0, q.valueCount);
                EXPECT_EQ(0x0000, q.valueOffset);
                continue;
            }

            // 値の個数を評価
            EXPECT_EQ(exp.tag.valueCount, q.valueCount);

            // 値の種類を評価
            if (exp.tag.type == e::IfdTagType_ShortOrLong)
            {
                EXPECT_TRUE(q.type == e::IfdTagType_Short || q.type == e::IfdTagType_Long);
            }
            else
            {
                EXPECT_EQ(exp.tag.type, q.type);
            }

            if (q.valueCount > 0)
            {
                // 値が存在する場合は内容を評価
                switch (q.type)
                {
                case e::IfdTagType_Byte:
                case e::IfdTagType_Ascii:
                case e::IfdTagType_Undefined:
                    EXPECT_EQ(0, memcmp(exp.value.vBytes, exifData + q.valueOffset, q.valueCount));
                    break;
                case e::IfdTagType_Short:
                    for (int j = 0; j < q.valueCount; j++)
                    {
                        uint16_t value = io.read2(exifData + q.valueOffset + j * sizeof(value));
                        EXPECT_EQ(exp.value.vShort[j], value);
                    }
                    break;
                case e::IfdTagType_Long:
                    for (int j = 0; j < q.valueCount; j++)
                    {
                        uint32_t value = io.read4(exifData + q.valueOffset + j * sizeof(value));
                        EXPECT_EQ(exp.value.vLong[j], value);
                    }
                    break;
                case e::IfdTagType_Rational:
                    for (int j = 0; j < q.valueCount; j++)
                    {
                        uint32_t numer = io.read4(exifData + q.valueOffset + (j * 2) * sizeof(numer));
                        uint32_t denom = io.read4(exifData + q.valueOffset + (j * 2 + 1) * sizeof(numer));
                        EXPECT_EQ(exp.value.vRational[j].numerator, numer);
                        EXPECT_EQ(exp.value.vRational[j].denominator, denom);
                    }
                    break;
                case e::IfdTagType_SignedLong:
                    for (int j = 0; j < q.valueCount; j++)
                    {
                        int32_t value = io.read4(exifData + q.valueOffset + j * sizeof(value));
                        EXPECT_EQ(exp.value.vSignedLong[j], value);
                    }
                    break;
                case e::IfdTagType_SignedRational:
                    for (int j = 0; j < q.valueCount; j++)
                    {
                        int32_t numer = io.read4(exifData + q.valueOffset + (j * 2) * sizeof(numer));
                        int32_t denom = io.read4(exifData + q.valueOffset + (j * 2 + 1) * sizeof(numer));
                        EXPECT_EQ(exp.value.vSignedRational[j].numerator, numer);
                        EXPECT_EQ(exp.value.vSignedRational[j].denominator, denom);
                    }
                    break;
                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
        }

        delete[] query;
    }
    free(exifData);
} // NOLINT(readability/fn_size)

/**
    @brief IFD 中のタグを検索するテスト (失敗するケース)
 */
void FailTestSearchingIfdTags(const e::BinaryIo &io, const IfdSearchTestCase *testCases, const int caseNum)
{
    nn::Bit8 *exifData = (nn::Bit8*)malloc(0xFFFF);

    for (int testIndex = 0; testIndex < caseNum; testIndex++)
    {
        auto &test = testCases[testIndex];

        NN_ASSERT(test.offset + test.ifdSize <= 0xFFFF);

        NN_LOG("   - Test ID: %s\n", test.label);

        e::ExifBinary bin = {exifData, static_cast<uint16_t>(test.offset + test.ifdSize), io};
        memcpy(exifData + test.offset, test.ifd, test.ifdSize);

        e::IfdHeader header = {};
        e::ReadIfdHeader(&header, test.offset, bin);

        // クエリの構築
        auto *query = new e::IfdTag[test.tagNum];
        for (int i = 0; i < test.tagNum; i++)
        {
            query[i] = test.tags[i].tag;
            if (test.maskValueCount)
            {
                query[i].valueCount = 0;
            }
        }

        // 検索
        EXPECT_FALSE(e::SearchIfdTags(query, test.tagNum, header, bin));

        delete[] query;
    }
    free(exifData);
}
}

/**
    @brief TIFF ヘッダの解析テスト
 */
TEST(ImageExifParser, TiffHeader)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing TIFF header functions in Exif parser\n");

    NN_LOG(" - Normal input (Big endian)\n");
    // ここからテストケース
    static const nn::Bit8 inputET01[14] = {0x4D, 0x4D, 0x00, 0x2A, 0x00, 0x00, 0x00, 0x08};
    static const uint16_t expectedET01 = 0x08;
    static const nn::Bit8 inputET02[20] = {0x49, 0x49, 0x2A, 0x00, 0x0E, 0x00, 0x00, 0x00};
    static const uint16_t expectedET02 = 0x0E;
    static const TiffTestCase normalCases[] = {
        {"E-T-01", inputET01, sizeof(inputET01), expectedET01},
        {"E-T-02", inputET02, sizeof(inputET02), expectedET02},
    };
    // ここまでテストケース

    for (const TiffTestCase &v: normalCases)
    {
        NN_LOG("   - Test ID: %s\n", v.label);

        // 与えられたバイナリから TIFF ヘッダを解析
        e::BinaryIo io = {};
        uint16_t offset = 0x00;
        ASSERT_TRUE(e::ReadTiffHeader(&io, &offset, v.data, v.size));
        EXPECT_EQ(v.expectedOffset, offset); // 次の IFD へのオフセットが一致していれば OK

        // TIFF ヘッダの生成
        e::TiffHeader header;
        e::InitializeTiffHeader(&header, v.data[0] == 0x4D? e::Endian_Big: e::Endian_Little);
        nn::Bit8 tiffOut[15] = {};
        e::WriteTiffHeader(tiffOut, sizeof(tiffOut), header);

        // 出力されたバイナリを、事前に準備したものと比較
        EXPECT_TRUE(std::memcmp(v.data, tiffOut, 4) == 0);

        // 再度解析し、最初に解析したものと一致するかを確認
        ASSERT_TRUE(e::ReadTiffHeader(&io, &offset, tiffOut, sizeof(tiffOut)));
        EXPECT_EQ(header.offset0thIfd, offset);
    }

    NN_LOG("--------------------------\n");
    NN_LOG(" - Abnormal input (Big endian)\n");
    // ここからテストケース
    static const nn::Bit8 inputET79[13] = {0x4D, 0x4D, 0x00, 0x2A, 0x00, 0x00, 0x00, 0x08};
    static const nn::Bit8 inputET78Base[] = {0x49, 0x49, 0x2A, 0x00, 0x08, 0x00, 0x00, 0x00};;
    nn::Bit8 *inputET78 = (nn::Bit8*)malloc(65528);
    std::memcpy(inputET78, inputET78Base, sizeof(inputET78Base));
    static const nn::Bit8 inputET77[14] = {0x4B, 0x4B, 0x00, 0x2A, 0x00, 0x00, 0x00, 0x08};
    static const nn::Bit8 inputET76[14] = {0x4D, 0x4D, 0x00, 0x2B, 0x00, 0x00, 0x00, 0x08};
    static const nn::Bit8 inputET75Base[] = {0x49, 0x49, 0x2A, 0x00, 0x00, 0x01, 0x00, 0x00};
    nn::Bit8 *inputET75 = (nn::Bit8*)malloc(256);
    std::memcpy(inputET75, inputET75Base, sizeof(inputET75Base));
    static const nn::Bit8 inputET74[14] = {0x49, 0x49, 0x2A, 0x00, 0x00, 0x00, 0x00, 0x00};
    static const TiffTestCase abnormals[] = {
        {"E-T-79", inputET79, sizeof(inputET79), 0x00},
        {"E-T-78", inputET78, 65528, 0x00},
        {"E-T-77", inputET77, sizeof(inputET77), 0x00},
        {"E-T-76", inputET76, sizeof(inputET76), 0x00},
        {"E-T-75", inputET75, 256, 0x00},
        {"E-T-74", inputET74, sizeof(inputET74), 0x00},
    };
    // ここまでテストケース

    for (const TiffTestCase &v: abnormals)
    {
        NN_LOG("   - Test ID: %s\n", v.label);

        // 失敗することを確認
        e::BinaryIo io = {};
        uint16_t offset = 0x00;
        ASSERT_FALSE(e::ReadTiffHeader(&io, &offset, v.data, v.size));
    }
    free(inputET78);
    free(inputET75);
} // NOLINT(readability/fn_size)

/**
    @brief IFD ヘッダの解析テスト
 */
TEST(ImageExifParser, IfdHeader)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing IFD header functions in Exif parser\n");

    e::TiffHeader tiff = {};
    e::InitializeTiffHeader(&tiff, e::Endian_Big);

    NN_LOG(" - Normal input (Big endian)\n");
    // ここからテストケース
    static const nn::Bit8 inputEI01[6] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x08};
    static const nn::Bit8 inputEI02[18] = {
        0x00, 0x01,
        0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0xFF, 0x00, 0x00, 0x00, // Tag0
        0x00, 0x00, 0x00, 0x00};
    static const nn::Bit8 inputEI03Base[] = {0x15, 0x53};
    static const nn::Bit8 inputEI04[6] = {0x00, 0x00, 0x00, 0x00, 0xFF, 0xF0};
    IfdHeaderTestCase normalCases[] = {
        {"E-I-01", 14, 8, inputEI01, sizeof(inputEI01), 0, 0x08},
        {"E-I-02", 26, 8, inputEI02, sizeof(inputEI01), 1, 0x00},
        {"E-I-03", 65522, 8, inputEI03Base, sizeof(inputEI03Base), 5459, 0x00},
        {"E-I-04", 65527, 65520, inputEI04, sizeof(inputEI01), 0, 0xFFF0},
    };
    // ここまでテストケース

    nn::Bit8 *exifData = (nn::Bit8*)malloc(0xFFFF);
    for (IfdHeaderTestCase &test: normalCases)
    {
        NN_LOG("   - Test ID: %s\n", test.label);

        std::memset(exifData, 0x00, 0xFFFF);
        std::memcpy(exifData + test.offset, test.data, test.size);

        // 与えられた IFD バイナリを解析
        const e::ExifBinary bin = {exifData, test.total, tiff.io};
        e::IfdHeader header;
        ASSERT_TRUE(e::ReadIfdHeader(&header, test.offset, bin));

        // IFD のオフセット, タグ数, 次の IFD へのオフセットが意図通りかを確認
        EXPECT_EQ(test.offset, header.offset);
        EXPECT_EQ(test.expectedCount, header.tagCount);
        EXPECT_EQ(test.expectedOffset, header.nextOffset);
    }

    NN_LOG("--------------------------\n");
    NN_LOG(" - Abnormal input (Bid endian)\n");
    // ここからテストケース
    static const nn::Bit8 inputEI89[6] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x08};
    static const nn::Bit8 inputEI88Base[18] = {0x15, 0x54};
    static const nn::Bit8 inputEI87[6] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x07};
    static const nn::Bit8 inputEI86[6] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x09};
    IfdHeaderTestCase abnormals[] = {
        {"E-I-89", 14, 9, inputEI89, sizeof(inputEI89), 0, 0x08},
        {"E-I-88", 65527, 8, inputEI88Base, sizeof(inputEI88Base), 5460, 0x00},
        {"E-I-87", 14, 8, inputEI87, sizeof(inputEI87), 0, 0x07},
        {"E-I-86", 14, 8, inputEI86, sizeof(inputEI86), 0, 0x09},
    };
    // ここまでテストケース

    for (IfdHeaderTestCase &test: abnormals)
    {
        NN_LOG("   - Test ID: %s\n", test.label);

        std::memset(exifData, 0x00, 0xFFFF);
        std::memcpy(exifData + test.offset, test.data, test.size);

        // 失敗することを確認
        const e::ExifBinary bin = {exifData, test.total, tiff.io};
        e::IfdHeader header;
        ASSERT_FALSE(e::ReadIfdHeader(&header, test.offset, bin));
    }
    free(exifData);
} // NOLINT(readability/fn_size)

/**
    @brief IFD からのタグ検索テスト
 */
TEST(ImageExifParser, IfdSearch)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing IFD search functions in Exif parser\n");

    e::TiffHeader tiff = {};

    NN_LOG(" - Normal input (Big endian)\n");
    e::InitializeTiffHeader(&tiff, e::Endian_Big);

    // ここからテストケース
    static const nn::Bit8 ifdES01Be[] = {
        0x00, 0x08,
        // {0x0000, Undef, 1, {0xFF}}
        0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x01, 0xFF, 0x00, 0x00, 0x00,
        // {0x0001, Byte, 1, {0xEE}}
        0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0xEE, 0x00, 0x00, 0x00,
        // {0x0002, Ascii, 2, "X"}
        0x00, 0x02, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x58, 0x00, 0x00, 0x00,
        // {0x0003, Short, 1, 0xBEEF}
        0x00, 0x03, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0xBE, 0xEF, 0x00, 0x00,
        // {0x0004, Long, 1, 0xDEADBEEF}
        0x00, 0x04, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0xDE, 0xAD, 0xBE, 0xEF,
        // {0x0005, Rational, 1, @110 = {0x01234567, 0x89ABCDEF}}
        0x00, 0x05, 0x00, 0x05, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x6E,
        // {0x0006, SignedLong, 1, 0xBEADCAFE}
        0x00, 0x06, 0x00, 0x09, 0x00, 0x00, 0x00, 0x01, 0xBE, 0xAD, 0xCA, 0xFE,
        // {0x0007, SignedRational, 2, @118 = {{0xF0E1D2C3, 0x4B5A6978}, {8796A5B4, 0x3C2D1E0F}}}
        0x00, 0x07, 0x00, 0x0A, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x76,
        0x00, 0x00, 0x00, 0x00,
        // {0x01234567, 0x89ABCDEF}
        0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
        // {{0xF0E1D2C3, 0x4B5A6978}, {8796A5B4, 0x3C2D1E0F}}
        0xF0, 0xE1, 0xD2, 0xC3, 0x4B, 0x5A, 0x69, 0x78,
        0x87, 0x96, 0xA5, 0xB4, 0x3C, 0x2D, 0x1E, 0x0F};
    NN_STATIC_ASSERT(sizeof(ifdES01Be) == 126);
    static const nn::Bit8 expectedES01_00 = 0xFF;
    static const nn::Bit8 expectedES01_01 = 0xEE;
    static const uint16_t expectedES01_03 = 0xBEEF;
    static const uint32_t expectedES01_04 = 0xDEADBEEF;
    static const e::Rational expectedES01_05 = {0x01234567, 0x89ABCDEF};
    static const int32_t expectedES01_06 = 0xBEADCAFE;
    static const e::SignedRational expectedES01_07[] = {{static_cast<int32_t>(0xF0E1D2C3), 0x4B5A6978}, {static_cast<int32_t>(0x8796A5B4), 0x3C2D1E0F}};
    static const e::ValuedIfdTag tagsES01[] = {
        {{0x0000, e::IfdTagType_Undefined, 1, 0x0000},     {&expectedES01_00}},
        {{0x0001, e::IfdTagType_Byte, 1, 0x0000},          {&expectedES01_01}},
        {{0x0002, e::IfdTagType_Ascii, 2, 0x0000},         {(uint8_t*)"X"}},
        {{0x0003, e::IfdTagType_Short, 1, 0x0000},         {(uint8_t*)&expectedES01_03}},
        {{0x0004, e::IfdTagType_Long, 1, 0x0000},          {(uint8_t*)&expectedES01_04}},
        {{0x0005, e::IfdTagType_Rational, 1, 0x0000},      {(uint8_t*)&expectedES01_05}},
        {{0x0006, e::IfdTagType_SignedLong, 1, 0x0000},    {(uint8_t*)&expectedES01_06}},
        {{0x0007, e::IfdTagType_SignedRational, 2, 0x0000}, {(uint8_t*)&expectedES01_07}},
    };

    static const nn::Bit8 ifdES02_04[] = {
        0x00, 0x03,
        0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x02, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x02, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00};
    static const nn::Bit8 expectedES02 = 0x00;
    static const e::ValuedIfdTag tagsES02[] = {
        {{0x0000, e::IfdTagType_Undefined, 1, 0x0000}, {(uint8_t*)&expectedES02}}
    };
    static const int8_t expectedES03[] = {0x02, 0x00};
    static const e::ValuedIfdTag tagsES03[] = {
        {{0x0002, e::IfdTagType_Ascii, 2, 0x0000}, {(uint8_t*)&expectedES03}}
    };
    static const e::ValuedIfdTag tagsES04[] = {
        tagsES02[0],
        tagsES03[0],
    };

    static const nn::Bit8 ifdES05[] = {
        0x00, 0x01,
        0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00};
    static const e::ValuedIfdTag tagsES05[] = {
        {{0x0001, e::IfdTagType_Byte, 1, 0x0000}, {nullptr}},
    };

    static const nn::Bit8 ifdES06[] = {
        0x00, 0x01,
        0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0xFF, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00};
    static const nn::Bit8 expectedES06 = 0xFF;
    static const e::ValuedIfdTag tagsES06[] = {
        {{0x0000, e::IfdTagType_Byte, 1, 0x0000}, {nullptr}},
        {{0x0001, e::IfdTagType_Byte, 1, 0x0000}, {&expectedES06}},
    };

    static const nn::Bit8 ifdES07[] = {
        0x00, 0x01,
        0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0xBE, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00};
    static const nn::Bit8 expectedES07 = 0xBE;
    static const e::ValuedIfdTag tagsES07[] = {
        {{0x0000, e::IfdTagType_Byte, 1, 0x0000}, {&expectedES07}},
        {{0x0001, e::IfdTagType_Byte, 1, 0x0000}, {nullptr}},
    };

    static const nn::Bit8 ifdES08_09[] = {
        0x00, 0x01,
        0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00};
    static const e::ValuedIfdTag tagsES08[] = {
        {{0x0000, e::IfdTagType_Ascii, 1, 0x0000}, {nullptr}},
    };
    static const e::ValuedIfdTag tagsES09[] = {
        {{0x0000, e::IfdTagType_ShortOrLong, 1, 0x0000}, {nullptr}},
    };

    static const nn::Bit8 ifdES10[] = {
        0x00, 0x01,
        0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x1A,
        0x00, 0x00, 0x00, 0x00,
        // TagType_Undefined x20
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
    static const e::ValuedIfdTag tagsES10[] = {
        {{0x0000, e::IfdTagType_Undefined, 19, 0x0000}, {nullptr}},
    };

    static const nn::Bit8 ifdES11Base[] = {0x15, 0x53};
    nn::Bit8 *ifdES11 = (nn::Bit8*)malloc(2 + 12 * 0x1553 + 4);
    std::memset(ifdES11, 0x00, 2 + 4 + 12 * 0x1553);
    std::memcpy(ifdES11, ifdES11Base, sizeof(ifdES11Base));
    for (int i = 0; i < 0x1553; i++)
    {
        nn::Bit8 tag[] = {
            static_cast<nn::Bit8>((i & 0xFF00) >> 8), static_cast<nn::Bit8>(i & 0x00FF), 0x00, 0x03, 0x00, 0x00,
            0x00, 0x01, static_cast<nn::Bit8>((i & 0xFF00) >> 8), static_cast<nn::Bit8>(i & 0x00FF), 0x00, 0x00};
        NN_STATIC_ASSERT(sizeof(tag) == 12);
        std::memcpy(ifdES11 + 2 + 12 * i, tag, sizeof(tag));
    }
    static const uint16_t expectedES11 = 0x1552;
    static const e::ValuedIfdTag tagsES11[] = {
        {{0x1552, e::IfdTagType_Short, 1, 0x0000}, {(uint8_t*)&expectedES11}}
    };

    static const nn::Bit8 ifdES12[] = {
        0x00, 0x03,
        0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x01, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0xBE, 0xEF, 0x00, 0x00,
        0x00, 0x02, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01,  0xB, 0xAD, 0xCA, 0xFE,
        0x00, 0x00, 0x00, 0x00,
    };
    static const nn::Bit8 expectedES12_01_dummy = 0xFA;
    static const uint16_t expectedES12_02 = 0xBEEF;
    static const uint32_t expectedES12_03 = 0xBADCAFE;
    static const e::ValuedIfdTag tagsES12[] = {
        {{0x0000, e::IfdTagType_Undefined, 0, 0x0000}, {&expectedES12_01_dummy}},
        {{0x0001, e::IfdTagType_ShortOrLong, 1, 0x0000}, {(uint8_t*)&expectedES12_02}},
        {{0x0002, e::IfdTagType_ShortOrLong, 1, 0x0000}, {(uint8_t*)&expectedES12_03}},
    };

    static const IfdSearchTestCase normalCases[] =
    {
        {"E-S-01", 8, ifdES01Be,    sizeof(ifdES01Be),  false, tagsES01, sizeof(tagsES01) / sizeof(e::ValuedIfdTag)},
        {"E-S-01'", 8, ifdES01Be,   sizeof(ifdES01Be),  true,  tagsES01, sizeof(tagsES01) / sizeof(e::ValuedIfdTag)},
        {"E-S-02", 8, ifdES02_04,   sizeof(ifdES02_04), false, tagsES02, sizeof(tagsES02) / sizeof(e::ValuedIfdTag)},
        {"E-S-02'", 8, ifdES02_04,  sizeof(ifdES02_04), true,  tagsES02, sizeof(tagsES02) / sizeof(e::ValuedIfdTag)},
        {"E-S-03", 8, ifdES02_04,   sizeof(ifdES02_04), false, tagsES03, sizeof(tagsES03) / sizeof(e::ValuedIfdTag)},
        {"E-S-03'", 8, ifdES02_04,  sizeof(ifdES02_04), true,  tagsES03, sizeof(tagsES03) / sizeof(e::ValuedIfdTag)},
        {"E-S-04", 8, ifdES02_04,   sizeof(ifdES02_04), false, tagsES04, sizeof(tagsES04) / sizeof(e::ValuedIfdTag)},
        {"E-S-04'", 8, ifdES02_04,  sizeof(ifdES02_04), true,  tagsES04, sizeof(tagsES04) / sizeof(e::ValuedIfdTag)},
        {"E-S-05", 8, ifdES05,      sizeof(ifdES05),    false, tagsES05, sizeof(tagsES05) / sizeof(e::ValuedIfdTag)},
        {"E-S-05'", 8, ifdES05,     sizeof(ifdES05),    true,  tagsES05, sizeof(tagsES05) / sizeof(e::ValuedIfdTag)},
        {"E-S-06", 8, ifdES06,      sizeof(ifdES06),    false, tagsES06, sizeof(tagsES06) / sizeof(e::ValuedIfdTag)},
        {"E-S-06'", 8, ifdES06,     sizeof(ifdES06),    true,  tagsES06, sizeof(tagsES06) / sizeof(e::ValuedIfdTag)},
        {"E-S-07", 8, ifdES07,      sizeof(ifdES07),    false, tagsES07, sizeof(tagsES07) / sizeof(e::ValuedIfdTag)},
        {"E-S-07'", 8, ifdES07,     sizeof(ifdES07),    true,  tagsES07, sizeof(tagsES07) / sizeof(e::ValuedIfdTag)},
        {"E-S-08", 8, ifdES08_09,   sizeof(ifdES08_09), false, tagsES08, sizeof(tagsES08) / sizeof(e::ValuedIfdTag)},
        {"E-S-08'", 8, ifdES08_09,  sizeof(ifdES08_09), true,  tagsES08, sizeof(tagsES08) / sizeof(e::ValuedIfdTag)},
        {"E-S-09", 8, ifdES08_09,   sizeof(ifdES08_09), false, tagsES09, sizeof(tagsES09) / sizeof(e::ValuedIfdTag)},
        {"E-S-09'", 8, ifdES08_09,  sizeof(ifdES08_09), true,  tagsES09, sizeof(tagsES09) / sizeof(e::ValuedIfdTag)},
        {"E-S-10", 8, ifdES10,      sizeof(ifdES10),    false, tagsES10, sizeof(tagsES10) / sizeof(e::ValuedIfdTag)},
        // E-S-10 は値の個数の違いで not found になるかを確かめるもののため、任意長テストは不可
        // {"E-S-10", 8, ifdES10,      sizeof(ifdES10),    true, tagsES10, sizeof(tagsES10) / sizeof(e::ValuedIfdTag)},
        {"E-S-11", 13, ifdES11,     2 + 12 * 0x1553 + 4,  false, tagsES11, sizeof(tagsES11) / sizeof(e::ValuedIfdTag)},
        {"E-S-11'", 13, ifdES11,    2 + 12 * 0x1553 + 4,  true,  tagsES11, sizeof(tagsES11) / sizeof(e::ValuedIfdTag)},
        {"E-S-12", 8, ifdES12,      sizeof(ifdES12),    false, tagsES12, sizeof(tagsES12) / sizeof(e::ValuedIfdTag)},
        {"E-S-12'", 8, ifdES12,     sizeof(ifdES12),    true,  tagsES12, sizeof(tagsES12) / sizeof(e::ValuedIfdTag)},
    };
    // ここまでテストケース

    TestSearchingIfdTags(tiff.io, normalCases, sizeof(normalCases) / sizeof(IfdSearchTestCase));
    ASSERT_FALSE(HasFatalFailure());

    free(ifdES11);

    NN_LOG("--------------------------\n");
    NN_LOG(" - Abnormal input (Big endian)\n");
    // ここからテストケース
    static const nn::Bit8 ifdES89Be[] = {
        0x00, 0x01,
        // {0x0000, Undef, 5, {0x1A}}
        0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x1A,
        0x00, 0x00, 0x00, 0x00,
        0xFE, 0xDC, 0xBA, 0x98};
    NN_STATIC_ASSERT(sizeof(ifdES89Be) == 22);
    static const nn::Bit8 expectedES89_dummy = 0xFA;
    static const e::ValuedIfdTag tagsES89[] = {
        {{0x0000, e::IfdTagType_Undefined, 5, 0x0000}, {&expectedES89_dummy}},
    };
    static const IfdSearchTestCase abnormalCases[] =
    {
        {"E-S-89", 8, ifdES89Be, sizeof(ifdES89Be), false, tagsES89, sizeof(tagsES89) / sizeof(e::ValuedIfdTag)},
        {"E-S-89'", 8, ifdES89Be, sizeof(ifdES89Be), true, tagsES89, sizeof(tagsES89) / sizeof(e::ValuedIfdTag)}
    };
    // ここまでテストケース

    FailTestSearchingIfdTags(tiff.io, abnormalCases, sizeof(abnormalCases) / sizeof(IfdSearchTestCase));
    ASSERT_FALSE(HasFatalFailure());


    NN_LOG("--------------------------\n");
    NN_LOG(" - Normal input (Little endian)\n");
    e::InitializeTiffHeader(&tiff, e::Endian_Little);
    // ここからテストケース
    static const nn::Bit8 ifdES01Le[] = {
        0x08, 0x00,
        // {0x0000, Undef, 1, {0xFF}}
        0x00, 0x00, 0x07, 0x00, 0x01, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x00, 0x00,
        // {0x0001, Byte, 1, {0xEE}}
        0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0xEE, 0x00, 0x00, 0x00,
        // {0x0002, Ascii, 2, "X"}
        0x02, 0x00, 0x02, 0x00, 0x02, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00,
        // {0x0003, Short, 1, 0xBEEF}
        0x03, 0x00, 0x03, 0x00, 0x01, 0x00, 0x00, 0x00, 0xEF, 0xBE, 0x00, 0x00,
        // {0x0004, Long, 1, 0xDEADBEEF}
        0x04, 0x00, 0x04, 0x00, 0x01, 0x00, 0x00, 0x00, 0xEF, 0xBE, 0xAD, 0xDE,
        // {0x0005, Rational, 1, @106 = {0x01234567, 0x89ABCDEF}}
        0x05, 0x00, 0x05, 0x00, 0x01, 0x00, 0x00, 0x00, 0x6E, 0x00, 0x00, 0x00,
        // {0x0006, SignedLong, 1, 0xBEADCAFE}
        0x06, 0x00, 0x09, 0x00, 0x01, 0x00, 0x00, 0x00, 0xFE, 0xCA, 0xAD, 0xBE,
        // {0x0007, SignedRational, 2, @114 = {{0xF0E1D2C3, 0x4B5A6978}, {8796A5B4, 0x3C2D1E0F}}}
        0x07, 0x00, 0x0A, 0x00, 0x02, 0x00, 0x00, 0x00, 0x76, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00,
        // {0x01234567, 0x89ABCDEF}
        0x67, 0x45, 0x23, 0x01, 0xEF, 0xCD, 0xAB, 0x89,
        // {{0xF0E1D2C3, 0x4B5A6978}, {8796A5B4, 0x3C2D1E0F}}
        0xC3, 0xD2, 0xE1, 0xF0, 0x78, 0x69, 0x5A, 0x4B,
        0xB4, 0xA5, 0x96, 0x87, 0x0F, 0x1E, 0x2D, 0x3C};
    NN_STATIC_ASSERT(sizeof(ifdES01Le) == 126);
    static const IfdSearchTestCase normalCasesLe[] =
    {
        {"E-S-01 (LE)", 8, ifdES01Le, sizeof(ifdES01Le), false, tagsES01, sizeof(tagsES01) / sizeof(e::ValuedIfdTag)},
        {"E-S-01' (LE)", 8, ifdES01Le, sizeof(ifdES01Le), true, tagsES01, sizeof(tagsES01) / sizeof(e::ValuedIfdTag)},
    };
    // ここまでテストケース

    TestSearchingIfdTags(tiff.io, normalCasesLe, sizeof(normalCasesLe) / sizeof(IfdSearchTestCase));
    ASSERT_FALSE(HasFatalFailure());


    NN_LOG("--------------------------\n");
    NN_LOG(" - Abnormal input (Little endian)\n");
    // ここからテストケース
    static const nn::Bit8 ifdES89Le[] = {
        0x01, 0x00,
        0x00, 0x00, 0x07, 0x00, 0x05, 0x00, 0x00, 0x00, 0x1A, 0x00, 0x00, 0x00, // {0x0000, Undef, 5, {0x1A}}
        0x00, 0x00, 0x00, 0x00,
        0x76, 0x98, 0xBA, 0xDC};
    NN_STATIC_ASSERT(sizeof(ifdES89Le) == 22);
    static const IfdSearchTestCase abnormalCasesLe[] =
    {
        {"E-S-89", 8, ifdES89Le, sizeof(ifdES89Le),  false, tagsES89, sizeof(tagsES89) / sizeof(e::ValuedIfdTag)}
    };
    // ここまでテストケース

    FailTestSearchingIfdTags(tiff.io, abnormalCasesLe, sizeof(abnormalCasesLe) / sizeof(IfdSearchTestCase));
    ASSERT_FALSE(HasFatalFailure());
} // NOLINT(readability/fn_size)

/**
    @brief IFD の構築テスト
 */
TEST(ImageExifParser, IfdCreate)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing IFD create functions in Exif parser\n");

    NN_LOG(" - Normal input (Big endian)\n");
    e::TiffHeader tiff = {};
    e::InitializeTiffHeader(&tiff, e::Endian_Big);
    // ここからテストケース
    static const e::ValuedIfdTag *inputEC01 = nullptr;
    static const nn::Bit8 expectedEC01[6] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x0E};

    static const e::ValuedIfdTag inputEC02[] = {
        {{0x0000, e::IfdTagType_Undefined, 0, 0x0000}, {nullptr}},
    };
    static const nn::Bit8 expectedEC02[] = {
        0x00, 0x01,
        0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x1A};

    static const nn::Bit8 valueEC03_00 = 0xFF;
    static const nn::Bit8 valueEC03_01 = 0xEE;
    static const uint16_t valueEC03_03 = 0xBEEF;
    static const uint32_t valueEC03_04 = 0xDEADBEEF;
    static const e::Rational valueEC03_05 = {0x01234567, 0x89ABCDEF};
    static const int32_t valueEC03_06 = 0xBEADCAFE;
    static const e::SignedRational valueEC03_07[] = {{static_cast<int32_t>(0xF0E1D2C3), 0x4B5A6978}, {static_cast<int32_t>(0x8796A5B4), 0x3C2D1E0F}};
    static const e::ValuedIfdTag inputEC03[] = {
        {{0x0000, e::IfdTagType_Undefined, 1, 0x0000},     {&valueEC03_00}},
        {{0x0001, e::IfdTagType_Byte, 1, 0x0000},          {&valueEC03_01}},
        {{0x0002, e::IfdTagType_Ascii, 2, 0x0000},         {(uint8_t*)"X"}},
        {{0x0003, e::IfdTagType_Short, 1, 0x0000},         {(uint8_t*)&valueEC03_03}},
        {{0x0004, e::IfdTagType_Long, 1, 0x0000},          {(uint8_t*)&valueEC03_04}},
        {{0x0005, e::IfdTagType_Rational, 1, 0x0000},      {(uint8_t*)&valueEC03_05}},
        {{0x0006, e::IfdTagType_SignedLong, 1, 0x0000},    {(uint8_t*)&valueEC03_06}},
        {{0x0007, e::IfdTagType_SignedRational, 2, 0x0000}, {(uint8_t*)&valueEC03_07}},
    };
    static const nn::Bit8 expectedEC03Be[] = {
        0x00, 0x08,
        // {0x0000, Undef, 1, {0xFF}}
        0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x01, 0xFF, 0x00, 0x00, 0x00,
        // {0x0001, Byte, 1, {0xEE}}
        0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x01, 0xEE, 0x00, 0x00, 0x00,
        // {0x0002, Ascii, 2, "X"}
        0x00, 0x02, 0x00, 0x02, 0x00, 0x00, 0x00, 0x02, 0x58, 0x00, 0x00, 0x00,
        // {0x0003, Short, 1, 0xBEEF}
        0x00, 0x03, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0xBE, 0xEF, 0x00, 0x00,
        // {0x0004, Long, 1, 0xDEADBEEF}
        0x00, 0x04, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0xDE, 0xAD, 0xBE, 0xEF,
        // {0x0005, Rational, 1, @106 = {0x01234567, 0x89ABCDEF}}
        0x00, 0x05, 0x00, 0x05, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x6E,
        // {0x0006, SignedLong, 1, 0xBEADCAFE}
        0x00, 0x06, 0x00, 0x09, 0x00, 0x00, 0x00, 0x01, 0xBE, 0xAD, 0xCA, 0xFE,
        // {0x0007, SignedRational, 2, @114 = {{0xF0E1D2C3, 0x4B5A6978}, {8796A5B4, 0x3C2D1E0F}}}
        0x00, 0x07, 0x00, 0x0A, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x76,
        0x00, 0x00, 0x00, 0x86,
        // {0x01234567, 0x89ABCDEF}
        0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
        // {{0xF0E1D2C3, 0x4B5A6978}, {8796A5B4, 0x3C2D1E0F}}
        0xF0, 0xE1, 0xD2, 0xC3, 0x4B, 0x5A, 0x69, 0x78,
        0x87, 0x96, 0xA5, 0xB4, 0x3C, 0x2D, 0x1E, 0x0F};
    NN_STATIC_ASSERT(sizeof(expectedEC03Be) == 126);

    uint8_t *valueEC04_01_dummy = (uint8_t*)malloc(0xFFDD);
    std::memset(valueEC04_01_dummy, 0xFA, 0xFFDD);
    static const e::ValuedIfdTag inputEC04[] = {
        {{0x0000, e::IfdTagType_Undefined, 0xFFDD, 0x0000}, {valueEC04_01_dummy}},
    };
    static const nn::Bit8 expectedEC04_Base[] = {
        0x00, 0x01,
        0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0xFF, 0xDD, 0x00, 0x00, 0x00, 0x1A,
        0x00, 0x00, 0xFF, 0xF7};
    NN_STATIC_ASSERT(sizeof(expectedEC04_Base) + 0xFFDD == 0xFFEF);
    uint8_t *expectedEC04 = (uint8_t*)malloc(0xFFEF);
    std::memcpy(expectedEC04, expectedEC04_Base, sizeof(expectedEC04_Base));
    std::memcpy(expectedEC04 + sizeof(expectedEC04_Base), valueEC04_01_dummy, 0xFFDD);

    e::ValuedIfdTag *inputEC05 = new e::ValuedIfdTag[0x1553];
    static const nn::Bit8 expectedEC05_Head[] = {0x15, 0x53};
    static const nn::Bit8 expectedEC05_Foot[] = {0x00, 0x00, 0xFF, 0xF7};
    uint8_t *expectedEC05 = (uint8_t*)malloc(0xFFEA);
    for (int i = 0; i < 0x1553; i++)
    {
        e::ValuedIfdTag &input = inputEC05[i];
        input.tag.id = static_cast<uint16_t>(i);
        input.tag.type = e::IfdTagType_Undefined;
        input.tag.valueCount = 0;
        input.tag.valueOffset = 0x0000;
        input.value.vUndefined = nullptr;

        nn::Bit8 tag[] = {static_cast<nn::Bit8>((i & 0xFF00) >> 8), static_cast<nn::Bit8>(i & 0x00FF), 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
        NN_STATIC_ASSERT(sizeof(tag) == 12);
        std::memcpy(expectedEC05 + 2 + 12 * i, tag, sizeof(tag));
    }
    std::memcpy(expectedEC05, expectedEC05_Head, sizeof(expectedEC05_Head));
    std::memcpy(expectedEC05 + 0xFFEA - sizeof(expectedEC05_Foot), expectedEC05_Foot, sizeof(expectedEC05_Foot));

    IfdBuildTestCase normalCases[] = {
        {"E-C-01", 8, inputEC01, 0, expectedEC01, sizeof(expectedEC01)},
        {"E-C-02", 8, inputEC02, sizeof(inputEC02) / sizeof(e::ValuedIfdTag), expectedEC02, sizeof(expectedEC02)},
        {"E-C-03", 8, inputEC03, sizeof(inputEC03) / sizeof(e::ValuedIfdTag), expectedEC03Be, sizeof(expectedEC03Be)},
        {"E-C-04", 8, inputEC04, sizeof(inputEC04) / sizeof(e::ValuedIfdTag), expectedEC04, 0xFFEF},
        {"E-C-05", 13, inputEC05, 0x1553, expectedEC05, 0xFFEA},
    };
    // ここまでテストケース

    nn::Bit8 *exifData = (nn::Bit8*)malloc(0xFFFF);
    for (int i = 0; i < static_cast<int>(sizeof(normalCases) / sizeof(IfdBuildTestCase)); i++)
    {
        IfdBuildTestCase &test = normalCases[i];

        NN_LOG("  - Test ID: %s\n", test.label);

        e::IfdHeader ifd = {};
        ASSERT_TRUE(e::InitializeIfdHeader(&ifd, test.tags, test.tagNum, test.offset));

        // IFD ヘッダの初期化 (自身のオフセット, タグ数, 次の IFD へのオフセット を確認)
        EXPECT_EQ(test.offset, ifd.offset);
        EXPECT_EQ(test.tagNum, ifd.tagCount);
        EXPECT_EQ(test.offset + test.expectedSize, ifd.nextOffset);

        // IFD を生成し、期待値 (バイナリ列) と一致するかを確認
        e::WriteIfd(exifData, ifd.nextOffset, test.tags, ifd, tiff.io);
        EXPECT_EQ(0, std::memcmp(test.expected, exifData + ifd.offset, test.expectedSize));
    }
    free(exifData);

    free(valueEC04_01_dummy);
    free(expectedEC04);
    delete[] inputEC05;
    free(expectedEC05);


    NN_LOG("--------------------------\n");
    NN_LOG(" - Abnormal input (Big endian)\n");
    // ここからテストケース
    e::ValuedIfdTag *inputEC89 = new e::ValuedIfdTag[0x1554];
    for (int i = 0; i < 0x1554; i++)
    {
        e::ValuedIfdTag &input = inputEC89[i];
        input.tag.id = static_cast<uint16_t>(i);
        input.tag.type = e::IfdTagType_Undefined;
        input.tag.valueCount = 0;
        input.tag.valueOffset = 0x0000;
        input.value.vUndefined = nullptr;
    }

    static const e::ValuedIfdTag inputEC88[] = {
        {{0x0000, e::IfdTagType_Undefined, 0xFFDE, 0x0000}, {nullptr}},
    };

    IfdBuildTestCase abnormalCases[] = {
        {"E-C-89", 8, inputEC89, 0x1554, nullptr, 0},
        {"E-C-88", 8, inputEC88, sizeof(inputEC88) / sizeof(e::ValuedIfdTag), nullptr, 0},
    };
    // ここまでテストケース

    exifData = (nn::Bit8*)malloc(0xFFFF);
    for (int i = 0; i < static_cast<int>(sizeof(abnormalCases) / sizeof(IfdBuildTestCase)); i++)
    {
        IfdBuildTestCase &test = abnormalCases[i];

        NN_LOG("  - Test ID: %s\n", test.label);

        e::IfdHeader ifd = {};
        EXPECT_FALSE(e::InitializeIfdHeader(&ifd, test.tags, test.tagNum, test.offset));
    }
    free(exifData);

    delete[] inputEC89;


    NN_LOG("--------------------------\n");
    NN_LOG(" - Normal input (Little endian)\n");
    e::InitializeTiffHeader(&tiff, e::Endian_Little);
    // ここからテストケース
    static const nn::Bit8 expectedEC03Le[] = {
        0x08, 0x00,
        // {0x0000, Undef, 1, {0xFF}}
        0x00, 0x00, 0x07, 0x00, 0x01, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x00, 0x00,
        // {0x0001, Byte, 1, {0xEE}}
        0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0xEE, 0x00, 0x00, 0x00,
        // {0x0002, Ascii, 2, "X"}
        0x02, 0x00, 0x02, 0x00, 0x02, 0x00, 0x00, 0x00, 0x58, 0x00, 0x00, 0x00,
        // {0x0003, Short, 1, 0xBEEF}
        0x03, 0x00, 0x03, 0x00, 0x01, 0x00, 0x00, 0x00, 0xEF, 0xBE, 0x00, 0x00,
        // {0x0004, Long, 1, 0xDEADBEEF}
        0x04, 0x00, 0x04, 0x00, 0x01, 0x00, 0x00, 0x00, 0xEF, 0xBE, 0xAD, 0xDE,
        // {0x0005, Rational, 1, @106 = {0x01234567, 0x89ABCDEF}}
        0x05, 0x00, 0x05, 0x00, 0x01, 0x00, 0x00, 0x00, 0x6E, 0x00, 0x00, 0x00,
        // {0x0006, SignedLong, 1, 0xBEADCAFE}
        0x06, 0x00, 0x09, 0x00, 0x01, 0x00, 0x00, 0x00, 0xFE, 0xCA, 0xAD, 0xBE,
        // {0x0007, SignedRational, 2, @114 = {{0xF0E1D2C3, 0x4B5A6978}, {8796A5B4, 0x3C2D1E0F}}}
        0x07, 0x00, 0x0A, 0x00, 0x02, 0x00, 0x00, 0x00, 0x76, 0x00, 0x00, 0x00,
        0x86, 0x00, 0x00, 0x00,
        // {0x01234567, 0x89ABCDEF}
        0x67, 0x45, 0x23, 0x01, 0xEF, 0xCD, 0xAB, 0x89,
        // {{0xF0E1D2C3, 0x4B5A6978}, {8796A5B4, 0x3C2D1E0F}}
        0xC3, 0xD2, 0xE1, 0xF0, 0x78, 0x69, 0x5A, 0x4B,
        0xB4, 0xA5, 0x96, 0x87, 0x0F, 0x1E, 0x2D, 0x3C};
    NN_STATIC_ASSERT(sizeof(expectedEC03Be) == 126);

    IfdBuildTestCase normalCasesLe[] = {
        {"E-C-03 (LE)", 8, inputEC03, sizeof(inputEC03) / sizeof(e::ValuedIfdTag), expectedEC03Le, sizeof(expectedEC03Le)},
    };
    // ここまでテストケース

    exifData = (nn::Bit8*)malloc(0xFFFF);
    for (int i = 0; i < static_cast<int>(sizeof(normalCasesLe) / sizeof(IfdBuildTestCase)); i++)
    {
        IfdBuildTestCase &test = normalCasesLe[i];

        NN_LOG("  - Test ID: %s\n", test.label);

        e::IfdHeader ifd = {};
        ASSERT_TRUE(e::InitializeIfdHeader(&ifd, test.tags, test.tagNum, test.offset));

        // IFD ヘッダの初期化 (自身のオフセット, タグ数, 次の IFD へのオフセット を確認)
        EXPECT_EQ(test.offset, ifd.offset);
        EXPECT_EQ(test.tagNum, ifd.tagCount);
        EXPECT_EQ(test.offset + test.expectedSize, ifd.nextOffset);

        // IFD を生成し、期待値 (バイナリ列) と一致するかを確認
        e::WriteIfd(exifData, ifd.nextOffset, test.tags, ifd, tiff.io);
        EXPECT_EQ(0, std::memcmp(test.expected, exifData + ifd.offset, test.expectedSize));
    }
    free(exifData);
} // NOLINT(readability/fn_size)

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    // TeamCity の表示を適切にするため、イベントリスナの登録を一旦すべて解除し、
    // ServiceMessageLogger -> デフォルトのイベントリスナ の順で登録し直す。
    ::testing::TestEventListeners& listeners = ::testing::UnitTest::GetInstance()->listeners();
    ::testing::TestEventListener* defaultResultPrinter = listeners.Release(listeners.default_result_printer());
    listeners.Append(new nnt::teamcity::ServiceMessageLogger());
    listeners.Append(defaultResultPrinter);

    int result = RUN_ALL_TESTS();

    nnt::Exit(result);
}
