﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/nn_Assert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Log.h>
#include <nn/nn_BitTypes.h>
#include <nnt/nntest.h>

#include <nn/image/image_ExifExtractor.h>
#include <image_ExifParser.h>

/**
    @file nn::image::ExifExtractor のテスト
 */

namespace exif = nn::image::detail;

namespace
{
/**
    @brief テスト用の Exif データを生成する関数型
    @details
    生成に使う JPEG ライブラリの Exif 解析内部実装は単独でテストされているため、
    正常に動作すると仮定してテストデータを動的に生成している。
 */
typedef void (*BuilderFunction)(
    size_t *pActualOut,
    void *pExifBuf,
    const size_t exifBufSize,
    const exif::ValuedIfdTag *tags0thTiff,
    const uint16_t num0thTiff,
    uint32_t * const pExifOffset,
    const exif::ValuedIfdTag *tags0thExif,
    const uint16_t num0thExif,
    const exif::ValuedIfdTag *tags1stTiff,
    const uint16_t num1stTiff,
    uint32_t * const pThumbOffset,
    const size_t thumbSize);

/**
    @brief バイナリの先頭から 0th TIFF, 0th Exif, 1st TIFF を詰めて Exif データを生成する
 */
void BuildExifData(
    size_t *pActualOut,
    void *pExifBuf,
    const size_t exifBufSize,
    const exif::ValuedIfdTag *tags0thTiff,
    const uint16_t num0thTiff,
    uint32_t * const pExifOffset,
    const exif::ValuedIfdTag *tags0thExif,
    const uint16_t num0thExif,
    const exif::ValuedIfdTag *tags1stTiff,
    const uint16_t num1stTiff,
    uint32_t * const pThumbOffset,
    const size_t thumbSize)
{
    exif::TiffHeader tiff = {};
    exif::InitializeTiffHeader(&tiff, exif::Endian_Big);

    exif::IfdHeader ifd0thTiff = {};
    ASSERT_TRUE(
        exif::InitializeIfdHeader(&ifd0thTiff, tags0thTiff, num0thTiff, tiff.offset0thIfd));

    exif::IfdHeader ifd0thExif = {};
    if (tags0thExif != nullptr)
    {
        // 0th Exif IFD は、タグが指定されていれば生成する。
        *pExifOffset = ifd0thTiff.nextOffset;
        ASSERT_TRUE(
            exif::InitializeIfdHeader(&ifd0thExif, tags0thExif, num0thExif, static_cast<uint16_t>(*pExifOffset)));
        ifd0thTiff.nextOffset = ifd0thExif.nextOffset;
        ifd0thExif.nextOffset = 0;
    }

    exif::IfdHeader ifd1stTiff = {};
    if (tags1stTiff != nullptr)
    {
        // 1st TIFF IFD は、タグが指定されていれば生成する。
        ASSERT_TRUE(
            exif::InitializeIfdHeader(&ifd1stTiff, tags1stTiff, num1stTiff, ifd0thTiff.nextOffset));
        *pThumbOffset = ifd1stTiff.nextOffset;

        NN_ASSERT(ifd1stTiff.nextOffset + thumbSize <= exifBufSize);
        *pActualOut = ifd1stTiff.nextOffset + thumbSize;
        ifd1stTiff.nextOffset = 0;
    }
    else
    {
        *pActualOut = ifd0thTiff.nextOffset;
        ifd0thTiff.nextOffset = 0;
    }

    // バイナリの書き出し
    exif::WriteTiffHeader((uint8_t*)pExifBuf, (uint16_t)exifBufSize, tiff);
    exif::WriteIfd((uint8_t*)pExifBuf, (uint16_t)exifBufSize, tags0thTiff, ifd0thTiff, tiff.io);
    if (tags0thExif != nullptr)
    {
        exif::WriteIfd((uint8_t*)pExifBuf, (uint16_t)exifBufSize, tags0thExif, ifd0thExif, tiff.io);
    }
    if (tags1stTiff != nullptr)
    {
        exif::WriteIfd((uint8_t*)pExifBuf, (uint16_t)exifBufSize, tags1stTiff, ifd1stTiff, tiff.io);
    }
}

/**
    @brief バイナリの末尾から 0th TIFF, 0th Exif, 1st TIFF を詰めて Exif データを生成する
    @detail
    与えた領域サイズをすべて使って作成するため、TIFF ヘッダの直後に不定領域が作成される。
 */
void BuildExifDataReverse(
    size_t *pActualOut,
    void *pExifBuf,
    const size_t exifBufSize,
    const exif::ValuedIfdTag *tags0thTiff,
    const uint16_t num0thTiff,
    uint32_t * const pExifOffset,
    const exif::ValuedIfdTag *tags0thExif,
    const uint16_t num0thExif,
    const exif::ValuedIfdTag *tags1stTiff,
    const uint16_t num1stTiff,
    uint32_t * const pThumbOffset,
    const size_t thumbSize)
{
    exif::TiffHeader tiff = {};
    exif::InitializeTiffHeader(&tiff, exif::Endian_Big);

    uint16_t head = tiff.offset0thIfd;
    uint16_t tail = (uint16_t)exifBufSize;

    exif::IfdHeader ifd0thTiff = {};
    ASSERT_TRUE(
        exif::InitializeIfdHeader(&ifd0thTiff, tags0thTiff, num0thTiff, tiff.offset0thIfd));
    tail = tiff.offset0thIfd = (uint16_t)exifBufSize - (ifd0thTiff.nextOffset - ifd0thTiff.offset);
    ifd0thTiff.offset = tiff.offset0thIfd;

    exif::IfdHeader ifd0thExif = {};
    if (tags0thExif != nullptr)
    {
        ASSERT_TRUE(
            exif::InitializeIfdHeader(&ifd0thExif, tags0thExif, num0thExif, head));

        tail
            = ifd0thExif.offset
            = tail - (ifd0thExif.nextOffset - ifd0thExif.offset);
        *pExifOffset = tail;
        ifd0thExif.nextOffset = 0;
    }

    exif::IfdHeader ifd1stTiff = {};
    if (tags1stTiff != nullptr)
    {
        ASSERT_TRUE(
            exif::InitializeIfdHeader(&ifd1stTiff, tags1stTiff, num1stTiff, head));

        tail
            = ifd0thTiff.nextOffset
            = ifd1stTiff.offset
            = tail - (ifd1stTiff.nextOffset - ifd1stTiff.offset);
        ifd1stTiff.nextOffset = 0;

        NN_ASSERT(head + thumbSize <= tail);
        *pThumbOffset = static_cast<uint32_t>(tail - thumbSize);
    }
    else
    {
        ifd0thTiff.nextOffset = 0;
    }
    *pActualOut = exifBufSize;

    // バイナリの書き出し
    exif::WriteTiffHeader((uint8_t*)pExifBuf, (uint16_t)exifBufSize, tiff);
    exif::WriteIfd((uint8_t*)pExifBuf, (uint16_t)exifBufSize, tags0thTiff, ifd0thTiff, tiff.io);
    if (tags0thExif != nullptr)
    {
        exif::WriteIfd((uint8_t*)pExifBuf, (uint16_t)exifBufSize, tags0thExif, ifd0thExif, tiff.io);
    }
    if (tags1stTiff != nullptr)
    {
        exif::WriteIfd((uint8_t*)pExifBuf, (uint16_t)exifBufSize, tags1stTiff, ifd1stTiff, tiff.io);
    }
}
}

TEST(ImageExifExtractor, AssertionTest)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing nn::image::ExifExtractor with assertion error\n");

    static const nn::Bit8 dummyExif[] = {
        0x4D, 0x4D, 0x00, 0x2A, 0x00, 0x00, 0x00, 0x08, // TIFF header (Big endian)
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00              // 0th TIFF IFD (no contents)
    };

    size_t workBufSize = nn::image::ExifExtractor::GetWorkBufferSize();
    void *workBuf = malloc(workBufSize);

    NN_LOG(" - Test ID: %s\n", "Constructor");
    EXPECT_DEATH_IF_SUPPORTED({nn::image::ExifExtractor extractor(nullptr, 0);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({nn::image::ExifExtractor extractor(workBuf, workBufSize - 1);}, ".*");

    NN_LOG(" - Test ID: %s\n", "SetExifData");
    nn::image::ExifExtractor extractor(workBuf, workBufSize);
    EXPECT_DEATH_IF_SUPPORTED({extractor.SetExifData(nullptr, 0);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({extractor.SetExifData(dummyExif, 0);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({extractor.SetExifData(dummyExif, 0xFFF8);}, ".*");

    // Un-registered
    NN_LOG(" - Test ID: %s\n", "Analyze");
    EXPECT_DEATH_IF_SUPPORTED({extractor.Analyze();}, ".*");


    size_t size;
    nn::image::ExifOrientation orientation;
    nn::image::Dimension dim;

    NN_LOG(" - Test ID: %s\n", "Extract****");
    extractor.SetExifData(dummyExif, sizeof(dummyExif));
    EXPECT_DEATH_IF_SUPPORTED({extractor.ExtractMaker(&size);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({extractor.ExtractModel(&size);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({extractor.ExtractOrientation(&orientation);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({extractor.ExtractSoftware(&size);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({extractor.ExtractDateTime(&size);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({extractor.ExtractMakerNote(&size);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({extractor.ExtractEffectiveDimension(&dim);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({extractor.ExtractUniqueId(&size);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({extractor.ExtractThumbnail(&size);}, ".*");

    NN_LOG(" - Test ID: %s\n", "Extract**** (again with state rollback)");
    ASSERT_EQ(nn::image::JpegStatus_Ok, extractor.Analyze());
    extractor.SetExifData(dummyExif, sizeof(dummyExif));
    EXPECT_DEATH_IF_SUPPORTED({extractor.ExtractMaker(&size);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({extractor.ExtractModel(&size);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({extractor.ExtractOrientation(&orientation);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({extractor.ExtractSoftware(&size);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({extractor.ExtractDateTime(&size);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({extractor.ExtractMakerNote(&size);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({extractor.ExtractEffectiveDimension(&dim);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({extractor.ExtractUniqueId(&size);}, ".*");
    EXPECT_DEATH_IF_SUPPORTED({extractor.ExtractThumbnail(&size);}, ".*");

    free(workBuf);
}

TEST(ImageExifExtractor, ExtractTest)
{
    NN_LOG("----------------------------------------------------\n");
    NN_LOG("Testing nn::image::ExifExtractor\n");

    size_t workBufSize = nn::image::ExifExtractor::GetWorkBufferSize();
    void *workBuf = malloc(workBufSize);
    nn::image::ExifExtractor extractor(workBuf, workBufSize);

    // U-X-01
    {
        NN_LOG(" - Test ID: %s\n", "U-X-01");
        static const nn::Bit8 exifUX01Be[] = {
            0x4D, 0x4D, 0x00, 0x2A, 0x00, 0x00, 0x00, 0x08, // TIFF header (Big endian)
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00              // 0th TIFF IFD (no contents)
        };
        NN_STATIC_ASSERT(sizeof(exifUX01Be) == 14);

        extractor.SetExifData(exifUX01Be, sizeof(exifUX01Be));
        ASSERT_EQ(nn::image::JpegStatus_Ok, extractor.Analyze());

        size_t length;
        EXPECT_EQ(nullptr, extractor.ExtractMaker(&length));
        EXPECT_EQ(nullptr, extractor.ExtractModel(&length));
        nn::image::ExifOrientation orientation;
        EXPECT_FALSE(extractor.ExtractOrientation(&orientation));
        EXPECT_EQ(nullptr, extractor.ExtractSoftware(&length));
        EXPECT_EQ(nullptr, extractor.ExtractDateTime(&length));

        EXPECT_EQ(nullptr, extractor.ExtractMakerNote(&length));
        nn::image::Dimension dimension;
        EXPECT_FALSE(extractor.ExtractEffectiveDimension(&dimension));
        EXPECT_EQ(nullptr, extractor.ExtractUniqueId(&length));

        EXPECT_EQ(nullptr, extractor.ExtractThumbnail(&length));
    }

    // U-X-02
    {
        NN_LOG(" - Test ID: %s\n", "U-X-02");
        static const char maker[] = "A";
        static const char model[] = "A";
        const nn::image::ExifOrientation kOrientation = nn::image::ExifOrientation_Normal;
        static const char software[] = "A";
        static const char dateTime[] = "    :  :     :  :  ";
        NN_STATIC_ASSERT(sizeof(dateTime) == 20);
        uint32_t exifOffset = 0x0000;
        static const exif::ValuedIfdTag inputUX02_0thTiff[] = {
            {{0x010F, exif::IfdTagType_Ascii, sizeof(maker),       0x0000}, {(const uint8_t*)maker}},
            {{0x0110, exif::IfdTagType_Ascii, sizeof(model),       0x0000}, {(const uint8_t*)model}},
            {{0x0112, exif::IfdTagType_Short, 1,                   0x0000}, {(const uint8_t*)&kOrientation}},
            {{0x0131, exif::IfdTagType_Ascii, sizeof(software),    0x0000}, {(const uint8_t*)software}},
            {{0x0132, exif::IfdTagType_Ascii, sizeof(dateTime),    0x0000}, {(const uint8_t*)dateTime}},
            {{0x8769, exif::IfdTagType_Long, 1,                    0x0000}, {(const uint8_t*)&exifOffset}},
        };
        static const uint8_t note[] = {0xFA};
        static const uint16_t dimWidth = 0x1FA;
        static const uint16_t dimHeight = 0x2EA;
        static const exif::ValuedIfdTag inputUX02_0thExif[] = {
            {{0x927C, exif::IfdTagType_Undefined, sizeof(note), 0x0000}, {note}},
            {{0xA002, exif::IfdTagType_Short, 1, 0x0000}, {(const uint8_t*)&dimWidth}},
            {{0xA003, exif::IfdTagType_Short, 1, 0x0000}, {(const uint8_t*)&dimHeight}},
        };
        static const uint16_t compr = 0x0006;
        static const uint8_t thumbnail[] = {0xFF, 0xD8, 0xFF, 0xD9}; // SOI + EOI
        uint32_t thumbOffset = 0x0000;
        static const uint32_t thumbSize = sizeof(thumbnail);
        static const exif::ValuedIfdTag inputUX02_1stTiff[] = {
            {{0x0103, exif::IfdTagType_Short, 1,   0x0000}, {(const uint8_t*)&compr}},
            {{0x0201, exif::IfdTagType_Long, 1,    0x0000}, {(const uint8_t*)&thumbOffset}},
            {{0x0202, exif::IfdTagType_Long, 1,    0x0000}, {(const uint8_t*)&thumbSize}},
        };

        // 2 種類の作成手法でテストデータを生成する
        BuilderFunction funcs[] = {
            BuildExifData,
            BuildExifDataReverse};
        for (int i = 0; i < static_cast<int>(sizeof(funcs) / sizeof(BuilderFunction)); i++)
        {
            NN_LOG("   - Builder#%d\n", i);

            // データの生成
            BuilderFunction &f = funcs[i];
            size_t actualSize = 0;
            void *exifData_tmp = malloc(0xFFF7);
            f(
                &actualSize,
                exifData_tmp, 0xFFF7,
                inputUX02_0thTiff, sizeof(inputUX02_0thTiff) / sizeof(exif::ValuedIfdTag), &exifOffset,
                inputUX02_0thExif, sizeof(inputUX02_0thExif) / sizeof(exif::ValuedIfdTag),
                inputUX02_1stTiff, sizeof(inputUX02_1stTiff) / sizeof(exif::ValuedIfdTag), &thumbOffset, thumbSize);
            ASSERT_FALSE(HasFatalFailure());

            std::memcpy((uint8_t*)exifData_tmp + thumbOffset, thumbnail, thumbSize);

            void *exifData = malloc(actualSize);
            std::memcpy(exifData, exifData_tmp, actualSize);
            free(exifData_tmp);

            // 解析の実施
            extractor.SetExifData(exifData, actualSize);
            ASSERT_EQ(nn::image::JpegStatus_Ok, extractor.Analyze());

            // 値の検査
            size_t length;
            const char *strPtr;
            EXPECT_NE(nullptr, strPtr = extractor.ExtractMaker(&length));
            EXPECT_EQ(sizeof(maker), length);
            EXPECT_EQ(0, std::memcmp(maker, strPtr, length));

            EXPECT_NE(nullptr, strPtr = extractor.ExtractModel(&length));
            EXPECT_EQ(sizeof(model), length);
            EXPECT_EQ(0, std::memcmp(model, strPtr, length));

            nn::image::ExifOrientation orientation;
            EXPECT_NE(false, extractor.ExtractOrientation(&orientation));
            EXPECT_EQ(kOrientation, orientation);

            EXPECT_NE(nullptr, strPtr = extractor.ExtractSoftware(&length));
            EXPECT_EQ(sizeof(software), length);
            EXPECT_EQ(0, std::memcmp(software, strPtr, length));

            EXPECT_NE(nullptr, strPtr = extractor.ExtractDateTime(&length));
            EXPECT_EQ(sizeof(dateTime), length);
            EXPECT_EQ(0, std::memcmp(dateTime, strPtr, length));


            const void *dataPtr;
            EXPECT_NE(nullptr, dataPtr = extractor.ExtractMakerNote(&length));
            EXPECT_EQ(sizeof(note), length);
            EXPECT_EQ(0, std::memcmp(note, dataPtr, length));

            nn::image::Dimension dimension;
            EXPECT_NE(false, extractor.ExtractEffectiveDimension(&dimension));
            EXPECT_EQ(dimWidth, dimension.width);
            EXPECT_EQ(dimHeight, dimension.height);

            EXPECT_EQ(nullptr, dataPtr = extractor.ExtractUniqueId(&length));


            EXPECT_NE(nullptr, dataPtr = extractor.ExtractThumbnail(&length));
            EXPECT_EQ(thumbSize, length);
            EXPECT_EQ(0, std::memcmp(thumbnail, dataPtr, length));

            free(exifData);
        }
    }

    // U-X-03
    {
        NN_LOG(" - Test ID: %s\n", "U-X-03");
        static const char maker[] = " !\"#$%&'()*+`-./0123456789";
        static const char model[] = ":;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ";
        const nn::image::ExifOrientation kOrientation = nn::image::ExifOrientation_Rotate90;
        static const char software[] = "[\\]^_`abcdefghijklmnopqrstuvwxyz{}~";
        static const char dateTime[] = "2015:01:01 00:00:59";
        NN_STATIC_ASSERT(sizeof(dateTime) == 20);
        uint32_t exifOffset = 0x0000;
        static const exif::ValuedIfdTag inputUX03_0thTiff[] = {
            {{0x010F, exif::IfdTagType_Ascii, sizeof(maker),       0x0000}, {(const uint8_t*)maker}},
            {{0x0110, exif::IfdTagType_Ascii, sizeof(model),       0x0000}, {(const uint8_t*)model}},
            {{0x0112, exif::IfdTagType_Short, 1,                   0x0000}, {(const uint8_t*)&kOrientation}},
            {{0x0131, exif::IfdTagType_Ascii, sizeof(software),    0x0000}, {(const uint8_t*)software}},
            {{0x0132, exif::IfdTagType_Ascii, sizeof(dateTime),    0x0000}, {(const uint8_t*)dateTime}},
            {{0x8769, exif::IfdTagType_Long, 1,                    0x0000}, {(const uint8_t*)&exifOffset}},
        };
        static const uint8_t note[] = {0xFA};
        static const uint32_t dimWidth = 0x1FA;
        static const uint32_t dimHeight = 0x2EA;
        static const char uniqueId[] = "000102030405060708090a0b0c0d0e0f";
        NN_STATIC_ASSERT(sizeof(uniqueId) == 33);
        static const exif::ValuedIfdTag inputUX03_0thExif[] = {
            {{0x927C, exif::IfdTagType_Undefined, sizeof(note), 0x0000}, {note}},
            {{0xA002, exif::IfdTagType_Long, 1, 0x0000}, {(const uint8_t*)&dimWidth}},
            {{0xA003, exif::IfdTagType_Long, 1, 0x0000}, {(const uint8_t*)&dimHeight}},
            {{0xA420, exif::IfdTagType_Ascii, sizeof(uniqueId), 0x0000}, {(const uint8_t*)uniqueId}},
        };
        static const uint16_t compr = 0x0006;
        static const uint8_t thumbnail[] = {0xFF, 0xD8, 0xFF, 0xD9}; // SOI + EOI
        uint32_t thumbOffset = 0x0000;
        static const uint32_t thumbSize = sizeof(thumbnail);
        static const exif::ValuedIfdTag inputUX03_1stTiff[] = {
            {{0x0103, exif::IfdTagType_Short, 1,   0x0000}, {(const uint8_t*)&compr}},
            {{0x0201, exif::IfdTagType_Long, 1,    0x0000}, {(const uint8_t*)&thumbOffset}},
            {{0x0202, exif::IfdTagType_Long, 1,    0x0000}, {(const uint8_t*)&thumbSize}},
        };

        // 2 種類の作成手法でテストデータを生成する
        BuilderFunction funcs[] = {
            BuildExifData,
            BuildExifDataReverse};
        for (int i = 0; i < static_cast<int>(sizeof(funcs) / sizeof(BuilderFunction)); i++)
        {
            NN_LOG("   - Builder#%d\n", i);

            // データの生成
            BuilderFunction &f = funcs[i];
            size_t actualSize = 0;
            void *exifData_tmp = malloc(0xFFF7);
            f(
                &actualSize,
                exifData_tmp, 0xFFF7,
                inputUX03_0thTiff, sizeof(inputUX03_0thTiff) / sizeof(exif::ValuedIfdTag), &exifOffset,
                inputUX03_0thExif, sizeof(inputUX03_0thExif) / sizeof(exif::ValuedIfdTag),
                inputUX03_1stTiff, sizeof(inputUX03_1stTiff) / sizeof(exif::ValuedIfdTag), &thumbOffset, thumbSize);
            ASSERT_FALSE(HasFatalFailure());

            std::memcpy((uint8_t*)exifData_tmp + thumbOffset, thumbnail, thumbSize);

            void *exifData = malloc(actualSize);
            std::memcpy(exifData, exifData_tmp, actualSize);
            free(exifData_tmp);

            // 解析の実施
            extractor.SetExifData(exifData, actualSize);
            ASSERT_EQ(nn::image::JpegStatus_Ok, extractor.Analyze());

            // 値の検査
            size_t length;
            const char *strPtr;
            EXPECT_NE(nullptr, strPtr = extractor.ExtractMaker(&length));
            EXPECT_EQ(sizeof(maker), length);
            EXPECT_EQ(0, std::memcmp(maker, strPtr, length));

            EXPECT_NE(nullptr, strPtr = extractor.ExtractModel(&length));
            EXPECT_EQ(sizeof(model), length);
            EXPECT_EQ(0, std::memcmp(model, strPtr, length));

            nn::image::ExifOrientation orientation;
            EXPECT_NE(false, extractor.ExtractOrientation(&orientation));
            EXPECT_EQ(kOrientation, orientation);

            EXPECT_NE(nullptr, strPtr = extractor.ExtractSoftware(&length));
            EXPECT_EQ(sizeof(software), length);
            EXPECT_EQ(0, std::memcmp(software, strPtr, length));

            EXPECT_NE(nullptr, strPtr = extractor.ExtractDateTime(&length));
            EXPECT_EQ(sizeof(dateTime), length);
            EXPECT_EQ(0, std::memcmp(dateTime, strPtr, length));


            const void *dataPtr;
            EXPECT_NE(nullptr, dataPtr = extractor.ExtractMakerNote(&length));
            EXPECT_EQ(sizeof(note), length);
            EXPECT_EQ(0, std::memcmp(note, dataPtr, length));

            nn::image::Dimension dimension;
            EXPECT_NE(false, extractor.ExtractEffectiveDimension(&dimension));
            EXPECT_EQ(static_cast<int32_t>(dimWidth), dimension.width);
            EXPECT_EQ(static_cast<int32_t>(dimHeight), dimension.height);

            EXPECT_NE(nullptr, strPtr = extractor.ExtractUniqueId(&length));
            EXPECT_EQ(sizeof(uniqueId), length);
            EXPECT_EQ(0, std::memcmp(uniqueId, strPtr, length));

            EXPECT_NE(nullptr, dataPtr = extractor.ExtractThumbnail(&length));
            EXPECT_EQ(thumbSize, length);
            EXPECT_EQ(0, std::memcmp(thumbnail, dataPtr, length));

            free(exifData);
        }
    }

    // U-X-99
    {
        // 破壊済みデータ
        static const uint8_t data[] = {
            0x4D, 0x4D, 0x00, 0x2A, 0x00, 0x00, 0x00, 0x08, 0x00, 0x05, 0x01, 0x0F, 0x00, 0x02, 0x00, 0x00,
            0x00, 0x10, 0x00, 0x00, 0x00, 0x4A, 0x01, 0x10, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00,
            0x00, 0x5A, 0x01, 0x31, 0x00, 0x02, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x6A, 0x01, 0x32,
            0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x7A, 0x87, 0x69, 0x00, 0x04, 0x00, 0x00,
            0x00, 0x01, 0x00, 0x00, 0x00, 0x8E, 0x00, 0x00, 0x00, 0xC1, 0xE3, 0x81, 0x82, 0xE3, 0x81, 0x84,
            0xE3, 0x81, 0x86, 0xE3, 0x81, 0x88, 0xE3, 0x81, 0x8A, 0x00, 0xE3, 0x81, 0x8B, 0xE3, 0x81, 0x8D,
            0xE3, 0x81, 0x8F, 0xE3, 0x81, 0x91, 0xE3, 0x81, 0x93, 0x00, 0xE3, 0x81, 0x95, 0xE3, 0x81, 0x97,
            0xE3, 0x81, 0x99, 0xE3, 0x81, 0x9B, 0xE3, 0x81, 0x9D, 0x00, 0x32, 0x30, 0x31, 0x35, 0x3A, 0x30,
            0x31, 0x3A, 0x30, 0x31, 0x20, 0x30, 0x30, 0x3A, 0x30, 0x30, 0x3A, 0x35, 0x00, 0x00, 0x00, 0x01,
            0xA4, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x20, 0x00, 0x00, 0x00, 0xA0, 0x00, 0x00, 0x00, 0x00,
            0x30, 0x30, 0x30, 0x31, 0x30, 0x32, 0x30, 0x33, 0x30, 0x34, 0x30, 0x35, 0x30, 0x36, 0x30, 0x37,
            0x30, 0x38, 0x30, 0x39, 0x30, 0x61, 0x30, 0x62, 0x30, 0x63, 0x30, 0x64, 0x30, 0x65, 0x30, 0x00,
            0x00, 0x00, 0x02, 0x01, 0x03, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x06, 0x00, 0x00, 0x02,
            0x01, 0x00, 0x04, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xEB, 0x00, 0x00, 0x00, 0x00, 0xCD,
            0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xCD, 0xFF, 0xD8, 0xFF, 0xD9
        };

        NN_LOG(" - Test ID: %s\n", "U-X-99");
        extractor.SetExifData(data, sizeof(data));
        ASSERT_EQ(nn::image::JpegStatus_Ok, extractor.Analyze());

        // 破壊されたフィールドがなかったことにされることを確認
        size_t length;
        EXPECT_EQ(nullptr, extractor.ExtractMaker(&length)); // 破壊
        EXPECT_EQ(nullptr, extractor.ExtractModel(&length)); // 破壊
        nn::image::ExifOrientation orientation;
        EXPECT_FALSE(extractor.ExtractOrientation(&orientation)); // 不在
        EXPECT_EQ(nullptr, extractor.ExtractSoftware(&length)); // 破壊
        EXPECT_EQ(nullptr, extractor.ExtractDateTime(&length)); // 破壊

        EXPECT_EQ(nullptr, extractor.ExtractMakerNote(&length)); // 不在
        nn::image::Dimension dimension;
        EXPECT_FALSE(extractor.ExtractEffectiveDimension(&dimension)); // 不在
        EXPECT_EQ(nullptr, extractor.ExtractUniqueId(&length)); // 破壊

        EXPECT_EQ(nullptr, extractor.ExtractThumbnail(&length)); // 破壊
    }

    // U-X-98
    {
        NN_LOG(" - Test ID: %s\n", "U-X-98");
        static const nn::Bit8 exifUX98Be[] = {
            0x4D, 0x4D, 0x00, 0x2A, 0x00, 0x00, 0x00, 0x08, // TIFF header (Big endian)
            0x00, 0x01,
            0x01, 0x12, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, // Orientation = 0
            0x00, 0x00, 0x00, 0x00              // 0th TIFF IFD (no contents)
        };

        // 解析の実施
        extractor.SetExifData(exifUX98Be, sizeof(exifUX98Be));
        ASSERT_EQ(nn::image::JpegStatus_Ok, extractor.Analyze());

        // 値の検査
        nn::image::ExifOrientation orientation;
        EXPECT_FALSE(extractor.ExtractOrientation(&orientation));
    }

    free(workBuf);
} // NOLINT(readability/fn_size)
