﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <cstdlib>

#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/util/util_FormatString.h>

#include "testImageJpeg_Path.h"

namespace nnt { namespace image { namespace jpeg { namespace io {

namespace
{
const char *kResourceFsName = "TestAssets";
#ifdef NN_TEST_JPEG_CONFIG_SAVE_ENABLED
const char *kOutputFsName = "TestOutput";
#endif

const unsigned int kMaxPathSize = 1024;

#ifdef NN_TEST_JPEG_CONFIG_SAVE_ENABLED
char *outputDirectoryPath = nullptr;
#endif

/**
    @brief ファイルサイズ取得のための基礎的な関数
 */
bool GetFileSizeCore(size_t *pSize, const char *storage, const char *path)
{
    nn::fs::FileHandle file;
    int64_t fullsize = 0;

    char *tmpPath = new char[kMaxPathSize];
    nn::util::SNPrintf(tmpPath, kMaxPathSize, "%s:/%s", storage, path);
    if (!(nn::fs::OpenFile(&file, tmpPath, nn::fs::OpenMode_Read).IsSuccess() &&
        nn::fs::GetFileSize(&fullsize, file).IsSuccess() &&
        fullsize <= 0xFFFFFFFF)) // ファイルサイズは 0xFFFFFFFF までしか対応しない。
    {
        NN_ABORT("Faield to stat file: %s\n", path);
    }
    nn::fs::CloseFile(file);
    delete[] tmpPath;

    *pSize = static_cast<size_t>(fullsize);
    return true;
}

/**
    @brief ファイル読み込みための基礎的な関数
 */
bool ReadFileCore(
    void *data,
    const size_t kDataSize,
    const char *storage,
    const char *path)
{
    nn::fs::FileHandle file;

    char *tmpPath = new char[kMaxPathSize];
    nn::util::SNPrintf(tmpPath, kMaxPathSize, "%s:/%s", storage, path);
    if (!(nn::fs::OpenFile(&file, tmpPath, nn::fs::OpenMode_Read).IsSuccess() &&
        nn::fs::ReadFile(file, 0, data, kDataSize).IsSuccess()))
    {
        NN_ABORT("Failed to read file: %s\n", path);
    }
    nn::fs::CloseFile(file);
    delete[] tmpPath;

    return true;
}

static bool gIsInitialized = false; ///< IO が初期化済みかどうかを示すフラグ。
}


/**
    @brief IOの初期化。多重呼び出しは無効(OK)
 */
bool Initialize()
{
    if (gIsInitialized)
    {
        // 多重呼び出し時は何もせず成功。
        return true;
    }

    char *path = new char[kMaxPathSize];
    if (path != nullptr)
    {

        if (!(GetAssetDirectoryPath(path, kMaxPathSize) &&
            nn::fs::MountHost(kResourceFsName, path).IsSuccess()))
        {
            NN_ABORT("Failed mounting resource directory\n");
        }
        NN_LOG("Mounted %s as %s\n", path, kResourceFsName);

#ifdef NN_TEST_JPEG_CONFIG_SAVE_ENABLED
        outputDirectoryPath = new char[kMaxPathSize];
        NN_ABORT_UNLESS(outputDirectoryPath != nullptr);

        if (!(GetOutputDirectoryPath(path, kMaxPathSize) &&
            nn::fs::CreateDirectory(path).IsSuccess() &&
            nn::fs::MountHost(kOutputFsName, path).IsSuccess()))
        {
            NN_ABORT("Failed mounting output directory\n");
        }
        NN_LOG("Mounting: %s as %s\n", path, kOutputFsName);

        std::strncpy(outputDirectoryPath, path, kMaxPathSize);
#endif
        // 初期化済みかどうかは Cleanup でしか見ない。他は FS のエラーに任せる。
        gIsInitialized = true;
    }
    delete[] path;

    return true;
}

/**
    @brief IOの解放。
 */
bool Cleanup()
{
    if (!gIsInitialized)
    {
        // 解放済みの場合には何もせず成功。
        return true;
    }

#ifdef NN_TEST_JPEG_CONFIG_SAVE_ENABLED
    nn::fs::Unmount(kOutputFsName);
    if (outputDirectoryPath != nullptr)
    {
        nn::fs::DeleteDirectoryRecursively(outputDirectoryPath);
        delete[] outputDirectoryPath;
        outputDirectoryPath = nullptr;
    }
#endif
    nn::fs::Unmount(kResourceFsName);

    gIsInitialized = false;
    return true;
}

bool GetAssetSize(size_t *pSize, const char *path)
{
    return GetFileSizeCore(pSize, kResourceFsName, path);
}
bool ReadAsset(
    void *data,
    const size_t kDataSize,
    const char *path)
{
    return ReadFileCore(data, kDataSize, kResourceFsName, path);
}

#ifdef NN_TEST_JPEG_CONFIG_SAVE_ENABLED
bool GetSavedataSize(unsigned int *pSize, const char *path)
{
    return GetFileSizeCore(pSize, kOutputFsName, path);
}
bool ReadSavedata(
    void *data,
    const unsigned int kDataSize,
    const char *path)
{
    return ReadFileCore(data, kDataSize, kOutputFsName, path);
}
bool WriteSavedata(
    const char *path,
    const void *data,
    const unsigned int kDataSize)
{
    nn::fs::FileHandle file;
    nn::fs::WriteOption op = {};

    // TODO FS の WriteOption が Flush に対応したら WriteFile で Flush する。
    char *tmpPath = new char[kMaxPathSize];
    nn::util::SNPrintf(tmpPath, kMaxPathSize, "%s:/%s", kOutputFsName, path);
    if (!(nn::fs::CreateFile(tmpPath, kDataSize).IsSuccess() &&
        nn::fs::OpenFile(&file, tmpPath, nn::fs::OpenMode_Write).IsSuccess() &&
        nn::fs::WriteFile(file, 0, data, kDataSize, op).IsSuccess() &&
        nn::fs::FlushFile(file).IsSuccess()))
    {
        NN_ABORT("Failed to write file: %s\n", path);
    }
    nn::fs::CloseFile(file);
    delete[] tmpPath;

    return true;
}
#endif

}}}}
