﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>

#include <nn/os.h>

#include <nn/result/result_HandlingUtility.h>

#include <nn/bpc/bpc_ResultPrivate.h>

#include <nn/i2c/i2c.h>

#include "tool_Max77620Rtc.h"

namespace nnt { namespace i2c { namespace max77620rtc {

namespace {

nn::Result ReadRtcDoubleBufferredRegister(uint8_t* pOutValue, nn::i2c::I2cSession& session, const uint8_t* pAddress) NN_NOEXCEPT
{
    const uint8_t RtcUpdate0Address = 0x04;
    const uint8_t RbudrMask = 0x10;
    const int MaxRetryCount = 4;

    NN_RESULT_DO(nn::i2c::WriteSingleRegister(session, &RtcUpdate0Address, &RbudrMask));

    int count = 0;
    uint8_t value = 0x00;
    do
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        NN_RESULT_DO(nn::i2c::ReadSingleRegister(&value, session, &RtcUpdate0Address));
        count++;
    }
    while ( ((value & RbudrMask) == RbudrMask) && count < MaxRetryCount );

    if ( (value & RbudrMask) == RbudrMask )
    {
        return nn::bpc::ResultRtcAccessFailed();
    }

    NN_RESULT_DO(nn::i2c::ReadSingleRegister(pOutValue, session, pAddress));

    NN_RESULT_SUCCESS;
}

} // namespace

nn::Result DumpRegisters()
{
    const uint8_t RegFieldLength = 0x1c;

    NN_LOG("[max77620rtc] Start to dump\n");

    nn::i2c::Initialize();

    nn::i2c::I2cSession i2c;
    uint8_t value = 0;

    nn::i2c::OpenSession(&i2c, nn::i2c::I2cDevice_Max77620Rtc);

    for ( uint8_t index = 0; index < RegFieldLength; index++ )
    {
        // 下記のレジスタ以外はダブルバッファドレジスタなので読み込みに手順が必要
        // 0: RTCINT
        // 4: RTCUPDATE0
        // 5: RTCUPDATE1
        // 6: RTCSMPL
        if ( index == 0 || index == 4 || index == 5 || index == 6 )
        {
            NN_RESULT_DO(nn::i2c::ReadSingleRegister(&value, i2c, &index));
            NN_LOG("[max77620rtc] %02x : %02x\n", index, value);
        }
        else
        {
            NN_RESULT_DO(ReadRtcDoubleBufferredRegister(&value, i2c, &index));
            NN_LOG("[max77620rtc] %02x : %02x\n", index, value);
        }
    }

    nn::i2c::CloseSession(i2c);

    nn::i2c::Finalize();

    NN_LOG("[max77620rtc] End\n");

    NN_RESULT_SUCCESS;
}

}}}  // namespace nnt::i2c::max77620rtc
