﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>
#include <nnt/nntest.h>
#include <nnt/htcsUtil/testHtcs_util.h>
#include <nnt/htcsUtil/testHtcs_utilMemoryLeakDetector.h>

/*
*   典型的な送受信を行うテスト。
*   事前に Target Manager を立ち上げておく必要があります。
*   テスト時に Tests/Htcs/Sources/Tools/StressTestTool を実行する必要があります。
*/

namespace
{
    const size_t MaxThreadCount = 8;
    const size_t ThreadStackSize = 16 * 1024;
    NN_ALIGNAS(4096) uint8_t g_ThreadStacks[MaxThreadCount][ThreadStackSize];

    static const size_t KeyStringSize = 11;
    static char g_KeyString[KeyStringSize];

    void* Allocate(size_t size)
    {
        return malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        free(p);
    }

    class StressTest : public ::testing::Test
    {
    protected:
        static void SetUpTestCase() {
            for (int i = 0; i < (KeyStringSize - 1); i++)
            {
                int seed = nn::os::GetSystemTick().GetInt64Value() % 26;
                g_KeyString[i] = static_cast<char>('a' + seed);
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            }
            g_KeyString[KeyStringSize - 1] = '\0';
        }

        virtual void SetUp()
        {
            nn::htcs::Initialize(Allocate, Deallocate);
        }
        virtual void TearDown()
        {
            nn::htcs::Finalize();
        }
    };
}

TEST_F(StressTest, Start)
{
    int socket = nn::htcs::Socket();
    ASSERT_LE(0, socket);

    nnt::htcs::util::ConnectToTestServer(socket);

    const char* testName = "TypicalCase";

    size_t bufferSize = std::strlen(g_KeyString) + 1 + std::strlen(testName) + 1;
    char* buffer = reinterpret_cast<char*>(std::malloc(bufferSize));
    ASSERT_NE(nullptr, buffer);
    buffer[0] = '\0';

    std::strcat(buffer, g_KeyString);
    std::strcat(buffer, ",");
    std::strcat(buffer, testName);

    ASSERT_EQ(nn::htcs::Send(socket, buffer, bufferSize, 0), bufferSize);

    std::free(buffer);
}

// 送受信を何度も繰り返すテストです。
TEST_F(StressTest, Repeat)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    ASSERT_GE(MaxThreadCount, (size_t)2);

    nn::os::Barrier barrier(2);
    nn::os::ThreadType thread0, thread1;
    nnt::htcs::util::ThreadArgs arg0, arg1;

    std::strcpy(arg0.address.portName.name, "HostToTarget_");
    std::strcat(arg0.address.portName.name, g_KeyString);
    arg0.mode = nnt::htcs::util::ThreadMode_Recv;
    arg0.pBarrier = &barrier;
    arg0.size = 1024;
    arg0.count = 1024;
    arg0.verbose = false;

    std::strcpy(arg1.address.portName.name, "TargetToHost_");
    std::strcat(arg1.address.portName.name, g_KeyString);
    arg1.mode = nnt::htcs::util::ThreadMode_Send;
    arg1.pBarrier = &barrier;
    arg1.size = 1024;
    arg1.count = 1024;
    arg1.verbose = false;

    nn::os::CreateThread(&thread0, nnt::htcs::util::ClientThread, &arg0, g_ThreadStacks[0], ThreadStackSize, nn::os::DefaultThreadPriority);
    nn::os::CreateThread(&thread1, nnt::htcs::util::ClientThread, &arg1, g_ThreadStacks[1], ThreadStackSize, nn::os::DefaultThreadPriority);

    nn::os::StartThread(&thread0);
    nn::os::StartThread(&thread1);

    nn::os::WaitThread(&thread0);
    nn::os::WaitThread(&thread1);

    nn::os::DestroyThread(&thread0);
    nn::os::DestroyThread(&thread1);
}

// 複数のポートで、適当なサイズの送受信を繰り返すテストです。
TEST_F(StressTest, MultiPort)
{
    ScopedMemoryLeakDetector memoryLeakDetector;

    const size_t ThreadCount = 8;
    ASSERT_GE(MaxThreadCount, ThreadCount);

    nn::os::Barrier barrier(ThreadCount);
    nn::os::ThreadType threads[ThreadCount];
    nnt::htcs::util::ThreadArgs args[ThreadCount];

    for (int i = 0; i < ThreadCount; i++)
    {
        nn::util::SNPrintf(args[i].address.portName.name, sizeof(args[i].address.portName), "Echo%d_%s", i, g_KeyString);
        args[i].mode = nnt::htcs::util::ThreadMode_Echo;
        args[i].pBarrier = &barrier;
        args[i].size = 1024 * 1024;
        args[i].count = 128;
        args[i].verbose = false;
    }

    for (int i = 0; i < ThreadCount; i++)
    {
        nn::os::CreateThread(&threads[i], nnt::htcs::util::ClientThread, &args[i], g_ThreadStacks[i], ThreadStackSize, nn::os::DefaultThreadPriority);
    }

    for (int i = 0; i < ThreadCount; i++)
    {
        nn::os::StartThread(&threads[i]);
    }

    for (int i = 0; i < ThreadCount; i++)
    {
        nn::os::WaitThread(&threads[i]);
    }

    for (int i = 0; i < ThreadCount; i++)
    {
        nn::os::DestroyThread(&threads[i]);
    }
}
