﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <thread>

#include <nn/nn_Common.h>

#include <nnt/nntest.h>

#include <nn/htclow/detail/htclow_InternalTypes.h>

#include "../testHtclow_Util.h"
#include "testHtclow_PacketUtil.h"
#include "testHtclow_HandshakeUtil.h"

namespace nnt { namespace htclow {

// 以下の手順を実行する。ホストが先に Connect を呼ぶパターン。
// 1. ターゲット側 Connect() 実行
// 2. ホスト側 Syn パケット送信
// 3. ホスト側 SynAck パケット受信
void DoHandshake(nn::htclow::Channel* pChannel) NN_NOEXCEPT
{
    const auto channelInternal = nn::htclow::detail::ConvertChannelType(*(pChannel->GetBase()));

    // ターゲット側 Connect() 実行
    std::thread connectThread([&]() NN_NOEXCEPT {
        NNT_HTCLOW_ASSERT_RESULT_SUCCESS(pChannel->Connect());
    });

    // ホスト側 Syn パケット受信
    const auto synPacket = ReceivePacketForTest();
    AssertSynPacket(*synPacket, channelInternal);

    // ホスト側 SynAck パケット送信
    const auto packet = MakeSynAckPacket(channelInternal);
    SendPacketForTest(*packet);

    connectThread.join();
}

// 以下のシャットダウンシーケンスを実行する。ターゲットが先にシーケンスを開始するパターン。
// 1. ターゲット側 Shutdown() 実行 (並列)
// 2. ホスト側 Fin パケット受信
// 3. ホスト側 FinAck パケット送信
// 4. ホスト側 Fin パケット送信
// 5. ホスト側 FinAck パケット受信
void DoShutdownByTarget(nn::htclow::Channel* pChannel, int64_t hostSeqeunceId, int64_t targetSequenceId) NN_NOEXCEPT
{
    const auto channelInternal = nn::htclow::detail::ConvertChannelType(*(pChannel->GetBase()));

    std::thread shutdownThread([&]() NN_NOEXCEPT {
        pChannel->Shutdown();
    });

    const auto packet1 = ReceivePacketForTest();
    AssertFinPacket(*packet1, channelInternal, targetSequenceId);

    const auto finAck = MakeFinAckPacket(channelInternal, targetSequenceId);
    SendPacketForTest(*finAck);

    const auto fin = MakeFinPacket(channelInternal, hostSeqeunceId);
    SendPacketForTest(*fin);

    const auto packet2 = ReceivePacketForTest();
    AssertFinAckPacket(*packet2, channelInternal, hostSeqeunceId);

    shutdownThread.join();
}

// 以下のシャットダウンシーケンスを実行する。ホストが先にシーケンスを開始するパターン。
// 1. ホスト側 Fin パケット送信
// 2. ホスト側 FinAck パケット受信
// 3. ターゲット側 Shutdown() 実行 (並列)
// 4. ホスト側 Fin パケット受信
// 5. ホスト側 FinAck パケット送信
void DoShutdownByHost(nn::htclow::Channel* pChannel, int64_t hostSeqeunceId, int64_t targetSequenceId) NN_NOEXCEPT
{
    const auto channelInternal = nn::htclow::detail::ConvertChannelType(*(pChannel->GetBase()));

    const auto fin = MakeFinPacket(channelInternal, hostSeqeunceId);
    SendPacketForTest(*fin);

    const auto packet1 = ReceivePacketForTest();
    AssertFinAckPacket(*packet1, channelInternal, hostSeqeunceId);

    std::thread shutdownThread([&]() NN_NOEXCEPT {
        pChannel->Shutdown();
    });

    const auto packet2 = ReceivePacketForTest();
    AssertFinPacket(*packet2, channelInternal, targetSequenceId);

    const auto packet = MakeFinAckPacket(channelInternal, targetSequenceId);
    SendPacketForTest(*packet);

    shutdownThread.join();
}

// 以下のシャットダウンシーケンスを実行する。ホストとターゲットが同時にシーケンスを開始するパターン。
// 1.  ターゲット側 Shutdown() 実行 (並列)
// 2.  ホスト側 Fin パケット送信
// 3a. ホスト側 Fin パケット受信
// 3b. ホスト側 FinAck パケット受信
// 4.  ホスト側 FinAck パケット送信
void DoShutdownSimultaneously(nn::htclow::Channel* pChannel, int64_t hostSeqeunceId, int64_t targetSequenceId) NN_NOEXCEPT
{
    const auto channelInternal = nn::htclow::detail::ConvertChannelType(*(pChannel->GetBase()));

    std::thread shutdownThread([&]() NN_NOEXCEPT {
        pChannel->Shutdown();
    });

    const auto fin = MakeFinPacket(channelInternal, hostSeqeunceId);
    SendPacketForTest(*fin);

    const auto packet1 = ReceivePacketForTest();
    if (packet1->GetHeader()->packetType == nn::htclow::server::PacketType::Fin)
    {
        AssertFinPacket(*packet1, channelInternal, targetSequenceId);

        const auto packet2 = ReceivePacketForTest();
        AssertFinAckPacket(*packet2, channelInternal, hostSeqeunceId);
    }
    else
    {
        AssertFinAckPacket(*packet1, channelInternal, hostSeqeunceId);

        const auto packet2 = ReceivePacketForTest();
        AssertFinPacket(*packet2, channelInternal, targetSequenceId);
    }

    const auto finAck = MakeFinAckPacket(channelInternal, targetSequenceId);
    SendPacketForTest(*finAck);

    shutdownThread.join();
}


}}
