﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       Ronde に関する API の宣言
 */

#pragma once

#include <nn/hidbus/hidbus.h>
#include <nns/hidbus/hidbus_Ronde.h>

namespace nnt { namespace hidbus {

/**
* @brief RondeのdeviceID定義
*/
const uint32_t RondeDeviceId = 0x20;

/**
* @brief データサイズ (Byte)定義
*/
struct RondeDataSize
{
    const size_t Header          = 4;
    const size_t UniqueId        = 12;
    const size_t PollingFormat    = 4;
    const size_t FwVer           = 2;
    const size_t Calibration     = 2;
};
const RondeDataSize SizeList;

/**
* @brief データコマンド定義
*/
struct RondeDataCommand
{
    const uint8_t UniqueId[4]            = { 0x00, 0x01, 0x02, 0x00 };
    const uint8_t PollingFormat[4]       = { 0x01, 0x01, 0x02, 0x00 };
    const uint8_t FwVer[4]               = { 0x00, 0x00, 0x02, 0x00 };
    const uint8_t SCalOsMax[4]           = { 0x04, 0x01, 0x02, 0x00 };
    const uint8_t SCalHkMax[4]           = { 0x04, 0x02, 0x02, 0x00 };
    const uint8_t SCalZeroMax[4]         = { 0x04, 0x03, 0x02, 0x00 };
    const uint8_t SCalZeroMin[4]         = { 0x04, 0x04, 0x02, 0x00 };
    const uint8_t UCalOsMax[4]           = { 0x04, 0x11, 0x02, 0x00 };
    const uint8_t UCalHkMax[4]           = { 0x04, 0x12, 0x02, 0x00 };
    const uint8_t UCalZeroMin[4]         = { 0x04, 0x13, 0x02, 0x00 };
    const uint8_t UCalZeroMax[4]         = { 0x04, 0x14, 0x02, 0x00 };
    const uint8_t WriteUCalOsMax[4]      = { 0x04, 0x11, 0x01, 0x04 };
    const uint8_t WriteUCalHkMax[4]      = { 0x04, 0x12, 0x01, 0x04 };
    const uint8_t WriteUCalZeroMin[4]    = { 0x04, 0x13, 0x01, 0x04 };
    const uint8_t WriteUCalZeroMax[4]    = { 0x04, 0x14, 0x01, 0x04 };
};
const RondeDataCommand CmdList;

/**
* @brief Ronde センサーの受信データです。
*/
struct RondeSensorData
{
    int16_t data;                    //!< 受信データです。
    int64_t samplingNumber;          //!< 受信データのサンプリングナンバーです。
};

/**
* @brief Ronde のユーザーキャル値を列挙した構造体です。
*/
struct RondeUserCalibrationData
{
    int16_t osMax;     //!< osMax
    int16_t hkMax;     //!< hkMax
    int16_t zeroMin;   //!< zeroMin
    int16_t zeroMax;   //!< zeroMax
};

/**
* @brief       Ronde センサーの Thermistor の値を過去に遡って取得します。
*
* @details     Ronde センサーの Thermistor の値を最新のものから過去に遡って利用可能な数だけ順に取得します。
*
* @param[in]   pOutData                最新の値
* @param[in]   count                   受信データを受け取る数
* @param[in]   handle                  受信データをコピーする拡張バスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                              処理に成功しました。}
*      @handleresult{nn::hidbus::ResultExternalDeviceNotEnabled,     拡張バスに接続されたデバイスが有効化されていません。}
*      @handleresult{nn::hidbus::ResultInvalidHandle,                無効な handle です。}
* @endretresult
*
* @pre
*  - nn::hidbus::Initialize() の呼び出しが完了している。
*  - EnableRonde() の呼び出しが完了している
*  - handle 取得時に busTypeId に BusType_LeftJoyRail または BusType_RightJoyRail を指定している。
*/
nn::Result SendAndReceiveRondeData(uint8_t * cmd, size_t datasize, uint8_t * pOutBuffer, const nn::hidbus::BusHandle & handle) NN_NOEXCEPT;

/**
* @brief       Ronde センサーのユーザーキャルの値を取得します。
*
* @details     Ronde センサーのユーザーキャルの値を取得します。
*              本関数はデバイスとの通信が発生するため、データの受信完了までブロッキングします。
*
* @param[in]   pOutData                ユーザーキャルの値をまとめた構造体
* @param[in]   handle                   拡張バスへのハンドル
*
* @retresult
*      @handleresult{nn::ResultSuccess,                              処理に成功しました。}
*      @handleresult{nn::hidbus::ResultExternalDeviceNotEnabled,     拡張バスに接続されたデバイスが有効化されていません。}
*      @handleresult{nn::hidbus::ResultExternalDeviceTimeout,        拡張バスに接続されたデバイスとの通信のタイムアウトです。}
*      @handleresult{nn::hidbus::ResultInvalidHandle,                無効な handle です。}
* @endretresult
*
* @pre
*  - nn::hidbus::Initialize() の呼び出しが完了している。
*  - EnableRonde() の呼び出しが完了している
*  - handle 取得時に busTypeId に BusType_LeftJoyRail または BusType_RightJoyRail を指定している。
*/
nn::Result GetPollingRondeData(nn::hidbus::JoyPollingReceivedData * pOutData, size_t datasize, int count, const nn::hidbus::BusHandle & handle)NN_NOEXCEPT;

}} // namespace nns::hidbus
