﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{HidbusRonde.cpp,PageSampleHidbusRonde}

    @brief
    RONDE を使用するためのサンプルプログラム
 */

#include "hidfw/gfx.h"
#include "hidfw/hid.h"
#include "hidfw/layout.h"

#include "Applet.h"
#include "ExclusiveFunc.h"
#include "CursolSelect.h"

#include <nn/oe.h>
#include <nn/os.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/hidbus/hidbus.h>
#include <nns/gfxLog.h>
#include <nns/hidbus/hidbus_Ronde.h>

#include "MenuFunc.h"
#include "RondeUtility.h"
#include "JoyAssignment.h"
#include "PrintNpadLog.h"

#include "HidbusRonde.h"

namespace {
    const int NpadIdNum = 9;

    DrawMode drawMode = DrawMode_1to2;

    HidbusState stateL[NpadIdNum];
    HidbusState stateR[NpadIdNum];

    RondeData dataL[NpadIdNum];
    RondeData dataR[NpadIdNum];

    nnt::Cursol cursol[NpadIdNum];

    nn::oe::FocusHandlingMode forcusmode = nn::oe::FocusHandlingMode_Notify;

    uint8_t PollingCount[NpadIdNum];

    bool IsAssignmentSelectMode = false;

    /////////////////////////////
    // ツール内状態の文字変換
    /////////////////////////////
    const char* ToString(HidbusState state)
    {
        if (state.mode == HidbusState_NotGotHandle)
        {
            return "NotGotHandle";
        }
        else if (state.mode == HidbusState_GotHandle)
        {
            return "NotGotHandle ⇒ GotHandle";
        }
        else if (state.mode == HidbusState_Enabled)
        {
            return "NotGotHandle ⇒ GotHandle ⇒ Enabled";
        }
        else if (state.mode == HidbusState_Polling)
        {
            return "NotGotHandle ⇒ GotHandle ⇒ Enabled ⇒ Polling";
        }
        else
        {
            return "Undefined";
        }
    }

    /////////////////////////////
    // ツール内状態の文字変換
    /////////////////////////////
    const char* ToString(BusStatus mode)
    {
        if (mode == HidbusState_NotGotHandle)
        {
            return "NotGotHandle";
        }
        else if (mode == HidbusState_GotHandle)
        {
            return "GotHandle";
        }
        else if (mode == HidbusState_Enabled)
        {
            return "Enabled";
        }
        else if (mode == HidbusState_Polling)
        {
            return "Polling";
        }
        else
        {
            return "Undefined";
        }
    }

    /////////////////////////////
    // NpadStyleの文字変換
    /////////////////////////////
    const char* ToString(nn::hid::NpadStyleSet style)
    {
        if (style == nn::hid::NpadStyleJoyLeft::Mask)
        {
            return "Style:JoyLeft";
        }
        else if (style == nn::hid::NpadStyleJoyRight::Mask)
        {
            return "Style:JoyRight";
        }
        else if (style == nn::hid::NpadStyleJoyDual::Mask)
        {
            return "Style:JoyDual";
        }
        else if (style == nn::hid::NpadStyleFullKey::Mask)
        {
            return "Style:FullKey";
        }
        else if (style == nn::hid::NpadStyleHandheld::Mask)
        {
            return "Style:Handheld";
        }
        else
        {
            return "Style:Undefined";
        }
    }

    /////////////////////////////////
    // ツール描画状態の判定
    /////////////////////////////////
    void JudgeDrawMode()
    {
        if (drawMode == DrawMode::DrawMode_1to2)
        {
            drawMode = DrawMode::DrawMode_3to4;
        }
        else if (drawMode == DrawMode::DrawMode_3to4)
        {
            drawMode = DrawMode::DrawMode_5to6;
        }
        else if (drawMode == DrawMode::DrawMode_5to6)
        {
            drawMode = DrawMode::DrawMode_7to8;
        }
        else if (drawMode == DrawMode::DrawMode_7to8)
        {
            drawMode = DrawMode::DrawMode_Hh;
        }
        else
        {
            drawMode = DrawMode::DrawMode_1to2;
        }
    }

    ///////////////////////////////
    // BG遷移時のFocusModeの設定
    ///////////////////////////////
    void SetAppHandlingMd() NN_NOEXCEPT
    {
        if (forcusmode == nn::oe::FocusHandlingMode_InFocusOnly)
        {
            forcusmode = nn::oe::FocusHandlingMode_Notify;
            NN_LOG("Focus Mode : Notify\n");
        }
        else
        {
            forcusmode = nn::oe::FocusHandlingMode_InFocusOnly;
            NN_LOG("Focus Mode : InFocusOnly\n");
        }
        nn::oe::SetFocusHandlingMode(forcusmode);
    }

    /////////////////////////////
    // ボタンの共通入力判定
    /////////////////////////////
    bool IsPushed()
    {
        // 接続中のコントローラのボタンの押下げ状況を確認します
        for (std::vector<nns::hidfw::hid::Controller*>::iterator it = gController.GetConnectedControllerList().begin();
             it != gController.GetConnectedControllerList().end();
             ++it)
        {
            if ((*it)->IsTrigger(nn::hid::NpadButton::Plus::Mask | nn::hid::NpadButton::Minus::Mask))
            {
                IsAssignmentSelectMode = IsAssignmentSelectMode == true ? false : true;
            }
            if ((*it)->IsTrigger(nn::hid::NpadButton::StickL::Mask | nn::hid::NpadButton::StickR::Mask))
            {
                JudgeDrawMode();
            }
            if ((*it)->IsTrigger(nn::hid::NpadButton::ZL::Mask | nn::hid::NpadButton::ZR::Mask))
            {
                SetAppHandlingMd();
            }
        }
        return false;
    }

    ////////////////////////////////////////////////////////////////////
    // データ格納
    // 取得したデータを、選択したデータ体系に応じた変数に格納します
    ////////////////////////////////////////////////////////////////////
    void UpdateData(HidbusState* state, RondeData* data, size_t index)
    {
        // FWVersion
        if (memcmp(&state->sendingCmd, &nnt::hidbus::CmdList.FwVer[0], 4) == 0)
        {
            data->fwVer[0] = state->packet[1] << 8 | state->packet[0];
        }
        // PollingFormat
        if (memcmp(&state->sendingCmd, &nnt::hidbus::CmdList.PollingFormat[0], 4) == 0)
        {
            data->strain[0] = state->packet[1] << 8 | state->packet[0];
            data->thermistor[0] = state->packet[3] << 8 | state->packet[2];
        }
        // UniqueID
        if (memcmp(&state->sendingCmd, &nnt::hidbus::CmdList.UniqueId[0], 4) == 0)
        {
            for (int i = 0; i < nnt::hidbus::SizeList.UniqueId; i++)
            {
                data->uniqueId[0][i] = state->packet[i];
            }
        }
        if (state->isPolling == IsPolling_Enable)
        {
            for (auto cnt = 0; cnt < PollingCountMax; cnt++)
            {
                // FWVersion
                if (memcmp(&state->PollingCmd, &nnt::hidbus::CmdList.FwVer[0], 4) == 0)
                {
                    data->fwVer[cnt] = state->PollingPacket[cnt][1] << 8 | state->PollingPacket[cnt][0];
                }
                // PollingFormat
                if (memcmp(&state->PollingCmd, &nnt::hidbus::CmdList.PollingFormat[0], 4) == 0)
                {
                    data->strain[cnt] = state->PollingPacket[cnt][1] << 8 | state->PollingPacket[cnt][0];
                    data->thermistor[cnt] = state->PollingPacket[cnt][3] << 8 | state->PollingPacket[cnt][2];
                }
                // UniqueID
                if (memcmp(&state->PollingCmd, &nnt::hidbus::CmdList.UniqueId[0], 4) == 0)
                {
                    for (int i = 0; i < nnt::hidbus::SizeList.UniqueId; i++)
                    {
                        data->uniqueId[cnt][i] = state->PollingPacket[cnt][i];
                    }
                }
            }
        }
    }

    ///////////////////////////////////
    // Ronde情報の取得、表示
    ///////////////////////////////////
    void DrawRondeInfo(float x, float y, size_t index)
    {
        int AxisXLeft = 170;
        int AxisXRight = 280;
        int AxisYCount = 0;

        gTextWriter.SetTextColor(nn::util::Color4u8::Black());
        gTextWriter.SetCursor(x - 5, y + (25 * AxisYCount));
        gTextWriter.Print("[Rondo Info View]");
        gTextWriter.SetCursor(x + AxisXLeft, y);
        gTextWriter.Print("[Left]");
        gTextWriter.SetCursor(x + AxisXRight, y);
        gTextWriter.Print("[Right]");
        AxisYCount++;

        // UniqueID
        gTextWriter.SetCursor(x, y + (25 * AxisYCount));
        gTextWriter.Print("Unique ID(MSB4)：");
        gTextWriter.SetCursor(x + AxisXLeft, y + (25 * AxisYCount));
        for (int i = 0; i < 4; i++)
        {
            gTextWriter.Print("%02x", dataL[index].uniqueId[0][i]);
        }
        gTextWriter.SetCursor(x + AxisXRight, y + (25 * AxisYCount));
        for (int i = 0; i < 4; i++)
        {
            gTextWriter.Print("%02x", dataR[index].uniqueId[0][i]);
        }
        gTextWriter.Print("\n");
        AxisYCount++;

        // FwVersion
        gTextWriter.SetCursor(x, y + (25 * AxisYCount));
        gTextWriter.Print("FwVersion：");
        gTextWriter.SetCursor(x + AxisXLeft, y + (25 * AxisYCount));
        gTextWriter.Print("%04x", dataL[index].fwVer[0]);
        gTextWriter.SetCursor(x + AxisXRight, y + (25 * AxisYCount));
        gTextWriter.Print("%04x", dataR[index].fwVer[0]);
        gTextWriter.Print("\n");
        AxisYCount++;

        // 歪センサーサンプリング値
        gTextWriter.SetCursor(x, y + (25 * AxisYCount));
        gTextWriter.Print("SamplingNumber：");
        gTextWriter.SetCursor(x + AxisXLeft, y + (25 * AxisYCount));
        gTextWriter.Print("%d", dataL[index].samplingNumber[0]);
        gTextWriter.SetCursor(x + AxisXRight, y + (25 * AxisYCount));
        gTextWriter.Print("%d", dataR[index].samplingNumber[0]);
        AxisYCount++;

        // 歪センサー生値(PollingMode)
        gTextWriter.SetCursor(x, y + (25 * AxisYCount));
        gTextWriter.Print("Strain：");
        gTextWriter.SetCursor(x + AxisXLeft, y + (25 * AxisYCount));
        gTextWriter.Print("%d", dataL[index].strain[0]);
        gTextWriter.SetCursor(x + AxisXRight, y + (25 * AxisYCount));
        gTextWriter.Print("%d", dataR[index].strain[0]);
        AxisYCount++;

        // Thermister生値(PollingMode)
        gTextWriter.SetCursor(x, y + (25 * AxisYCount));
        gTextWriter.Print("Thermister：");
        gTextWriter.SetCursor(x + AxisXLeft, y + (25 * AxisYCount));
        gTextWriter.Print("%d", dataL[index].thermistor[0]);
        gTextWriter.SetCursor(x + AxisXRight, y + (25 * AxisYCount));
        gTextWriter.Print("%d", dataR[index].thermistor[0]);
        AxisYCount++;

        // 工程Cal
        gTextWriter.SetCursor(x, y + (25 * AxisYCount));
        gTextWriter.Print("SCal(Push～Pull)：");
        gTextWriter.SetCursor(x + AxisXLeft, y + (25 * AxisYCount));
        gTextWriter.Print("%d～%d", dataL[index].scal[0].osMax, dataL[index].scal[0].hkMax);
        gTextWriter.SetCursor(x + AxisXRight, y + (25 * AxisYCount));
        gTextWriter.Print("%d～%d", dataR[index].scal[0].osMax, dataR[index].scal[0].hkMax);
        AxisYCount++;

        gTextWriter.SetCursor(x, y + (25 * AxisYCount));
        gTextWriter.Print("SCal(ZMax～ZMin)：");
        gTextWriter.SetCursor(x + AxisXLeft, y + (25 * AxisYCount));
        gTextWriter.Print("%d～%d", dataL[index].scal[0].zeroMax, dataL[index].scal[0].zeroMin);
        gTextWriter.SetCursor(x + AxisXRight, y + (25 * AxisYCount));
        gTextWriter.Print("%d～%d", dataR[index].scal[0].zeroMax, dataR[index].scal[0].zeroMin);
        AxisYCount++;

        // ユーザーCal
        gTextWriter.SetCursor(x, y + (25 * AxisYCount));
        gTextWriter.Print("UCal(Push～Pull)：");
        gTextWriter.SetCursor(x + AxisXLeft, y + (25 * AxisYCount));
        gTextWriter.Print("%d～%d", dataL[index].ucal[0].osMax, dataL[index].ucal[0].hkMax);
        gTextWriter.SetCursor(x + AxisXRight, y + (25 * AxisYCount));
        gTextWriter.Print("%d～%d", dataR[index].ucal[0].osMax, dataR[index].ucal[0].hkMax);
        AxisYCount++;

        gTextWriter.SetCursor(x, y + (25 * AxisYCount));
        gTextWriter.Print("UCal(ZMax～ZMin)：");
        gTextWriter.SetCursor(x + AxisXLeft, y + (25 * AxisYCount));
        gTextWriter.Print("%d～%d", dataL[index].ucal[0].zeroMax, dataL[index].ucal[0].zeroMin);
        gTextWriter.SetCursor(x + AxisXRight, y + (25 * AxisYCount));
        gTextWriter.Print("%d～%d", dataR[index].ucal[0].zeroMax, dataR[index].ucal[0].zeroMin);

    } // NOLINT(impl/function_size)

    ///////////////////////////////////////////////////////////////////
    // ツール操作に応じたAPI実行と、ツール内モード遷移処理
    ///////////////////////////////////////////////////////////////////
    void CallHidbusFunction(size_t index)
    {
        // 遷移前モードの格納
        BusStatus stateLPrev = stateL[index].mode;
        BusStatus stateRPrev = stateR[index].mode;

        // 左JoyCon
        if (cursol[index].IsSelectedLeft == true && cursol[index].menustate == nnt::Menustate_ForcusLeft)
        {
            switch (cursol[index].menu)
            {
                case nnt::Menu_InitFin:
                    nnt::menuf::FuncInitFin(&cursol[index], &stateL[index], index);
                    break;
                case nnt::Menu_Hidbus:
                    nnt::menuf::FuncHidbus(&cursol[index], &stateL[index]);
                    break;
                case nnt::Menu_TransMode:
                    nnt::menuf::FuncTransMode(&cursol[index], &stateL[index], &dataL[index], index);
                    break;
                case nnt::Menu_Data:
                    nnt::menuf::FuncData(&cursol[index], &stateL[index]);
                    break;
                case nnt::Menu_PollingNumSelect:
                    nnt::menuf::FuncPollingNumSelect(&cursol[index], &stateL[index]);
                    break;
                default:
                    break;
            }
        }
        // 右JoyCon
        if (cursol[index].IsSelectedRight == true && cursol[index].menustate == nnt::Menustate_ForcusRight)
        {
            switch (cursol[index].menu)
            {
                case nnt::Menu_InitFin:
                    nnt::menuf::FuncInitFin(&cursol[index], &stateR[index], index);
                    break;
                case nnt::Menu_Hidbus:
                    nnt::menuf::FuncHidbus(&cursol[index], &stateR[index]);
                    break;
                case nnt::Menu_TransMode:
                    nnt::menuf::FuncTransMode(&cursol[index], &stateR[index], &dataR[index], index);
                    break;
                case nnt::Menu_Data:
                    nnt::menuf::FuncData(&cursol[index], &stateR[index]);
                    break;
                case nnt::Menu_PollingNumSelect:
                    nnt::menuf::FuncPollingNumSelect(&cursol[index], &stateR[index]);
                    break;
                default:
                    break;
            }
        }
        // PollingData取得
        if (stateL[index].isPolling == IsPolling_Enable)
        {
            nnt::menuf::PollingReadRondeData(&stateL[index], &dataL[index], index);
        }
        if (stateR[index].isPolling == IsPolling_Enable)
        {
            nnt::menuf::PollingReadRondeData(&stateR[index], &dataR[index], index);
        }

        if (stateL[index].mode != stateLPrev)
        {
            NN_LOG("--------------------------------------------------------\n");
            NN_LOG("[Npad : %d]Change State(L) : %s ---> %s\n", index + 1, ToString(stateLPrev), ToString(stateL[index].mode));
            if (nn::oe::GetCurrentFocusState() == nn::oe::FocusState_InFocus)
            {
                NN_LOG("[ToolRun] : FG Running\n");
            }
            else
            {
                NN_LOG("[ToolRun] : BG Running\n");
            }
            NN_LOG("--------------------------------------------------------\n\n");
        }
        if (stateR[index].mode != stateRPrev)
        {
            NN_LOG("--------------------------------------------------------\n");
            NN_LOG("[Npad : %d]BusState(R) : %s ---> %s\n", index + 1, ToString(stateRPrev), ToString(stateR[index].mode));
            if (nn::oe::GetCurrentFocusState() == nn::oe::FocusState_InFocus)
            {
                NN_LOG("[ToolRun] : FG Running\n");
            }
            else
            {
                NN_LOG("[ToolRun] : BG Running\n");
            }
            NN_LOG("--------------------------------------------------------\n\n");
        }

    }

    ///////////////////////////////////////////////
    // 描画関係を詰め込むユーティリティ関数
    ///////////////////////////////////////////////
    void DrawAndUpdateUtility(size_t index)
    {
        nn::util::Float2 pos;
        nn::util::Float2 size;

        pos.x = 10.0f;
        pos.y = 10 + 330 * (index % 2);
        size.x = 1070.0f;
        size.y = 320.0f;

        if ((drawMode == DrawMode::DrawMode_1to2 && (index <= 1))
         || (drawMode == DrawMode::DrawMode_3to4 && (index >= 2 && index <= 3))
         || (drawMode == DrawMode::DrawMode_5to6 && (index >= 4 && index <= 5))
         || (drawMode == DrawMode::DrawMode_7to8 && (index >= 6 && index <= 7))
         || (drawMode == DrawMode::DrawMode_Hh && (index == 8)))
        {
            gTextWriter.SetTextColor(nn::util::Color4u8::Black());
            gTextWriter.SetScale(1.0f, 1.0f);
            gTextWriter.SetCursor(pos.x + 5, pos.y);

            if (gController.GetControllerList()[index]->isConnected())
            {
                gDrawer.SetColor(nn::util::Color4u8::Black());
                gTextWriter.SetTextColor(nn::util::Color4u8::Black());
            }
            else
            {
                gDrawer.SetColor(nn::util::Color4u8::Gray());
                gTextWriter.SetTextColor(nn::util::Color4u8::Gray());
            }
            gDrawer.Draw2DFrame(pos, size, 2.0f);

            // NpadId及びNpadStyleの表示
            gTextWriter.Print("NpadID:%d　%s", index + 1, ToString(gController.GetControllerList()[index]->GetStyleSet()));

            // Hidbusステートの表示
            gTextWriter.SetCursor(pos.x + 300, pos.y);
            gTextWriter.Print("BusState(L)：%s", ToString(stateL[index]));
            gTextWriter.SetCursor(pos.x + 300, pos.y + 20);
            gTextWriter.Print("BusState(R)：%s", ToString(stateR[index]));

            // Polling実施時の点滅表示
            if (stateL[index].isPolling == IsPolling_Enable)
            {
                if (PollingCount[index] > 0 && PollingCount[index] < 30)
                {
                    gTextWriter.SetTextColor(nn::util::Color4u8::Red());
                    gTextWriter.SetCursor(pos.x + 830, pos.y);
                    gTextWriter.Print("!!");
                }
            }
            if (stateR[index].isPolling == IsPolling_Enable)
            {
                if (PollingCount[index] > 0 && PollingCount[index] < 30)
                {
                    gTextWriter.SetTextColor(nn::util::Color4u8::Red());
                    gTextWriter.SetCursor(pos.x + 830, pos.y + 20);
                    gTextWriter.Print("!!");
                }
            }

            // 排他機能操作のUpdate及びボタン描画
            nnt::UpdateExclusiveFuncButtons(index);
            nnt::DrawExclusiveFuncButtons(index);

            // カーソル操作のUpdate及び描画
            if (IsAssignmentSelectMode == false)
            {
                nnt::GetAndDrawCursolPosition(&cursol[index], pos.x + 230, pos.y + 70, index);
            }
            // Rondeから取得した値の描画
            DrawRondeInfo(pos.x + 670, pos.y + 50, index);

            // Npad操作状態の表示
            nnt::PrintNpadButtonState(pos.x + 50, pos.y + 250, index);
        }
    }

    ///////////////////////////////////////////////
    // ツールVersionの表示
    ///////////////////////////////////////////////
    void DrawToolVersion(float x, float y)
    {
        nn::util::Float2 pos;
        nn::util::Float2 size;
        nn::util::Float2 LineBegin;
        nn::util::Float2 LineEnd;

        const float yOffset = 30.0f;
        const size_t dataNum = 2;

        gTextWriter.SetCursor(x, y - 30);
        gTextWriter.Print("RondeTestToolForMCT");

        pos.x = x;
        pos.y = y;
        size.x = 180.0f;
        size.y = yOffset * dataNum;

        LineBegin.x = pos.x + 120;
        LineBegin.y = pos.y;
        LineEnd.x = LineBegin.x;
        LineEnd.y = LineBegin.y + (yOffset * dataNum);

        gDrawer.SetColor(nn::util::Color4u8::Gray());
        gDrawer.Draw2DFrame(pos, size, 2.f);
        gDrawer.Draw2DLine(LineBegin, LineEnd);

        for (auto i = 0; i < dataNum - 1; i++)
        {
            LineBegin.x = pos.x;
            LineBegin.y = pos.y + yOffset * (i + 1);
            LineEnd.x = LineBegin.x + size.x;
            LineEnd.y = LineBegin.y;

            gDrawer.Draw2DLine(LineBegin, LineEnd);
        }

        nn::oe::DisplayVersion version;
        nn::oe::GetDisplayVersion(&version);

        gTextWriter.SetTextColor(nn::util::Color4u8::Black());
        gTextWriter.SetCursor(x + 5, y);
        gTextWriter.Print("AppVer");
        gTextWriter.SetCursor(x + 125, y);
        gTextWriter.Print("%s", version.value);

        gTextWriter.SetCursor(x + 5, y + yOffset + 2);
        gTextWriter.Print("BG Running");
        gTextWriter.SetCursor(x + 125, y + yOffset + 2);
        gTextWriter.Print("%s", (forcusmode == nn::oe::FocusHandlingMode_Notify) ? "ON" : "OFF");
    }
}//namespace

extern "C" void nnMain()
{
    gGraphics.Initialize();     //!< グラフィックス機能の初期化
    gInput.Initialize();        //!< 入力関連機能の初期化


    for (auto i = 0; i < NpadIdNum; i++)
    {
        nnt::InitializeExclusiveFunc(i);
        ///////////////////////////////////////////////////////////////////////////////////////
        // 妙案無いので、NN_SDK_ASSERT_LESS_EQUAL(index, MaxHidbusNumber)で引っ掛けます。
        ///////////////////////////////////////////////////////////////////////////////////////
        //stateL[i].handle._storage = UINT64_MAX;
        //stateR[i].handle._storage = UINT64_MAX;

        // カーソルの初期設定
        nnt::InitializeCursol(&cursol[i], i);

        stateL[i].PollingCount = 1;
        stateR[i].PollingCount = 1;

        memcpy(stateL[i].command, nnt::hidbus::CmdList.UniqueId, 4);
        memcpy(stateR[i].command, nnt::hidbus::CmdList.UniqueId, 4);
        stateL[i].packetSize = nnt::hidbus::SizeList.UniqueId;
        stateR[i].packetSize = nnt::hidbus::SizeList.UniqueId;
    }
    nnt::MakeAppletButtons(1100, 370);
    nnt::MakeExclusiveFuncButtons(20, 70);

    nn::oe::SetFocusHandlingMode(nn::oe::FocusHandlingMode_InFocusOnly);
    uint64_t    frameCount = 0;         // フレームカウント
    bool        fin = false;            // trueになったらループ終了

    // 全体のループ
    while (!fin)
    {
        gInput.Update();
        nnt::UpdateAppletButtons();
        //////////////////////////////////
        // 描画開始
        //////////////////////////////////
        gDrawer.BeginDraw();
        gDrawer.SetClearColor(nn::util::Color4u8::Gray());

        for (auto index = 0; index < NpadIdNum; index++)
        {
            nnt::Active(index);
            nnt::Deactive(index);
            UpdateData(&stateL[index], &dataL[index], index);
            UpdateData(&stateR[index], &dataR[index], index);
            CallHidbusFunction(index);
            DrawAndUpdateUtility(index);
        }
        nnt::DrawAppletButtons();
        DrawToolVersion(1090, 40);

        // Npadアサインの選択及び描画
        if (IsAssignmentSelectMode)
        {
            SelectedNpadAssign();
            DoNpadAssign();
            DrawNpadAssignMenu(30, 665);
        }
        ///////////////
        // 描画終了
        ///////////////
        gDrawer.EndDraw();
        ++frameCount;

        ////////////////////////
        // ボタン入力の判定
        ////////////////////////
        fin = IsPushed();
    }

    NN_LOG("Hidbus Ronde Sample Done\n");

    for (auto index = 0; index < NpadIdNum; index++)
    {
        if (stateL[index].mode != HidbusState_NotGotHandle)
        {
            nn::hidbus::Finalize(stateL[index].handle);
        }
        if (stateR[index].mode != HidbusState_NotGotHandle)
        {
            nn::hidbus::Finalize(stateR[index].handle);
        }
    }
    gInput.Finalize();
    gGraphics.Finalize();
}
