﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 * アプレット呼び出し系関数の集まり
 */

#include "Applet.h"

nns::hidfw::layout::ButtonGroup    AppletSelectGroup;        //!< ボタンのグループ
nns::hidfw::layout::ButtonGroup    ThredSelectGroup;         //!< ボタンのグループ

float xOrigin;
float yOrigin;

const size_t ThreadStackSize = 1024 * 8;
NN_OS_ALIGNAS_THREAD_STACK char thredStack[ThreadStackSize];   // スレッドのスタック
nn::os::ThreadType thred;
void* dummy;

bool thredNumber = false;

const char* buttonText[] =
{
    "ControlSup",
    "AccountSel",
    "SWKeyBoard",
    "ErrorViewer",
    "ForceAbort"
};

const char* thredText[] =
{
    "0",
    "1"
};

//======================================================================
// アプレット呼び出しアイテムの初期設定
// param    x：描画の左上頂点X座標
// param    y：描画の左上頂点Y座標
//======================================================================
void nnt::MakeAppletButtons(float x, float y)
{
    xOrigin = x;
    yOrigin = y;

    // コールする関数の登録
    nns::hidfw::layout::LayoutFunction pFunc[] =
    {
        nnt::CallControllerSupportApplet,
        nnt::CallAccountSelectApplet,
        nnt::CallSwKeyBoardApplet,
        nnt::CallErrorViewerApplet,
        nnt::CallNullAccess,
    };

    // アプレット呼び出しボタンの表示
    for (size_t i = 0; i < NN_ARRAY_SIZE(buttonText); ++i)
    {
        nns::hidfw::layout::Button button;
        button.SetPos(xOrigin + 15, yOrigin + 40 * i);
        button.SetSize(130, 30);
        button.SetText(buttonText[i]);
        button.SetTextColor(nn::util::Color4u8::Black());
        button.OnOffMode(false);                                // 押したら、自動で押される前の状態に戻るようにします
        button.ThroughCancel(true);                             // キャンセルを無効化します
        button.ThroughSelect(true);
        button.ThroughChoose(true);
        button.SetFunc(pFunc[i]);
        AppletSelectGroup.Add(new nns::hidfw::layout::Button(button));
        AppletSelectGroup.SetOrientation(nns::hidfw::layout::BaseItemSet::eOrientation_Vertical); // ボタン配置を縦方向に変更
        AppletSelectGroup.Focus(true);
    }

    // コールする関数の登録
    nns::hidfw::layout::LayoutFunction pFunc2[] =
    {
        nnt::ChangeThredZero,
        nnt::ChangeThredOne
    };

    // スレッド切替ボタンの表示
    for (size_t i = 0; i < NN_ARRAY_SIZE(thredText); ++i)
    {
        nns::hidfw::layout::Button button;
        button.SetPos(xOrigin + 15 + 80 * i, yOrigin + 240);
        button.SetSize(50, 30);
        button.SetText(thredText[i]);
        button.SetTextColor(nn::util::Color4u8::Black());
        button.ThroughCancel(true);                             // キャンセルを無効化します
        button.ThroughSelect(true);
        button.ThroughChoose(true);
        button.SetFunc(pFunc2[i]);
        ThredSelectGroup.Add(new nns::hidfw::layout::Button(button));
        ThredSelectGroup.SetOrientation(nns::hidfw::layout::BaseItemSet::eOrientation_Horizontal); // ボタン配置を横方向に変更
        ThredSelectGroup.Focus(true);
        ThredSelectGroup.FocusItem(0);
        ThredSelectGroup.Select(0);
    }
}

//======================================================================
// アプレット呼び出しアイテムの更新処理
//======================================================================
void nnt::UpdateAppletButtons()
{
    AppletSelectGroup.Update();
    ThredSelectGroup.Update();
    if (thred._state == thred.State_Exited)
    {
        nn::os::DestroyThread(&thred);
    }
}

//======================================================================
// アプレット呼び出しアイテムの描画処理
// gDrawer.BeginDraw()～EndDraw()内でコールすることを前提としています
//======================================================================
void nnt::DrawAppletButtons()
{
    AppletSelectGroup.Draw();
    ThredSelectGroup.Draw();

    nn::util::Float2 pos;
    pos.x = xOrigin - 10.0f;
    pos.y = yOrigin - 30.0f;

    nn::util::Float2 size;
    size.x = 180.0f;
    size.y = 40.0f * (NN_ARRAY_SIZE(buttonText) + NN_ARRAY_SIZE(thredText)) + 30.0f;

    gDrawer.SetColor(nn::util::Color4u8::Gray());
    gDrawer.Draw2DFrame(pos, size, 2.0f);
    gTextWriter.SetCursor(pos.x + 10.0f, pos.y);
    gTextWriter.SetTextColor(nn::util::Color4u8::Black());
    gTextWriter.SetScale(1.0f, 1.0f);
    gTextWriter.Print("Call LA(Only Touch)");

    gTextWriter.SetCursor(pos.x + 10.0f, pos.y + 240.0f);
    gTextWriter.Print("Change Thred Num");
}

//======================================================================
// コンサポアプレットの呼び出し
//======================================================================
void nnt::CallControllerSupportApplet(void* pushButton, void* param)
{
    if (thredNumber && thred._state == thred.State_NotInitialized)
    {
        nn::os::CreateThread(&thred, DoControllerSupportApplet, dummy, thredStack, ThreadStackSize, 1);
        nn::os::StartThread(&thred);
    }
    else if(!thredNumber)
    {
        dummy = (nullptr);
        DoControllerSupportApplet(dummy);
    }
    else
    {
        // do nothing
    }
}

//======================================================================
// アカウント選択アプレットの呼び出し
//======================================================================
void nnt::CallAccountSelectApplet(void* pushButton, void* param)
{
    if (thredNumber && thred._state == thred.State_NotInitialized)
    {
        nn::os::CreateThread(&thred, DoAccountSelectApplet, dummy, thredStack, ThreadStackSize, 1);
        nn::os::StartThread(&thred);
    }
    else if (!thredNumber)
    {
        dummy = (nullptr);
        DoAccountSelectApplet(dummy);
    }
    else
    {
        // do nothing
    }
}

//======================================================================
// ソフトウェアキーボードアプレットの呼び出し
//======================================================================
void nnt::CallSwKeyBoardApplet(void* pushButton, void* param)
{
    if (thredNumber && thred._state == thred.State_NotInitialized)
    {
        nn::os::CreateThread(&thred, DoSwKeyBoardApplet, dummy, thredStack, ThreadStackSize, 1);
        nn::os::StartThread(&thred);
    }
    else if (!thredNumber)
    {
        dummy = (nullptr);
        DoSwKeyBoardApplet(dummy);
    }
    else
    {
        // do nothing
    }
}

//======================================================================
// エラービュアーアプレットの呼び出し
//======================================================================
void nnt::CallErrorViewerApplet(void* pushButton, void* param)
{
    if (thredNumber && thred._state == thred.State_NotInitialized)
    {
        nn::os::CreateThread(&thred, DoErrorViewerApplet, dummy, thredStack, ThreadStackSize, 1);
        nn::os::StartThread(&thred);
    }
    else if (!thredNumber)
    {
        dummy = (nullptr);
        DoErrorViewerApplet(dummy);
    }
    else
    {
        // do nothing
    }
}

//======================================================================
// NULLポインタアクセスの呼び出し
//======================================================================
void nnt::CallNullAccess(void* pushButton, void* param)
{
    if (thredNumber && thred._state == thred.State_NotInitialized)
    {
        nn::os::CreateThread(&thred, DoNullAccess, dummy, thredStack, ThreadStackSize, 1);
        nn::os::StartThread(&thred);
    }
    else if (!thredNumber)
    {
        dummy = (nullptr);
        DoNullAccess(dummy);
    }
    else
    {
        // do nothing
    }

}

void nnt::ChangeThredZero(void* pushButton, void* param)
{
    thredNumber = false;
}

void nnt::ChangeThredOne(void* pushButton, void* param)
{
    thredNumber = true;
}

//======================================================================
// コンサポアプレットの呼び出し
//======================================================================
void nnt::DoControllerSupportApplet(void* dummy)
{
    if (!dummy)
    {
        nn::hid::ControllerSupportResultInfo result;
        nn::hid::ControllerSupportArg ctrlArg;
        ctrlArg.SetDefault();
        ctrlArg.enableIdentificationColor = true;
        nn::hid::SetNpadJoyHoldType(nn::hid::NpadJoyHoldType_Horizontal);

        nn::hid::ShowControllerSupport(&result, ctrlArg);
    }
}

//======================================================================
// アカウント選択アプレットの呼び出し
//======================================================================
void nnt::DoAccountSelectApplet(void* dummy)
{
    if (!dummy)
    {
        nn::account::Initialize();
        nn::account::ShowUserCreator();
    }
}

//======================================================================
// ソフトウェアキーボードアプレットの呼び出し
//======================================================================
void nnt::DoSwKeyBoardApplet(void* dummy)
{
    if (!dummy)
    {
        const size_t ApplicationHeapSize = 128 * 1024;

        ApplicationHeap applicationHeap;
        applicationHeap.Initialize(ApplicationHeapSize);

        nn::swkbd::ShowKeyboardArg showKeyboardArg;
        nn::swkbd::MakePreset(&(showKeyboardArg.keyboardConfig), nn::swkbd::Preset_Default);

        // ガイド文字列の設定
        const char* guide_string = u8"検証お疲れ様です。いつもありがとうございます。";
        nn::swkbd::SetGuideTextUtf8(&showKeyboardArg.keyboardConfig,
                                    guide_string);

        // 共有メモリ用バッファの割り当て
        size_t in_heap_size = nn::swkbd::GetRequiredWorkBufferSize(false);
        void* swkbd_work_buffer = applicationHeap.Allocate(in_heap_size, nn::os::MemoryPageSize);

        showKeyboardArg.workBuf = swkbd_work_buffer;
        showKeyboardArg.workBufSize = in_heap_size;

        // 終了パラメータの設定
        size_t out_heap_size = nn::swkbd::GetRequiredStringBufferSize();
        nn::swkbd::String output_string;
        output_string.ptr = reinterpret_cast<char16_t*>(applicationHeap.Allocate(out_heap_size, nn::os::MemoryPageSize));
        output_string.bufSize = out_heap_size;

        nn::swkbd::ShowKeyboard(&output_string, showKeyboardArg);

        applicationHeap.Deallocate(output_string.ptr);
        applicationHeap.Deallocate(swkbd_work_buffer);

        applicationHeap.Finalize();
    }
}

//======================================================================
// エラービュアーアプレットの呼び出し
//======================================================================
void nnt::DoErrorViewerApplet(void* dummy)
{
    if (!dummy)
    {
        nn::err::ErrorCode errorCode;
        errorCode.category = 2162;
        errorCode.number = 0002;
        nn::err::ShowError(errorCode);
    }
}

//======================================================================
// NULLポインタアクセスの呼び出し
//======================================================================
void nnt::DoNullAccess(void* dummy)
{
    if (!dummy)
    {
        NN_ABORT_UNLESS_NOT_NULL(nullptr);
    }
}
