﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nnt/nntest.h>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>

#include <nn/hidbus/hidbus.h>

//#define ENABLE_SYSTEM_DATA_WRITE

/*
// Ext Data Command Protocol
|     8bit   |    8bit  |   8bit    |   8bit   | 32 bit |
| DeviceType | DataName | CommandId | DataSize |  Data  |
*/
enum ExtDataProtocolOffset
{
    ExtDataProtocolOffset_DeviceType = 0,
    ExtDataProtocolOffset_DeviceName = 1,
    ExtDataProtocolOffset_CommandId = 2,
    ExtDataProtocolOffset_DataSize = 3,
    ExtDataProtocolOffset_Data = 4,
};

enum DeviceType
{
    DeviceType_DeviceInfo = 0x00,
    DeviceType_RondePollingFormat = 0x01,
    DeviceType_16BitExternalAdc = 0x02,
    DeviceType_12BitInternalAdc = 0x03,
    DeviceType_InternalFlash = 0x04,
};

enum CommandId
{
    CommandId_Write = 0x01,
    CommandId_Read = 0x02,
};

enum InternalFlash_DataName
{
    InternalFlash_DataName_ManuCalPushMax = 0x01,
    InternalFlash_DataName_ManuCalPullMax = 0x02,
    InternalFlash_DataName_ManuCal0Min = 0x03,
    InternalFlash_DataName_ManuCal0Max = 0x04,
    InternalFlash_DataName_UserCalPushMax = 0x11,
    InternalFlash_DataName_UserCalPullMax = 0x12,
    InternalFlash_DataName_UserCal0Min = 0x13,
    InternalFlash_DataName_UserCal0Max = 0x14,
    InternalFlash_DataName_ManuLog1 = 0x21,
    InternalFlash_DataName_ManuLog2 = 0x22,
    InternalFlash_DataName_ManuLog3 = 0x23,
    InternalFlash_DataName_ManuLog4 = 0x24,
    InternalFlash_DataName_ManuLog5 = 0x25,
    InternalFlash_DataName_ManuLog6 = 0x26,
    InternalFlash_DataName_DebugFlag = 0xa0,
};

enum DeviceInfo_DataName
{
    DeviceInfo_DataName_FwVersion = 0x00,
    DeviceInfo_DataName_DeviceId = 0x01,
};

enum InternalAdc_DataName
{
    InternalAdc_DataName_StrainGauge = 0x00,
    InternalAdc_DataName_Thermistor = 0x01,
    InternalAdc_DataName_1_8V = 0x02,
    InternalAdc_DataName_2_5V = 0x03,
};

namespace
{

    const int               responseHeaderSize = 4;
    const int  RondeId = 0x20;

    // 対向デバイスへの Internal ADC (Thermistor)取得コマンド
    uint8_t getThermistorValueCommand[4] =
    {
        DeviceType_12BitInternalAdc,
        InternalAdc_DataName_Thermistor,
        CommandId_Read,
        0x00 // なし
    };

    // 対向デバイスへの Internal ADC 取得コマンド
    /*
    uint8_t getInternalAdcValueCommand[4] =
    {
        DeviceType_12BitInternalAdc,
        InternalAdc_DataName_StrainGauge,
        CommandId_Read,
        0x00 // なし
    };
    */

    // 対向デバイスへの Internal ADC (1.8V) 取得コマンド
    uint8_t getInternalAdc1_8ValueCommand[4] =
    {
        DeviceType_12BitInternalAdc,
        InternalAdc_DataName_1_8V,
        CommandId_Read,
        0x00 // なし
    };

    // 対向デバイスへの Internal ADC (2.5V) 取得コマンド
    uint8_t getInternalAdc2_5ValueCommand[4] =
    {
        DeviceType_12BitInternalAdc,
        InternalAdc_DataName_2_5V,
        CommandId_Read,
        0x00 // なし
    };

#ifdef ENABLE_SYSTEM_DATA_WRITE
    // 対向デバイスの Flash への書き込みコマンド
    uint8_t writeDataCommand[8] =
    {
        DeviceType_InternalFlash,
        InternalFlash_DataName_ManuCalPushMax,
        CommandId_Write,
        0x04,
        0x01,
        0x02,
        0x03,
        0x04,
    };

    // 対向デバイスの Flash への大きめのデータの書き込みコマンド
    uint8_t writeLog1Command[20] =
    {
        DeviceType_InternalFlash,
        InternalFlash_DataName_ManuLog1,
        CommandId_Write,
        0x10,
        0x00, 0x01, 0x02, 0x03,
        0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0a, 0x0b,
        0x0c, 0x0d, 0x0e, 0x0f,
    };

    uint8_t writeLog2Command[20] =
    {
        DeviceType_InternalFlash,
        InternalFlash_DataName_ManuLog2,
        CommandId_Write,
        0x10,
        0x10, 0x11, 0x12, 0x13,
        0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b,
        0x1c, 0x1d, 0x1e, 0x1f,
    };

    uint8_t writeLog3Command[20] =
    {
        DeviceType_InternalFlash,
        InternalFlash_DataName_ManuLog3,
        CommandId_Write,
        0x10,
        0x20, 0x21, 0x22, 0x23,
        0x24, 0x25, 0x26, 0x27,
        0x28, 0x29, 0x2a, 0x2b,
        0x2c, 0x2d, 0x2e, 0x2f,
    };

    uint8_t writeLog4Command[20] =
    {
        DeviceType_InternalFlash,
        InternalFlash_DataName_ManuLog4,
        CommandId_Write,
        0x10,
        0x30, 0x31, 0x32, 0x33,
        0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x3a, 0x3b,
        0x3c, 0x3d, 0x3e, 0x3f,
    };

    uint8_t writeLog5Command[20] =
    {
        DeviceType_InternalFlash,
        InternalFlash_DataName_ManuLog5,
        CommandId_Write,
        0x10,
        0x40, 0x41, 0x42, 0x43,
        0x44, 0x45, 0x46, 0x47,
        0x48, 0x49, 0x4a, 0x4b,
        0x4c, 0x4d, 0x4e, 0x4f,
    };

    uint8_t writeLog6Command[20] =
    {
        DeviceType_InternalFlash,
        InternalFlash_DataName_ManuLog6,
        CommandId_Write,
        0x10,
        0x5a, 0x51, 0x52, 0x53,
        0x54, 0x55, 0x56, 0x57,
        0x58, 0x59, 0x5a, 0x5b,
        0x5c, 0x5d, 0x5e, 0x5f,
    };
#endif

    // 対向デバイスの Flash からの読み込みコマンド
    uint8_t readDataCommand[4] =
    {
        DeviceType_InternalFlash,
        InternalFlash_DataName_ManuCalPushMax,
        CommandId_Read,
        0x00,
    };

    // 対向デバイスの Flash からの読み込みコマンド
    uint8_t readLog1Command[4] =
    {
        DeviceType_InternalFlash,
        InternalFlash_DataName_ManuLog1,
        CommandId_Read,
        0x00,
    };

    uint8_t readLog2Command[4] =
    {
        DeviceType_InternalFlash,
        InternalFlash_DataName_ManuLog2,
        CommandId_Read,
        0x00,
    };

    uint8_t readLog3Command[4] =
    {
        DeviceType_InternalFlash,
        InternalFlash_DataName_ManuLog3,
        CommandId_Read,
        0x00,
    };

    uint8_t readLog4Command[4] =
    {
        DeviceType_InternalFlash,
        InternalFlash_DataName_ManuLog4,
        CommandId_Read,
        0x00,
    };

    uint8_t readLog5Command[4] =
    {
        DeviceType_InternalFlash,
        InternalFlash_DataName_ManuLog5,
        CommandId_Read,
        0x00,
    };

    uint8_t readLog6Command[4] =
    {
        DeviceType_InternalFlash,
        InternalFlash_DataName_ManuLog6,
        CommandId_Read,
        0x00,
    };

    void PrintDeviceVersion(const nn::hidbus::BusHandle& handle)
    {
        size_t      outDataSize;
        uint8_t     receivedData[16];
        // 対向デバイスへのバージョン取得コマンド
        uint8_t                 getVersionCommand[4] = {};
        getVersionCommand[ExtDataProtocolOffset_DeviceType] = DeviceType_DeviceInfo;
        getVersionCommand[ExtDataProtocolOffset_DeviceName] = DeviceInfo_DataName_FwVersion;
        getVersionCommand[ExtDataProtocolOffset_CommandId] = CommandId_Read;
        getVersionCommand[ExtDataProtocolOffset_DataSize] = 0x00; // なし

        nn::hidbus::SendAndReceive(&outDataSize, receivedData, responseHeaderSize + 2, handle, getVersionCommand, sizeof(getVersionCommand));

        if (outDataSize == responseHeaderSize + 2 && receivedData[0] == 0x00)
        {
            uint8_t subVersion = receivedData[responseHeaderSize + 0];
            uint8_t mainVersion = receivedData[responseHeaderSize + 1];
            NN_LOG("[hidbus] FW version  = %x.%x\n", mainVersion, subVersion);
        }
        else
        {
            NN_LOG("[hidbus] FW version ERROR\n");
        }
    }

    void PrintDeviceSerial(const nn::hidbus::BusHandle& handle)
    {
        size_t      outDataSize;
        uint8_t     receivedData[16];

        // 対向デバイスへのデバイス ID 取得コマンド
        uint8_t                 getDeviceIdCommand[4] = {};
        getDeviceIdCommand[ExtDataProtocolOffset_DeviceType] = DeviceType_DeviceInfo;
        getDeviceIdCommand[ExtDataProtocolOffset_DeviceName] = DeviceInfo_DataName_DeviceId;
        getDeviceIdCommand[ExtDataProtocolOffset_CommandId] = CommandId_Read;
        getDeviceIdCommand[ExtDataProtocolOffset_DataSize] = 0x00; // なし

        nn::hidbus::SendAndReceive(&outDataSize, receivedData, responseHeaderSize + 12, handle, getDeviceIdCommand, sizeof(getDeviceIdCommand));

        uint32_t deviceId[3];
        if (outDataSize == responseHeaderSize + 12 && receivedData[0] == 0x00)
        {
            deviceId[0] = receivedData[responseHeaderSize + 3] << 24 | receivedData[responseHeaderSize + 2] << 16 | receivedData[responseHeaderSize + 1] << 8 | receivedData[responseHeaderSize + 0];
            deviceId[1] = receivedData[responseHeaderSize + 7] << 24 | receivedData[responseHeaderSize + 6] << 16 | receivedData[responseHeaderSize + 5] << 8 | receivedData[responseHeaderSize + 4];
            deviceId[2] = receivedData[responseHeaderSize + 11] << 24 | receivedData[responseHeaderSize + 10] << 16 | receivedData[responseHeaderSize + 9] << 8 | receivedData[responseHeaderSize + 8];

            // 上位ビットから表示
            NN_LOG("[hidbus] Device ID   = ");
            for (int i = 0; i < 3; i++)
            {
                NN_LOG("%8x ", deviceId[i]);
            }
            NN_LOG("\n");
        }
        else
        {
            NN_LOG("[hidbus] Device ID ERROR\n");
        }
    }

    void PrintPollingFormat(const nn::hidbus::BusHandle& handle)
    {
        size_t      outDataSize;
        uint8_t     receivedData[16];

        // 対向デバイスへのデバイス ID 取得コマンド
        uint8_t                 getRondePollingFormatCommand[4] = {};
        getRondePollingFormatCommand[ExtDataProtocolOffset_DeviceType] = DeviceType_RondePollingFormat;
        getRondePollingFormatCommand[ExtDataProtocolOffset_DeviceName] = 0x01; // なし
        getRondePollingFormatCommand[ExtDataProtocolOffset_CommandId] = CommandId_Read;
        getRondePollingFormatCommand[ExtDataProtocolOffset_DataSize] = 0x00; // なし

        auto result = nn::hidbus::SendAndReceive(&outDataSize, receivedData, responseHeaderSize + 4, handle, getRondePollingFormatCommand, sizeof(getRondePollingFormatCommand));
        if (result.IsSuccess())
        {
            if (outDataSize == responseHeaderSize + 4 && receivedData[0] == 0x00)
            {
                auto strainGaugeData = receivedData[responseHeaderSize + 1] << 8 | receivedData[responseHeaderSize + 0];
                int16_t* pInt16Data = reinterpret_cast<int16_t*>(&strainGaugeData);

                auto thermistorData = receivedData[responseHeaderSize + 3] << 8 | receivedData[responseHeaderSize + 2];
                NN_LOG("[hidbus] Polling Format Strain Gauge = %d, Thermistor   = %d\n", *pInt16Data, thermistorData);
            }
            else
            {
                NN_LOG("[hidbus] Get Polling Format ERROR\n");
            }
        }
        else
        {
            if (nn::hidbus::ResultExternalDeviceNotEnabled::Includes(result))
            {
                NN_LOG("[hidbus] Detached!\n");
            }
            else
            {
                NN_ABORT("Unknown Error\n");
            }
        }
    }

    void CheckAndPrintInvalidCommand(const nn::hidbus::BusHandle& handle)
    {
        size_t      outDataSize;
        uint8_t     receivedData[16];

        // Invalid Command のテスト用のコマンド
        uint8_t                 invalidCommand[4] = {};
        invalidCommand[ExtDataProtocolOffset_DeviceType] = DeviceType_DeviceInfo;
        invalidCommand[ExtDataProtocolOffset_DeviceName] = DeviceInfo_DataName_DeviceId;
        invalidCommand[ExtDataProtocolOffset_CommandId] = CommandId_Write;
        invalidCommand[ExtDataProtocolOffset_DataSize] = 0x00; // なし

        nn::hidbus::SendAndReceive(&outDataSize, receivedData, responseHeaderSize + 2, handle, invalidCommand, sizeof(invalidCommand));
        if (receivedData[0] != 0x00)
        {
            NN_LOG("[hidbus] Invalid Command Check, ErrorCode = %x\n", receivedData[0]);
        }
    }

    /*
    void PrintExternalStrainGaugeAdcValue(const nn::hidbus::BusHandle& handle)
    {
        size_t      outDataSize;
        uint8_t     receivedData[16];

        // 対向デバイスへの External ADC 取得コマンド
        uint8_t                 getExternalAdcValueCommand[4];
        getExternalAdcValueCommand[ExtDataProtocolOffset_DeviceType] = DeviceType_16BitExternalAdc;
        getExternalAdcValueCommand[ExtDataProtocolOffset_DeviceName] = 0x00; // 指定なし
        getExternalAdcValueCommand[ExtDataProtocolOffset_CommandId] = CommandId_Read;
        getExternalAdcValueCommand[ExtDataProtocolOffset_DataSize] = 0x00; // なし

        nn::hidbus::SendAndReceive(&outDataSize, receivedData, responseHeaderSize + 4, getExternalAdcValueCommand, sizeof(getExternalAdcValueCommand), handle);

        // データを確認しやすい形に変換
        auto data = receivedData[responseHeaderSize + 1] << 8 | receivedData[responseHeaderSize + 0];
        int16_t* pInt16Data = reinterpret_cast<int16_t*>(&data);
        NN_LOG("[hidbus] External ADC value = %d\n", *pInt16Data);
    }
    */

    void PrintInternalAdcValue(uint8_t* pCommand, size_t commandSize, nn::hidbus::BusHandle& handle)
    {
        size_t      outDataSize;
        uint8_t     receivedData[16];

        nn::hidbus::SendAndReceive(&outDataSize, receivedData, responseHeaderSize + 4, handle, pCommand, commandSize);

        // データを確認しやすい形に変換
        auto data = receivedData[responseHeaderSize + 1] << 8 | receivedData[responseHeaderSize + 0];
        NN_LOG("%d\n", data);

    }

#ifdef ENABLE_SYSTEM_DATA_WRITE
    void WriteDataToInternalFlash(uint8_t* pCommand, size_t commandSize,const nn::hidbus::BusHandle& handle)
    {
        size_t      outDataSize;
        uint8_t     receivedData[16];

        nn::hidbus::SendAndReceive(&outDataSize, receivedData, responseHeaderSize, handle, pCommand, commandSize);

        // データを確認しやすい形に変換
        NN_LOG("[hidbus] Write Result = %d\n", receivedData[0]);
    }
#endif

    void ReadDataFromInternalFlash(uint8_t* pCommand, size_t commandSize, const nn::hidbus::BusHandle& handle)
    {
        size_t      outDataSize;
        uint8_t     receivedData[16];

        nn::hidbus::SendAndReceive(&outDataSize, receivedData, responseHeaderSize + 4, handle, pCommand, commandSize);

        NN_LOG("[hidbus] Internal Data (0x%2x) = ", pCommand[1]);

        for (int i = 0; i < 4; i++)
        {
            NN_LOG("%x ", receivedData[responseHeaderSize + i]);
        }
        NN_LOG("\n");
    }

    void ReadLogDataFromInternalFlash(uint8_t* pCommand, size_t commandSize, const nn::hidbus::BusHandle& handle)
    {
        size_t      outDataSize;
        uint8_t     receivedData[36];

        nn::hidbus::SendAndReceive(&outDataSize, receivedData, responseHeaderSize + 16, handle, pCommand, commandSize);

        NN_LOG("[hidbus] Internal Data (0x%2x) = ", pCommand[1]);

        for (int i = 0; i < 16; i++)
        {
            NN_LOG("%x ", receivedData[responseHeaderSize + i]);
        }
        NN_LOG("\n");
    }
}

/**
* RONDE に対する基本的なテスト
*/
TEST(RondeTest, Basic)
{
    nn::hidbus::BusHandle handle = {};
    nn::Result              result;
    uint64_t                     count = 0;

    // NpadID のリスト
    const nn::hid::NpadIdType NpadIds[] =
    {
        nn::hid::NpadId::No1,
    };

    // Npad の初期化
    nn::hid::InitializeNpad();

    // Style の設定
    nn::hid::SetSupportedNpadStyleSet(
        nn::hid::NpadStyleJoyLeft::Mask |
        nn::hid::NpadStyleJoyRight::Mask |
        nn::hid::NpadStyleJoyDual::Mask);

    // サポートする Npad ID のリスト
    nn::hid::SetSupportedNpadIdType(NpadIds, NN_ARRAY_SIZE(NpadIds));

    // ハンドルの取得
    while (!nn::hidbus::GetBusHandle(&handle, nn::hid::NpadId::No1, nn::hidbus::BusType_RightJoyRail))
    {
        NN_LOG("[hidbus] Please Connect Joy-Con\n");

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    NN_LOG("[hidbus] handle = %llu\n", handle);

    // 初期化
    nn::Result initializeResult = nn::ResultSuccess();
    do
    {
        initializeResult = nn::hidbus::Initialize(handle);

        if (initializeResult.IsSuccess())
        {
            break;
        }
        else
        {
            NN_ABORT("Unknown Error(%d, %d)\n", initializeResult.GetModule(), initializeResult.GetDescription());
        }

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    } while (initializeResult.IsFailure());

    // Attachment の接続待ち
    nn::Result enableResult;
    while (enableResult.IsFailure())
    {
        enableResult = nn::hidbus::EnableExternalDevice(true, RondeId, handle);
        NN_LOG("[hidbus] Attachment Device Waiting..\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }


    NN_LOG("[hidbus]-------------- Device info -------------\n");

    // デバイスのバージョンの取得
    PrintDeviceVersion(handle);

    // デバイス ID の取得
    PrintDeviceSerial(handle);

    NN_LOG("[hidbus] Thermistor  = ");
    PrintInternalAdcValue(getThermistorValueCommand, sizeof(getThermistorValueCommand), handle);

    NN_LOG("[hidbus] 1.8 value   = ");
    PrintInternalAdcValue(getInternalAdc1_8ValueCommand, sizeof(getInternalAdc1_8ValueCommand), handle);

    NN_LOG("[hidbus] 2.5 value   = ");
    PrintInternalAdcValue(getInternalAdc2_5ValueCommand, sizeof(getInternalAdc2_5ValueCommand), handle);

    // Invalid Command の確認用
    CheckAndPrintInvalidCommand(handle);

    NN_LOG("[hidbus]----------------------------------------\n");
    NN_LOG("[hidbus]------------ Internal Flash ------------\n");

    // CAL 値へがっつり書き込み
#ifdef ENABLE_SYSTEM_DATA_WRITE
    for (uint8_t i = 0; i < 4; i++)
    {
        writeDataCommand[1] = i + 1;
        writeDataCommand[4] = 4 * i + 1 + 1;
        writeDataCommand[5] = 4 * i + 2 + 1;
        writeDataCommand[6] = 4 * i + 3 + 1;
        writeDataCommand[7] = 4 * i + 4 + 1;
        WriteDataToInternalFlash(writeDataCommand, sizeof(writeDataCommand), handle);
    }

    for (uint8_t i = 0; i < 4; i++)
    {
        writeDataCommand[1] = i + 0x11;
        writeDataCommand[4] = 4 * i + 1 + 1;
        writeDataCommand[5] = 4 * i + 2 + 1;
        writeDataCommand[6] = 4 * i + 3 + 1;
        writeDataCommand[7] = 4 * i + 4 + 1;
        WriteDataToInternalFlash(writeDataCommand, sizeof(writeDataCommand),handle);
    }

    WriteDataToInternalFlash(writeLog1Command, sizeof(writeLog1Command), handle);
    WriteDataToInternalFlash(writeLog2Command, sizeof(writeLog2Command), handle);
    WriteDataToInternalFlash(writeLog3Command, sizeof(writeLog3Command), handle);
    WriteDataToInternalFlash(writeLog4Command, sizeof(writeLog4Command), handle);
    WriteDataToInternalFlash(writeLog5Command, sizeof(writeLog5Command), handle);
    WriteDataToInternalFlash(writeLog6Command, sizeof(writeLog6Command), handle);
#endif

    for (uint8_t i = 0; i < 4; i++)
    {
        readDataCommand[1] = i + 1;
        ReadDataFromInternalFlash(readDataCommand, sizeof(readDataCommand), handle);
    }
    for (uint8_t i = 0; i < 4; i++)
    {
        readDataCommand[1] = i + 0x11;
        ReadDataFromInternalFlash(readDataCommand, sizeof(readDataCommand), handle);
    }

    ReadLogDataFromInternalFlash(readLog1Command, sizeof(readLog1Command), handle);
    ReadLogDataFromInternalFlash(readLog2Command, sizeof(readLog2Command), handle);
    ReadLogDataFromInternalFlash(readLog3Command, sizeof(readLog3Command), handle);
    ReadLogDataFromInternalFlash(readLog4Command, sizeof(readLog4Command), handle);
    ReadLogDataFromInternalFlash(readLog5Command, sizeof(readLog5Command), handle);
    ReadLogDataFromInternalFlash(readLog6Command, sizeof(readLog6Command), handle);
    NN_LOG("[hidbus]----------------------------------------\n");
    nn::TimeSpan tickSum = 0;

    while (count < 100)
        //while (1)
    {
        auto startTick = nn::os::GetSystemTick();
        //NN_LOG("count = %d\n", count);
        //PrintExternalStrainGaugeAdcValue(handle);

        //NN_LOG("[hidbus] Internal ADC value = ");
        //PrintInternalAdcValue(getInternalAdcValueCommand, sizeof(getThermistorValueCommand), handle);

        PrintPollingFormat(handle);

        tickSum += (nn::os::GetSystemTick() - startTick).ToTimeSpan();

        //NN_LOG("[hidbus]----------------------------------------\n");
        count++;
        if (count % 100 == 0)
        {
            NN_LOG("count = %d\n", count);
            NN_LOG("Loop Time = %d ms\n", tickSum.GetMilliSeconds() / 100);
            tickSum = 0;
        }
    }

    // 拡張デバイスを無効化 (電源供給断)
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::hidbus::EnableExternalDevice(false, RondeId, handle));

    nn::hidbus::Finalize(handle);
} // NOLINT(impl/function_size)
