﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/util/util_Vector.h>

#include "SixAxisSensorGestureProcessor.h"
#include "ShakeMotionDetector.h"
#include "RotateMotionDetector.h"
#include "PeakMotionDetector.h"
#include "ThrowMotionDetector.h"
#include "SpinMotionDetector.h"
#include "PushMotionDetector.h"
// #include "TrackMotionDetector.h"

namespace {

static PeakMotionDetector s_PeakMotionDetector;
static ShakeMotionDetector s_ShakeMotionDetector;
static RotateMotionDetector s_RotateMotionDetector;
static ThrowMotionDetector s_ThrowMotionDetector;
static SpinMotionDetector s_SpinMotionDetector;
static PushMotionDetector s_PushMotionDetector;
// static TrackMotionDetector s_TrackMotionDetector;

enum DetectorIdx
{
    DetectorIdx_Peak,
    DetectorIdx_Shake,
    DetectorIdx_Rotate,
    DetectorIdx_Throw,
    DetectorIdx_Spin,
    DetectorIdx_Push,

    DetectorIdx_CountMax,
};

// https://garchiving.com/angle-from-acceleration/
void GetRotationFromAccelerometer(
    float* pOutRoll,
    float* pOutPitch,
    const ::nn::util::Vector3f& acc) NN_NOEXCEPT
{
    if (acc.Length() == 0.f)
    {
        return;
    }

    float x, y, z;
    acc.Get(&x, &y, &z);
    *pOutPitch = FromRadianToDegreeCoef * ::nn::util::Atan2Est(x, ::std::sqrt(y * y + z * z));
    *pOutRoll = FromRadianToDegreeCoef * ::nn::util::Atan2Est(y, z);
    // NN_LOG("%f, %f\n", *pOutPitch, *pOutRoll);
}

} // namespace

SixAxisSensorGestureProcessor::SixAxisSensorGestureProcessor() NN_NOEXCEPT
    : m_GestureStateLifo()
    , m_GestureState()
    , m_GestureSamplingNumber(0)
    , m_ThresholdDeltaSamplingNumber(50) // 検出無効区間[sample]
    , m_GravityFilterConstant(0.8f)
    , m_RawState()
    , m_SamplingNumber(0)
{
    // 何もしない
}

void SixAxisSensorGestureProcessor::Initialize() NN_NOEXCEPT
{
    m_pMotionDetectors[DetectorIdx_Peak]   = &s_PeakMotionDetector;
    m_pMotionDetectors[DetectorIdx_Shake]  = &s_ShakeMotionDetector;
    m_pMotionDetectors[DetectorIdx_Rotate] = &s_RotateMotionDetector;
    m_pMotionDetectors[DetectorIdx_Throw]  = &s_ThrowMotionDetector;
    m_pMotionDetectors[DetectorIdx_Spin]   = &s_SpinMotionDetector;
    m_pMotionDetectors[DetectorIdx_Push]   = &s_PushMotionDetector;

    // Track は精度が出ないので未実装
    // m_pMotionDetectors[m_MotionDetectorCount++] = &s_TrackMotionDetector;
}

void SixAxisSensorGestureProcessor::Update(const nn::hid::SixAxisSensorState& state) NN_NOEXCEPT
{
    // 生値の加工計算を実行
    Calculate(state);

    // 登録済モーション検出器を更新
    for (int i = 0; i < DetectorIdx_CountMax; i++)
    {
        m_pMotionDetectors[i]->Update(m_RawState);
    }

    UpdateGestureLifo();
}

int SixAxisSensorGestureProcessor::GetGestures(GestureState* pOutStates, int count) NN_NOEXCEPT
{
    return m_GestureStateLifo.Read(pOutStates, count);
}

void SixAxisSensorGestureProcessor::Calculate(const nn::hid::SixAxisSensorState& state) NN_NOEXCEPT
{
    // RawState
    {
        // ひとつ前のデータを保持
        m_RawState.accDelta = m_RawState.acc;
        m_RawState.gyroDelta = m_RawState.gyro;

        // raw data を更新
        ::nn::util::VectorLoad(&m_RawState.acc, state.acceleration);
        ::nn::util::VectorLoad(&m_RawState.gyro, state.angularVelocity);

        // delta raw data を更新
        ::nn::util::VectorSubtract(&m_RawState.accDelta, m_RawState.acc, m_RawState.accDelta);
        ::nn::util::VectorSubtract(&m_RawState.gyroDelta, m_RawState.gyro, m_RawState.gyroDelta);

        // gravity を計算
        // https://developer.android.com/guide/topics/sensors/sensors_motion.html
        ::nn::util::Vector3f workVector;
        ::nn::util::VectorMultiply(
            &workVector,
            m_RawState.acc,
            (1 - m_GravityFilterConstant));
        ::nn::util::VectorMultiply(
            &m_RawState.gravity,
            m_RawState.gravity,
            m_GravityFilterConstant);
        ::nn::util::VectorAdd(
            &m_RawState.gravity,
            m_RawState.gravity,
            workVector);

        // 正規化
        m_RawState.accLengthSquared = ::nn::util::VectorNormalize(&m_RawState.accNormalized,
                                                                  m_RawState.acc);
        m_RawState.gyroLengthSquared = ::nn::util::VectorNormalize(&m_RawState.gyroNormalized,
                                                                   m_RawState.gyro);

        // サンプリング番号を設定
        m_RawState.samplingNumber = m_SamplingNumber;

        // test
        {
            float roll, pitch;
            GetRotationFromAccelerometer(&roll, &pitch, m_RawState.acc);
        }
    }

    // サンプリング番号を更新
    m_SamplingNumber++;

} // NOLINT(impl/function_size)

void SixAxisSensorGestureProcessor::UpdateGestureLifo() NN_NOEXCEPT
{
    MotionState motionStates[DetectorIdx_CountMax];
    bool isDetectedStates[DetectorIdx_CountMax];

    for (int i = 0; i < DetectorIdx_CountMax; i++)
    {
        int count = m_pMotionDetectors[i]->GetMotions(&motionStates[i], 1);
        if (count != 1)
        {
            motionStates[i] = MotionState();
        }

        isDetectedStates[i] = (
            motionStates[i].motionSamplingNumber - m_GestureState.gestureSamplingNumber > m_ThresholdDeltaSamplingNumber
        );
    }

    if (isDetectedStates[DetectorIdx_Shake])
    {
        Append(motionStates[DetectorIdx_Shake]);
        return;
    }
    else if(isDetectedStates[DetectorIdx_Spin])
    {
        Append(motionStates[DetectorIdx_Spin]);
        return;
    }
    else if (isDetectedStates[DetectorIdx_Throw])
    {
        Append(motionStates[DetectorIdx_Throw]);
        return;
    }
    else if (isDetectedStates[DetectorIdx_Rotate])
    {
        Append(motionStates[DetectorIdx_Rotate]);
        return;
    }
    else if (isDetectedStates[DetectorIdx_Push])
    {
        Append(motionStates[DetectorIdx_Push]);
        return;
    }
    // else if (isDetectedStates[DetectorIdx_Peak])
    // {
    //     Append(motionStates[DetectorIdx_Peak]);
    //     return;
    // }
}

void SixAxisSensorGestureProcessor::Append(const MotionState& state) NN_NOEXCEPT
{
    m_GestureState = GestureState({
        m_GestureSamplingNumber,
        state.motionSamplingNumber,
        static_cast<GestureType>(state.motionType),
    });

    m_GestureStateLifo.Append(m_GestureState);
    m_GestureSamplingNumber++;
}
