﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/util/util_Vector.h>

#include "PeakMotionDetector.h"
#include "SixAxisSensorGestureProcessor.h"


PeakMotionDetector::PeakMotionDetector() NN_NOEXCEPT
    : m_RawState()
    , m_State(State_Stable)
    , m_ThresholdAccLengthSquared(1.0f) // 検出閾値
    , m_IgnoreDeltaSampleCount(20) // 検出後無視するサンプル数
{
    // 何もしない
}

PeakMotionDetector::~PeakMotionDetector() NN_NOEXCEPT
{

}

void PeakMotionDetector::Reset() NN_NOEXCEPT
{
    m_Lifo.Clear();
}

void PeakMotionDetector::Update(const SixAxisSensorRawState& rawState) NN_NOEXCEPT
{
    ::nn::util::Vector3f workVector;
    ::nn::util::VectorSubtract(&workVector, rawState.acc, rawState.gravity);
    auto accLengthSquared = ::nn::util::VectorDot(workVector, workVector);
    if (accLengthSquared > m_ThresholdAccLengthSquared && m_State == State::State_Stable &&
        m_SixAxisSensorRawState.samplingNumber + m_IgnoreDeltaSampleCount < rawState.samplingNumber) // チャタリング対策
    {
        m_State = State::State_Detecting;
        m_SixAxisSensorRawState = rawState;
        // NN_LOG("accLengthSquared : %f\n", accLengthSquared);
        // NN_LOG("Stable -> Detecting\n");
        m_Lifo.Append(
            MotionState({
                m_SamplingNumber,
                rawState.samplingNumber,
                MotionType::MotionType_Peak
            })
        );

        m_SamplingNumber++;
    }
    else if (accLengthSquared < m_ThresholdAccLengthSquared && m_State == State::State_Detecting)
    {
        m_State = State::State_Stable;
        // NN_LOG("Detecting -> Stable\n");
    }

    m_RawState = rawState;
}

int PeakMotionDetector::GetMotions(MotionState* pOutStates, int count) NN_NOEXCEPT
{
    return m_Lifo.Read(pOutStates, count);
}
