﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>

#include <nn/nn_Common.h>
#include <nn/util/util_Vector.h>
#include <nn/nn_StaticAssert.h>

#include "SixAxisSensorGestureProcessorTypes.h"
#include "../../../../../Programs/Iris/Sources/Libraries/hid/detail/hid_RingLifo.h"

// モーションを表す列挙体です。
enum MotionType
{
    // PeakMotionDetector
    MotionType_Peak,

    // ShakeMotionDetector
    MotionType_Shake,

    // RotateMotionDetector
    MotionType_RotatePlusX,
    MotionType_RotatePlusY,
    MotionType_RotatePlusZ,
    MotionType_RotateMinusX,
    MotionType_RotateMinusY,
    MotionType_RotateMinusZ,

    // ThrowMotionDetector
    MotionType_Throw,

    // PushMotionDetector
    MotionType_PushPlusX,
    MotionType_PushPlusY,
    MotionType_PushPlusZ,
    MotionType_PushMinusX,
    MotionType_PushMinusY,
    MotionType_PushMinusZ,

    // SpinMotionDetector
    MotionType_Spin,
};

const int MotionStateCountMax = 16; //!< 各モーション検出器が保持できるモーションの最大数です。

// モーションの入力状態です。
struct MotionState
{
    int64_t samplingNumber;
    int64_t motionSamplingNumber; //!< モーションが起きたサンプリング番号
    MotionType motionType;
};

// モーションの入力状態の LIFO です。
class MotionStateLifo final :
    public ::nn::hid::detail::RingLifo<MotionState, MotionStateCountMax>
{
public:
    MotionStateLifo() NN_NOEXCEPT { /* 何もしない */ }
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_CLANG
NN_STATIC_ASSERT(::std::is_standard_layout<MotionStateLifo>::value);
#endif

// モーション検出器のインタフェースです。
class IMotionDetector
{
public:
    IMotionDetector() NN_NOEXCEPT
        : m_Lifo()
        , m_SamplingNumber(0)
    {
        // 何もしない
    }

    virtual ~IMotionDetector() NN_NOEXCEPT { /* 何もしない */ }

    virtual void Reset() NN_NOEXCEPT = 0;

    virtual void Update(const SixAxisSensorRawState& rawState) NN_NOEXCEPT = 0;

    virtual int GetMotions(MotionState* pOutStates, int count) NN_NOEXCEPT = 0;

protected:

    MotionStateLifo m_Lifo;
    int64_t m_SamplingNumber;
};
