﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Applet.h"

/* アプレットは全てメインスレッドとは別のスレッドで動作させてください */
namespace frm {

    const size_t ThreadStackSize = 1024 * 8;
    NN_OS_ALIGNAS_THREAD_STACK char thredStack[ThreadStackSize];   // スレッドのスタック
    nn::os::ThreadType thred;
    void* dummy;

    /* Singleton パターン */
    Applet& Applet::GetInstance() NN_NOEXCEPT
    {
        static Applet Instance;
        return Instance;
    }

    /* ------------------------------------------------------------ */
    // PRIVATE関数
    /* ------------------------------------------------------------ */
    void Applet::UpdateTwinkleCounter()
    {
        if (m_TwinkleSwitchFlag == false)
        {
            m_TwinkleCounter = m_TwinkleCounter + 5;
            if (m_TwinkleCounter >= 250)
            {
                m_TwinkleSwitchFlag = true;
            }
        }
        else
        {
            m_TwinkleCounter = m_TwinkleCounter - 5;
            if (m_TwinkleCounter <= 50)
            {
                m_TwinkleSwitchFlag = false;
            }
        }
    }

    void Applet::UpdateSwipeCounter(int functionMenu)
    {
        m_SwipeCounterState = functionMenu == 2 ? 1 : 2;

        if (m_SwipeCounterState == 1)
        {
            if (m_SwipeCounter < 10)
            {
                m_SwipeCounter++;
            }
            // 0→Widthに向かう感じ
            m_DeltaX = m_DrawPotition.Width - m_DrawPotition.Width / (static_cast<float>(m_SwipeCounter) * 30.f);
        }
        else if (m_SwipeCounterState == 2)
        {
            if (m_SwipeCounter > 0)
            {
                m_SwipeCounter--;
            }
            else
            {
                m_SwipeCounterState = 0;
                m_DeltaX = 0.f;
            }
            // Width→0に向かう感じ
            m_DeltaX = pow(static_cast<double>(m_SwipeCounter), 6.0) * (m_DrawPotition.Width / pow(10.0, 6.0));
        }
    }

    /* ------------------------------------------------------------ */
    // PUBLIC関数
    /* ------------------------------------------------------------ */
    void Applet::Initialize()
    {
        // アカウントライブラリの初期化
        nn::account::Initialize();
    }

    // コントローラサポートアプレット(接続維持)
    void Applet::CountrollerSupportThred()
    {
        if (f_enbl == false)
        {
            nn::os::CreateThread(&thred, CountrollerSupport, dummy, thredStack, ThreadStackSize, 1);
            nn::os::StartThread(&thred);
            f_enbl = true;
        }
    }

    // コントローラサポートアプレット(切断)
    void Applet::CountrollerSupportDisconnectThred()
    {
        if (f_enbl == false)
        {
            nn::os::CreateThread(&thred, CountrollerSupportDisconnect, dummy, thredStack, ThreadStackSize, 1);
            nn::os::StartThread(&thred);
            f_enbl = true;
        }
    }

    // アカウント選択アプレット
    void Applet::AccountManagerThred()
    {
        if (f_enbl == false)
        {
            nn::os::CreateThread(&thred, AccountManager, dummy, thredStack, ThreadStackSize, 1);
            nn::os::StartThread(&thred);
            f_enbl = true;
        }
    }

    // ソフトウェアキーボードアプレット
    void Applet::SwKeyBoardThred()
    {
        if (f_enbl == false)
        {
            nn::os::CreateThread(&thred, SwKeyBoard, dummy, thredStack, ThreadStackSize, 1);
            nn::os::StartThread(&thred);
            f_enbl = true;
        }
    }

    // エラービュアーアプレット
    void Applet::ErrorViewerThred()
    {
        if (f_enbl == false)
        {
            nn::os::CreateThread(&thred, ErrorViewer, dummy, thredStack, ThreadStackSize, 1);
            nn::os::StartThread(&thred);
            f_enbl = true;
        }
    }

    // NULLアクセス
    void Applet::NullAccessThred()
    {
        if (f_enbl == false)
        {
            nn::os::CreateThread(&thred, NullAccess, dummy, thredStack, ThreadStackSize, 1);
            nn::os::StartThread(&thred);
            f_enbl = true;
        }
    }

    // スレッドの削除
    void Applet::ThredDestroy()
    {
        if (thred._state == thred.State_Exited)
        {
            nn::os::DestroyThread(&thred);
            f_enbl = false;
        }
    }

    void Applet::Control(frm::NpadState* npadState)
    {
        m_up = false;
        m_down = false;
        m_fix = false;

        // 接続中のコントローラのボタンの押下げ状況を確認します
        for (auto index = 0; index < frm::NpadIdNum; index++)
        {
            if (npadState[index].trigger.Test<nn::hid::NpadButton::StickLDown>()
                || npadState[index].trigger.Test<nn::hid::NpadButton::StickRDown>()
                ) {
                m_down = true;
            }
            if (npadState[index].trigger.Test<nn::hid::NpadButton::StickLUp>()
                || npadState[index].trigger.Test<nn::hid::NpadButton::StickRUp>()
                ) {
                m_up = true;
            }
            if (npadState[index].trigger.Test<nn::hid::NpadButton::StickL>()
                || npadState[index].trigger.Test<nn::hid::NpadButton::StickR>()
                || npadState[index].trigger.Test<nn::hid::NpadButton::L>()
                || npadState[index].trigger.Test<nn::hid::NpadButton::R>()
                ) {
                m_fix = true;
            }
        }

        if (m_down)
        {
            m_menu = m_menu == 5 ? 0 : m_menu + 1;
        }
        if (m_up)
        {
            m_menu = m_menu == 0 ? 5 : m_menu - 1;
        }

        if (m_fix)
        {
            switch (m_menu)
            {
                case 0:         // コンサポ
                    CountrollerSupportThred();
                    break;
                case 1:         // コンサポ
                    CountrollerSupportDisconnectThred();
                    break;
                case 2:         // U-Sel
                    AccountManagerThred();
                    break;
                case 3:         // Swキーボード
                    SwKeyBoardThred();
                    break;
                case 4:         // エラービュアー
                    ErrorViewerThred();
                    break;
                case 5:         // NULLアクセス
                    NullAccessThred();
                    break;
                default:
                    break;
            }
        }
        ThredDestroy();

    } // NOLINT(impl/function_size)

    void Applet::DrawControl(GraphicsSystem* pGraphicsSystem,
                             nn::gfx::util::DebugFontTextWriter* pTextWriter,
                             int functionMenu)
    {
        UpdateTwinkleCounter();
        UpdateSwipeCounter(functionMenu);

        m_DrawPotition.X = 1280.f - m_DeltaX - 5.f;

        nn::util::Color4u8 color = { 255, 10, 0, 255 };

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        // 小枠
        pPrimitiveRenderer->SetColor(Color::Gray);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), m_DrawPotition.X, m_DrawPotition.Y, m_DrawPotition.Width, m_DrawPotition.Height);
        pPrimitiveRenderer->SetLineWidth(1.f);
        pPrimitiveRenderer->SetColor(Color::Black);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), m_DrawPotition.X, m_DrawPotition.Y, m_DrawPotition.Width, m_DrawPotition.Height);
        pPrimitiveRenderer->SetColor(color);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), m_DrawPotition.X + 3, m_DrawPotition.Y + 3, m_DrawPotition.Width - 6, m_DrawPotition.Height - 6);

        pTextWriter->SetScale(1.0, 1.0);

        int DeltaX = 10;
        int DeltaY = 10;

        // TOP項目の作成
        pTextWriter->SetTextColor(Color::Black);
        pTextWriter->SetCursor(m_DrawPotition.X + DeltaX, m_DrawPotition.Y + DeltaY - 5);
        pTextWriter->Print("[Applet Caller]");

        DeltaY += 30;

        // 各Menu項目の作成

        if (m_menu == 0)
        {
            pTextWriter->SetCursor(m_DrawPotition.X + 30 + 2, m_DrawPotition.Y + DeltaY + 2);
            pTextWriter->SetTextColor(ExClearColor::Black);
            pTextWriter->Print("Controller Support");
            pTextWriter->SetTextColor(color);
        }
        else
        {
            pTextWriter->SetTextColor(Color::Black);
        }
        pTextWriter->SetCursor(m_DrawPotition.X + 30, m_DrawPotition.Y + DeltaY);
        pTextWriter->Print("Controller Support");

        DeltaY += 25;

        if (m_menu == 1)
        {
            pTextWriter->SetCursor(m_DrawPotition.X + 30 + 2, m_DrawPotition.Y + DeltaY + 2);
            pTextWriter->SetTextColor(ExClearColor::Black);
            pTextWriter->Print("Controller Support(Disconnect)");
            pTextWriter->SetTextColor(color);
        }
        else
        {
            pTextWriter->SetTextColor(Color::Black);
        }
        pTextWriter->SetCursor(m_DrawPotition.X + 30, m_DrawPotition.Y + DeltaY);
        pTextWriter->Print("Controller Support(Disconnect)");

        DeltaY += 25;

        if (m_menu == 2)
        {
            pTextWriter->SetCursor(m_DrawPotition.X + 30 + 2, m_DrawPotition.Y + DeltaY + 2);
            pTextWriter->SetTextColor(ExClearColor::Black);
            pTextWriter->Print("Account Select");
            pTextWriter->SetTextColor(color);
        }
        else
        {
            pTextWriter->SetTextColor(Color::Black);
        }
        pTextWriter->SetCursor(m_DrawPotition.X + 30, m_DrawPotition.Y + DeltaY);
        pTextWriter->Print("Account Select");

        DeltaY += 25;

        if (m_menu == 3)
        {
            pTextWriter->SetCursor(m_DrawPotition.X + 30 + 2, m_DrawPotition.Y + DeltaY + 2);
            pTextWriter->SetTextColor(ExClearColor::Black);
            pTextWriter->Print("Software Keyboard");
            pTextWriter->SetTextColor(color);
        }
        else
        {
            pTextWriter->SetTextColor(Color::Black);
        }
        pTextWriter->SetCursor(m_DrawPotition.X + 30, m_DrawPotition.Y + DeltaY);
        pTextWriter->Print("Software Keyboard");

        DeltaY += 25;

        if (m_menu == 4)
        {
            pTextWriter->SetCursor(m_DrawPotition.X + 30 + 2, m_DrawPotition.Y + DeltaY + 2);
            pTextWriter->SetTextColor(ExClearColor::Black);
            pTextWriter->Print("Error Viewer");
            pTextWriter->SetTextColor(color);
        }
        else
        {
            pTextWriter->SetTextColor(Color::Black);
        }
        pTextWriter->SetCursor(m_DrawPotition.X + 30, m_DrawPotition.Y + DeltaY);
        pTextWriter->Print("Error Viewer");

        DeltaY += 25;

        if (m_menu == 5)
        {
            pTextWriter->SetCursor(m_DrawPotition.X + 30 + 2, m_DrawPotition.Y + DeltaY + 2);
            pTextWriter->SetTextColor(ExClearColor::Black);
            pTextWriter->Print("NULL Access");
            pTextWriter->SetTextColor(color);
        }
        else
        {
            pTextWriter->SetTextColor(Color::Black);
        }
        pTextWriter->SetCursor(m_DrawPotition.X + 30, m_DrawPotition.Y + DeltaY);
        pTextWriter->Print("NULL Access");

    } // NOLINT(impl/function_size)

    /* ------------------------------------------------------------ */
    // クラス外関数
    /* ------------------------------------------------------------ */
     void CountrollerSupport(void* dummy)
    {
        if(!dummy)
        {
            nn::hid::ControllerSupportResultInfo result;
            nn::hid::ControllerSupportArg ctrlArg;
            ctrlArg.SetDefault();
            ctrlArg.enableIdentificationColor = true;
            nn::hid::SetNpadJoyHoldType(nn::hid::NpadJoyHoldType_Horizontal);

            nn::hid::ShowControllerSupport(&result, ctrlArg);
        }
    }

     void CountrollerSupportDisconnect(void* dummy)
     {
         if (!dummy)
         {
             nn::hid::ControllerSupportResultInfo result;
             nn::hid::ControllerSupportArg ctrlArg;
             ctrlArg.SetDefault();
             ctrlArg.enableTakeOverConnection = false;  // アプレット立ち上げ時に切断
             ctrlArg.enableIdentificationColor = true;
             nn::hid::SetNpadJoyHoldType(nn::hid::NpadJoyHoldType_Horizontal);

             nn::hid::ShowControllerSupport(&result, ctrlArg);
         }
     }

    void AccountManager(void* dummy)
    {
        if (!dummy)
        {
            nn::account::Uid uid;
            nn::account::ShowUserSelector(&uid);
        }
    }

    void SwKeyBoard(void* dummy)
    {
        if (!dummy)
        {
            const size_t ApplicationHeapSize = 128 * 1024;

            ApplicationHeap applicationHeap;
            applicationHeap.Initialize(ApplicationHeapSize);

            nn::swkbd::ShowKeyboardArg showKeyboardArg;
            nn::swkbd::MakePreset(&(showKeyboardArg.keyboardConfig), nn::swkbd::Preset_Default);

            // ガイド文字列の設定
            const char* guide_string = u8"検証お疲れ様です。いつもありがとうございます。";
            nn::swkbd::SetGuideTextUtf8(&showKeyboardArg.keyboardConfig,
                                        guide_string);

            // 共有メモリ用バッファの割り当て
            size_t in_heap_size = nn::swkbd::GetRequiredWorkBufferSize(false);
            void* swkbd_work_buffer = applicationHeap.Allocate(in_heap_size, nn::os::MemoryPageSize);

            showKeyboardArg.workBuf = swkbd_work_buffer;
            showKeyboardArg.workBufSize = in_heap_size;

            // 終了パラメータの設定
            size_t out_heap_size = nn::swkbd::GetRequiredStringBufferSize();
            nn::swkbd::String output_string;
            output_string.ptr = reinterpret_cast<char16_t*>(applicationHeap.Allocate(out_heap_size, nn::os::MemoryPageSize));
            output_string.bufSize = out_heap_size;

            nn::swkbd::ShowKeyboard(&output_string, showKeyboardArg);

            applicationHeap.Deallocate(output_string.ptr);
            applicationHeap.Deallocate(swkbd_work_buffer);

            applicationHeap.Finalize();
        }
    }

    void ErrorViewer(void* dummy)
    {
        if (!dummy)
        {
            nn::err::ErrorCode errorCode;
            errorCode.category = 2162;
            errorCode.number = 0002;
            nn::err::ShowError(errorCode);
        }
    }

    void NullAccess(void* dummy)
    {
        if (!dummy)
        {
            NN_ABORT_UNLESS_NOT_NULL(nullptr);
        }
    }
}
