﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
#include <cstdlib>
#include <string>
#include <list>
#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/os.h>

#include <nn/bluetooth.h>
#include <nn/bluetooth/bluetooth_ProprietaryLeApi.h>
#include <nn/bluetooth/bluetooth_BleScanParameterIdPalma.h>




namespace util{

    const nn::bluetooth::GattAttributeUuid ADV_SERVICE_UUID =
    {
        .length = nn::bluetooth::GattAttributeUuidLength_128,
        .uu.uuid128 = { 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF, 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99 }
    };

    static std::vector<nn::bluetooth::GattAttributeUuid> registeredUuid;

    struct Characteristics
    {
        nn::bluetooth::GattCharacteristic           data[nn::bluetooth::GattAttributeCountMaxClient];
        int                                         dataNum;
    };

    struct Service
    {
        nn::bluetooth::GattService                  data[nn::bluetooth::GattAttributeCountMaxClient];
        uint16_t                                    dataNum;
        util::Characteristics                       Char[nn::bluetooth::GattAttributeCountMaxClient];

        uint8_t                                     fwVer[32];
        size_t                                      fwVerLength;
    };

    struct ConnectionInfo
    {
        nn::bluetooth::BleConnectionInfo            data[nn::bluetooth::BleConnectionCountMaxClient];    //!< Bleの接続情報
        int                                         dataNum;

        util::Service                               Serv[nn::bluetooth::GattAttributeCountMaxClient];
    };

    //!< Palmaの状態をひとまとめにした構造体定義
    struct BleState
    {
        nn::bluetooth::BleAdvertisePacketParameter  palmaAdvertisePacketParameter;                                 //!< BLE アドバタイズパケット
        util::ConnectionInfo                        connectionInfo;                                                //!< Bleの接続情報
    };

    //!< Palmaの状態をひとまとめにした構造体定義
    struct BleEvent
    {
        nn::os::SystemEventType     scan;
        nn::os::SystemEventType     connection;        //!< 接続状態が変更される際にシグナルされるイベント
        nn::os::SystemEventType     serviceDiscovery;  //!< GATT Service Discovery でGATT Attribute が発見された際にシグナルされるイベント
        nn::os::SystemEventType     gattOperation;     //!< GATT Attribute に対する操作が完了した際にシグナルされるイベント
    };

    const size_t ThreadStackSize = 8192;                                //!< イベント待ちスレッドのスタックサイズ

    struct BleMultiWaitHolder
    {
        nn::os::MultiWaitType           header;                             //!< 多重待ちイベントのヘッダを示す変数
        nn::os::MultiWaitHolderType     scan;
        nn::os::MultiWaitHolderType     connection;                         //!< 接続状態が変更される際にシグナルされるイベント
        nn::os::MultiWaitHolderType     serviceDiscovery;                   //!< GATT Service Discovery でGATT Attribute が発見された際にシグナルされるイベント
        nn::os::MultiWaitHolderType     gattOperation;                      //!< GATT Attribute に対する操作が完了した際にシグナルされるイベント
        nn::os::ThreadType              thread;                             //!< イベントの多重待ちを行うスレッド
        NN_OS_ALIGNAS_THREAD_STACK char threadStack[ThreadStackSize];       //!< イベント待ちスレッドのスタック
    };

    class Ble
    {
    private:

    public:
        bool        m_IsScanningSmartDevice;
        BleState    m_bleState;

        //----------------------------------------------------------------
        // @brief   インスタンス取得
        //----------------------------------------------------------------
        static Ble& GetInstance(void) NN_NOEXCEPT;

        //----------------------------------------------------------------
        // @brief   Bleのイニシャライズを開始します
        //----------------------------------------------------------------
        void Initialize(void);

        void RegisterGattOperation(nn::bluetooth::GattService* service);

        void StartBleScanSmartDevice(void);

        void StopBleScanSmartDevice(void);

        //----------------------------------------------------------------
        // @brief   Bleのファイナライズを実施します
        //----------------------------------------------------------------
        void finalize(void);

        //----------------------------------------------------------------
        // @brief   独自ペアリングしたデバイスのペアリングを解除します(Debug用)
        //----------------------------------------------------------------
        void ResetPairing(void);
    };
}
#ifndef gBle
#define gBle (util::Ble::GetInstance())
#endif
