﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cmath>
#include <cstdarg>
#include <cstdio>
#include <cstring>
#include <mutex>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include "Simple2D.h"
#include "detail/GraphicsHelper.h"
#include "detail/Simple2D_Font.h"
#include "detail/Simple2D_Util.h"

// API 呼び出しを排他する
//#define S2D_ENABLE_LOCK

#if defined(S2D_ENABLE_LOCK)
#define SCOPED_LOCK     std::lock_guard<decltype(g_Mutex)> lock(g_Mutex)
#else
#define SCOPED_LOCK
#endif

// 拡張子の一致判定
// extension は文字列リテラルである必要がある
#define S2D_STRING_LITERAL_EQUAL(target, extension) \
    (nn::util::Strnlen(target, sizeof(extension)) == sizeof(extension) - 1 && \
     nn::util::Strnicmp(target, extension, sizeof(extension)) == 0)

namespace s2d
{

namespace
{

// 単純図形描画用頂点シェーダ
const char* VertexShaderSimple = R"(
#version 320 es
layout(location = 0) highp in vec3 a_Position;
layout(location = 1) lowp  in vec4 a_Color;
lowp  out vec4 v_Color;

void main() {
    v_Color     = a_Color;
    gl_Position = vec4(a_Position, 1.0);
}
)";

// テクスチャ描画用頂点シェーダ
const char* VertexShaderTexture = R"(
#version 320 es
layout(location = 0) highp in vec3 a_Position;
layout(location = 2) highp in vec2 a_TexCoord;
layout(location = 1) lowp  in vec4 a_Color;
highp out vec2 v_TexCoord;
lowp  out vec4 v_Color;

void main() {
    gl_Position = vec4(a_Position, 1.0);
    v_TexCoord  = a_TexCoord;
    v_Color     = a_Color;
}
)";

// 単純図形描画用フラグメントシェーダ
const char* FragmentShaderSimple = R"(
#version 320 es
layout(location = 0) lowp out vec4 o_FragColor;
lowp in  vec4 v_Color;

void main() {
    o_FragColor = v_Color;
}
)";

// テクスチャ描画用フラグメントシェーダ
const char* FragmentShaderTexture = R"(
#version 320 es
layout(location = 0) lowp  out vec4 o_FragColor;
highp in  vec2 v_TexCoord;
lowp  in  vec4 v_Color;
uniform sampler2D s_Texture;

void main() {
    lowp vec4 tempColor = texture(s_Texture, v_TexCoord);
    o_FragColor = tempColor * v_Color;
}
)";

/**
 * @brief   OpenGL 用の 0.0 ～ 1.0 に正規化された色
 */
union NormalizedColor
{
    struct
    {
        GLfloat r;
        GLfloat g;
        GLfloat b;
        GLfloat a;
    };
    GLfloat v[4];

    /**
     * @brief   8bit カラーを正規化して格納
     */
    void SetU8Color(const Color& color) NN_NOEXCEPT
    {
        r = ConvertFromU8(color.r);
        g = ConvertFromU8(color.g);
        b = ConvertFromU8(color.b);
        a = ConvertFromU8(color.a);
    }

    /**
     * @brief   8bit カラーから正規化カラーを生成
     */
    static NormalizedColor FromU8Color(const Color& color) NN_NOEXCEPT
    {
        NormalizedColor newColor;
        newColor.SetU8Color(color);

        return newColor;
    }

private:
    static GLfloat ConvertFromU8(uint8_t u8color) NN_NOEXCEPT
    {
        return u8color / 255.0f;
    }
};

struct ShaderPack
{
    GLuint vertexId;
    GLuint fragmentId;
    GLuint programId;
};

struct ShaderLocations
{
    GLuint position;
    GLuint vertexColor;
    GLuint textureCoord;
} g_ShaderLocations;

// フォントの描画に関する状態
struct FontContext
{
    Color color;
    float offsetX;
    float scaleX;
    float scaleY;

    // メンバを初期化
    void Initialize() NN_NOEXCEPT
    {
        color   = Colors::White;
        offsetX = 0.0f;
        scaleX  = 1.0f;
        scaleY  = 1.0f;
    }
} g_FontContext;

// 描画に使用するシェーダのタイプ
enum class ShaderType
{
    None,       //!< シェーダ不使用 (割り当て解除)
    Simple,     //!< 頂点カラーを使用した単純な描画
    Texture     //!< テクスチャ描画
};

// 頂点バッファの要素数のデフォルト値
const int DefaultVertexCountMax   = 64 * 1024;

// 多角形描画用の一時領域の頂点数
const int TemporaryVertexCountMax = 1024;

// 文字列描画用の一時領域サイズ
const int TemporaryStringBufferSize = 1024;

bool g_IsInitialized = false;
nn::os::Mutex g_Mutex(true);

ShaderType g_CurrentShader = ShaderType::Simple;

OriginPosition g_OriginPosition = OriginPosition::TopLeft;
Size g_CanvasSize = DefaultCanvasSize;

GLuint     g_ImageFrameBuffer;
ImageData* g_pRenderTargetImage = nullptr;

detail::GraphicsHelper g_GraphicsHelper;

ShaderPack g_SimpleShader;
ShaderPack g_TextureShader;

GLuint g_GlBuffers[3];
GLuint g_VertexBufferId;
GLuint g_VertexColorBufferId;
GLuint g_TexCoordBufferId;

int             g_VertexCountMax        = DefaultVertexCountMax;
int             g_NextVertexIndex       = 0;
char*           g_pVertexPointBuffer    = nullptr;
char*           g_pVertexColorBuffer    = nullptr;
char*           g_pVertexTexCoordBuffer = nullptr;
Point3D         g_TempPointBuffer[TemporaryVertexCountMax];
NormalizedColor g_TempColorBuffer[TemporaryVertexCountMax];
Color           g_TempColorBufferU8[TemporaryVertexCountMax];
Point2D         g_TempTexCoordBuffer[TemporaryVertexCountMax];

#if defined(S2D_ENABLE_DEBUG_LOG)
void GL_APIENTRY DebugCallbackFunc(GLenum source, GLenum type, GLuint id, GLenum severity, GLsizei length, const GLchar* message, const void* userParam)
{
    NN_UNUSED(length);
    NN_UNUSED(userParam);

    NN_LOG("GL Debug Callback:\n");
    NN_LOG("  source:       0x%08x\n", source);
    NN_LOG("  type:         0x%08x\n", type);
    NN_LOG("  id:           0x%08x\n", id);
    NN_LOG("  severity:     0x%08x\n", severity);
    NN_LOG("  message:      %s\n", message);

    NN_ASSERT(type != GL_DEBUG_TYPE_ERROR, "GL Error occurs.");
}
#endif  // if defined(S2D_ENABLE_DEBUG_LOG)

/**
 * @brief   グラフィックスの初期化
 */
void InitializeGraphics()
{
    g_GraphicsHelper.Initialize();

#if defined(S2D_ENABLE_DEBUG_LOG)
    NN_LOG("GL_VERSION: %s\n", glGetString(GL_VERSION));
    NN_LOG("GL_SHADING_LANGUAGE_VERSION: %s\n", glGetString(GL_SHADING_LANGUAGE_VERSION));

    glDebugMessageCallback(DebugCallbackFunc, nullptr);
#endif  // if defined(S2D_ENABLE_DEBUG_LOG)

    // アルファブレンドの有効化
    glEnable(GL_BLEND);

    // ブレンドモードの初期化
    SetBlendMode(BlendMode::Normal);
}

/**
 * @brief   シェーダ―の生成
 */
void CreateShader(
    ShaderPack* pOutShader,
    const GLchar* const* vertexProgram,
    const GLchar* const* fragmentProgram)
{
    NN_ASSERT_NOT_NULL(pOutShader);
    NN_ASSERT_NOT_NULL(vertexProgram);
    NN_ASSERT_NOT_NULL(fragmentProgram);

    GLint result;
    GLchar shaderLog[1024];
    GLsizei shaderLogSize;

    // Vertex shader
    auto vertexId = glCreateShader(GL_VERTEX_SHADER);
    NN_ASSERT(vertexId != 0, "Failed to create vertex shader\n");
    glShaderSource(vertexId, 1, vertexProgram, 0);
    glCompileShader(vertexId);
    glGetShaderiv(vertexId, GL_COMPILE_STATUS, &result);
    if (!result)
    {
        glGetShaderInfoLog(vertexId, sizeof(shaderLog), &shaderLogSize, shaderLog);
        NN_ASSERT(false, "Failed to compile vertex shader: %s\n", shaderLog);
    }

    // Fragment shader
    auto fragmentId = glCreateShader(GL_FRAGMENT_SHADER);
    NN_ASSERT(fragmentId != 0, "Failed to create fragment shader\n");
    glShaderSource(fragmentId, 1, fragmentProgram, 0);
    glCompileShader(fragmentId);
    glGetShaderiv(fragmentId, GL_COMPILE_STATUS, &result);
    if (!result)
    {
        glGetShaderInfoLog(fragmentId, sizeof(shaderLog), &shaderLogSize, shaderLog);
        NN_ASSERT(false, "Failed to compile fragment shader: %s\n", shaderLog);
    }

    // コンパイル
    auto programId = glCreateProgram();
    NN_ASSERT(programId != 0, "Failed to create shader program\n");

    glAttachShader(programId, vertexId);
    glAttachShader(programId, fragmentId);
    glLinkProgram(programId);

    pOutShader->vertexId = vertexId;
    pOutShader->fragmentId = fragmentId;
    pOutShader->programId = programId;
}

/**
 * @brief   シェーダ―の破棄
 */
void DeleteShader(const ShaderPack& shader)
{
    glDetachShader(shader.programId, shader.vertexId);
    glDetachShader(shader.programId, shader.fragmentId);

    glDeleteShader(shader.vertexId);
    glDeleteShader(shader.fragmentId);

    glDeleteProgram(shader.programId);
}

/**
 * @brief   シェーダ―の初期化
 */
void InitializeShaders()
{
    CreateShader(&g_SimpleShader, &VertexShaderSimple, &FragmentShaderSimple);
    g_ShaderLocations.position    = glGetAttribLocation(g_SimpleShader.programId, "a_Position");
    g_ShaderLocations.vertexColor = glGetAttribLocation(g_SimpleShader.programId, "a_Color");

    CreateShader(&g_TextureShader, &VertexShaderTexture, &FragmentShaderTexture);
    g_ShaderLocations.textureCoord = glGetAttribLocation(g_TextureShader.programId, "a_TexCoord");
}

/**
 * @brief   頂点バッファを作成
 */
void InitializeVertexData()
{
    glGenBuffers(NN_ARRAY_SIZE(g_GlBuffers), g_GlBuffers);

    g_VertexBufferId      = g_GlBuffers[0];
    g_VertexColorBufferId = g_GlBuffers[1];
    g_TexCoordBufferId    = g_GlBuffers[2];

    glBindBuffer(GL_ARRAY_BUFFER, g_VertexBufferId);
    glBufferData(GL_ARRAY_BUFFER, sizeof(Point3D) * g_VertexCountMax, g_pVertexPointBuffer, GL_STATIC_DRAW);
    glEnableVertexAttribArray(g_ShaderLocations.position);
    glVertexAttribPointer(g_ShaderLocations.position, NN_ARRAY_SIZE(Point3D::v), GL_FLOAT, GL_FALSE, 0, 0);

    glBindBuffer(GL_ARRAY_BUFFER, g_VertexColorBufferId);
    glBufferData(GL_ARRAY_BUFFER, sizeof(NormalizedColor) * g_VertexCountMax, g_pVertexColorBuffer, GL_STATIC_DRAW);
    glEnableVertexAttribArray(g_ShaderLocations.vertexColor);
    glVertexAttribPointer(g_ShaderLocations.vertexColor, NN_ARRAY_SIZE(NormalizedColor::v), GL_FLOAT, GL_FALSE, 0, 0);

    glBindBuffer(GL_ARRAY_BUFFER, g_TexCoordBufferId);
    glBufferData(GL_ARRAY_BUFFER, sizeof(Point2D) * g_VertexCountMax, g_pVertexTexCoordBuffer, GL_STATIC_DRAW);
    glEnableVertexAttribArray(g_ShaderLocations.textureCoord);
    glVertexAttribPointer(g_ShaderLocations.textureCoord, NN_ARRAY_SIZE(Point2D::v), GL_FLOAT, GL_FALSE, 0, 0);
}

/**
 * @brief   グラフィックス用オブジェクトの破棄
 */
void Destroy()
{
    DeleteShader(g_SimpleShader);
    DeleteShader(g_TextureShader);

    glDeleteBuffers(NN_ARRAY_SIZE(g_GlBuffers), g_GlBuffers);
}

/**
 * @brief   グラフィックスの終了処理
 */
void FinalizeGraphics()
{
    g_GraphicsHelper.Finalize();
}

/**
 * @brief   正規化した X 座標を取得
 */
NN_FORCEINLINE
GLfloat NormalizeX(float x) NN_NOEXCEPT
{
    GLfloat newX = static_cast<GLfloat>(x * 2.0f / g_CanvasSize.width);
    if (g_OriginPosition == OriginPosition::TopLeft)
    {
        newX -= 1.0f;
    }

    return newX;
}

/**
 * @brief   正規化した Y 座標を取得
 */
NN_FORCEINLINE
GLfloat NormalizeY(float y) NN_NOEXCEPT
{
    GLfloat newY = static_cast<GLfloat>(y * 2.0f / g_CanvasSize.height);
    if (g_OriginPosition == OriginPosition::TopLeft)
    {
        newY = 1.0f - newY;
    }

    return newY;
}

/**
 * @brief   2D 座標を正規化した 3D 座標に変換
 */
NN_FORCEINLINE
void Normalize(Point3D* pOutPoint, const Point2D& srcPoint) NN_NOEXCEPT
{
    pOutPoint->x = NormalizeX(srcPoint.x);
    pOutPoint->y = NormalizeY(srcPoint.y);
    pOutPoint->z = 0;
}

/**
 * @brief   描画対象テクスチャを変更済み状態にする
 */
NN_FORCEINLINE
void TouchRenderTarget() NN_NOEXCEPT
{
    if (g_pRenderTargetImage != nullptr)
    {
        g_pRenderTargetImage->isDirty = true;
    }
}

/**
 * @brief   使用するシェーダの切り替え
 */
void ChangeShader(ShaderType type) NN_NOEXCEPT
{
    if (g_CurrentShader == type)
    {
        return;
    }

    switch (type)
    {
    case ShaderType::None:
        glUseProgram(0);
        break;

    case ShaderType::Simple:
        glUseProgram(g_SimpleShader.programId);
        break;

    case ShaderType::Texture:
        glUseProgram(g_TextureShader.programId);
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    g_CurrentShader = type;
}

/**
 * @brief   円描画の共通処理
 */
void DrawCircleCommon(float centerX, float centerY, float radius, const Color& color, float lineWidth, GLenum primitiveType) NN_NOEXCEPT
{
    SCOPED_LOCK;

    ChangeShader(ShaderType::Simple);

    if (radius < lineWidth * 0.1f)
    {
        // 小さすぎる半径は点で処理
        DrawPoint(centerX, centerY, color, lineWidth);
        return;
    }

    const int vertexCount = 54;

    float step = 2 * M_PI / vertexCount;
    float vx = radius;
    float vy = 0.0f;

    for (int i = vertexCount - 1; i >= 0; i--)
    {
        g_TempPointBuffer[i].x = NormalizeX(centerX + vx);
        g_TempPointBuffer[i].y = NormalizeY(centerY + vy);

        // 角速度方向に進める
        float force = vx;
        vx -= vy * step;
        vy += force * step;

        // 半径修正
        force = radius / std::sqrtf(vx * vx + vy * vy);
        vx *= force;
        vy *= force;
    }

    // 色は統一
    for (int i = 0; i < vertexCount; i++)
    {
        g_TempColorBuffer[i].SetU8Color(color);
    }

    glBindBuffer(GL_ARRAY_BUFFER, g_VertexBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(Point3D),
        vertexCount * sizeof(Point3D),
        g_TempPointBuffer);

    glBindBuffer(GL_ARRAY_BUFFER, g_VertexColorBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(NormalizedColor),
        vertexCount * sizeof(NormalizedColor),
        g_TempColorBuffer);

    glLineWidth(lineWidth);
    glDrawArrays(primitiveType, g_NextVertexIndex, vertexCount);

    g_NextVertexIndex += vertexCount;
    NN_ASSERT_LESS(g_NextVertexIndex, g_VertexCountMax);

    TouchRenderTarget();
}

/**
 * @brief   テクスチャ生成の共通処理
 */
ResultCode CreateTextureCommon(ImageData* pOutImage, void* pImageOnMemory) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutImage);
    NN_ASSERT_NOT_NULL(pImageOnMemory);

    SCOPED_LOCK;

    // テクスチャを生成
    glGenTextures(1, &(pOutImage->textureId));
    // TODO: エラー処理

    // テクスチャを割り当てる
    glBindTexture(GL_TEXTURE_2D, pOutImage->textureId);
    glTexImage2D(
        GL_TEXTURE_2D,
        0,
        GL_RGBA,
        pOutImage->size.width,
        pOutImage->size.height,
        0,
        GL_RGBA,
        GL_UNSIGNED_BYTE,
        pImageOnMemory);

    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

    pOutImage->isDirty    = false;
    pOutImage->pPixelData = reinterpret_cast<char*>(pImageOnMemory);

    return ResultCode::Ok;
}

/**
 * @brief   単一の UTF-32 文字の処理
 */
void ProcessUtf32Char(float baseX, float baseY, int printedCount, uint32_t utf32) NN_NOEXCEPT
{
    if (!detail::IsValidSharedFontCharacter(utf32))
    {
        return;
    }

    const int vertexCount = 6;

    Size charSize;
    detail::GetSharedFontCharacterSize(&charSize, utf32);

    charSize.width  *= g_FontContext.scaleX;
    charSize.height *= g_FontContext.scaleY;

    // 座標は GL_TRIANGLES を想定して格納
    // (左上 -> 右上 -> 右下, 左上 -> 右下 -> 左下)

    float x = baseX + g_FontContext.offsetX;
    auto* pVertex = g_TempPointBuffer + printedCount * vertexCount;
    pVertex[0] = Point3D{ { NormalizeX(x), NormalizeY(baseY), 0 } };
    pVertex[1] = Point3D{ { NormalizeX(x + charSize.width), NormalizeY(baseY), 0 } };
    pVertex[2] = Point3D{ { NormalizeX(x + charSize.width), NormalizeY(baseY + charSize.height), 0 } };
    pVertex[3] = pVertex[0];
    pVertex[4] = pVertex[2];
    pVertex[5] = Point3D{ { NormalizeX(x), NormalizeY(baseY + charSize.height), 0 } };

    auto* pColor = g_TempColorBuffer + printedCount * vertexCount;
    pColor[5] = pColor[4] = pColor[3] = pColor[2] = pColor[1] = pColor[0] =
        NormalizedColor::FromU8Color(g_FontContext.color);

    Point2D texCoords[vertexCount];
    detail::GetSharedFontTextureCoord(texCoords, utf32);

    auto* pTexCoord = g_TempTexCoordBuffer + printedCount * vertexCount;
    pTexCoord[0] = texCoords[0];
    pTexCoord[1] = texCoords[1];
    pTexCoord[2] = texCoords[2];
    pTexCoord[3] = texCoords[0];
    pTexCoord[4] = texCoords[2];
    pTexCoord[5] = texCoords[3];

    g_FontContext.offsetX += charSize.width;
}

/**
 * @brief   UTF-32 文字列の描画
 */
void DrawUtf32Text(float x, float y, const uint32_t* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);

    if (y > g_CanvasSize.height + (detail::GetSharedFontLineHeight() * g_FontContext.scaleY) ||
        y < 0.0f)
    {
        return;
    }

    g_FontContext.offsetX = 0;

    auto pString = text;
    bool isNewLine = false;
    int count = 0;

    while (*pString != '\0')
    {
        if (*pString == '\n')
        {
            isNewLine = true;
            pString++;
            break;
        }
        else
        {
            ProcessUtf32Char(x, y, count, *pString);
            count++;
            pString++;
        }
    }

    if (count > 0)
    {
        ChangeShader(ShaderType::Texture);

        auto vertexCount = count * 6;

        glBindBuffer(GL_ARRAY_BUFFER, g_VertexBufferId);
        glBufferSubData(
            GL_ARRAY_BUFFER,
            g_NextVertexIndex * sizeof(Point3D),
            vertexCount * sizeof(Point3D),
            g_TempPointBuffer);

        glBindBuffer(GL_ARRAY_BUFFER, g_VertexColorBufferId);
        glBufferSubData(
            GL_ARRAY_BUFFER,
            g_NextVertexIndex * sizeof(NormalizedColor),
            vertexCount * sizeof(NormalizedColor),
            g_TempColorBuffer);

        glBindBuffer(GL_ARRAY_BUFFER, g_TexCoordBufferId);
        glBufferSubData(
            GL_ARRAY_BUFFER,
            g_NextVertexIndex * sizeof(Point2D),
            vertexCount * sizeof(Point2D),
            g_TempTexCoordBuffer);

        auto textureId = detail::GetSharedFontTextureId();
        glBindTexture(GL_TEXTURE_2D, textureId);
        glDrawArrays(GL_TRIANGLES, g_NextVertexIndex, vertexCount);

        g_NextVertexIndex += vertexCount;
        NN_ASSERT_LESS(g_NextVertexIndex, g_VertexCountMax);
    }

    if (isNewLine)
    {
        float offsetY = detail::GetSharedFontLineHeight() * g_FontContext.scaleY;
        DrawUtf32Text(x, y + offsetY, pString);
    }

    TouchRenderTarget();
}

/**
 * @brief   UTF-32 文字列の描画サイズ取得
 */
void GetUtf32TextSize(Size* pOutSize, const uint32_t* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutSize);
    NN_ASSERT_NOT_NULL(text);

    pOutSize->width  = 0;
    pOutSize->height = 0;

    auto pString = text;
    bool isNewLine = false;

    while (*pString != '\0')
    {
        if (*pString == '\n')
        {
            isNewLine = true;
            pString++;
            break;
        }
        else
        {
            Size charSize;
            detail::GetSharedFontCharacterSize(&charSize, *pString);

            charSize.width  *= g_FontContext.scaleX;
            charSize.height *= g_FontContext.scaleY;

            pOutSize->width += charSize.width;
            pOutSize->height = std::max(pOutSize->height, charSize.width);

            pString++;
        }
    }

    if (isNewLine)
    {
        Size plusSize;
        GetUtf32TextSize(&plusSize, pString);

        pOutSize->width   = std::max(pOutSize->width, plusSize.width);
        pOutSize->height += plusSize.height;
    }
}

}  // anonymous

const Color Colors::Black         = { {   0,   0,   0, 255 } };
const Color Colors::Shadow        = { {  32,  32,  32, 255 } };
const Color Colors::Smoke         = { {  64,  64,  64, 255 } };
const Color Colors::DimGray       = { { 105, 105, 105, 255 } };
const Color Colors::Gray          = { { 128, 128, 128, 255 } };
const Color Colors::DarkGray      = { { 169, 169, 169, 255 } };
const Color Colors::Silver        = { { 192, 192, 192, 255 } };
const Color Colors::White         = { { 255, 255, 255, 255 } };
const Color Colors::Red           = { { 255,   0,   0, 255 } };
const Color Colors::DarkRed       = { { 128,   0,   0, 255 } };
const Color Colors::Crymson       = { { 220,  20,  60, 255 } };
const Color Colors::NeonRed       = { { 255,  62,  52, 255 } };
const Color Colors::Green         = { {   0, 255,   0, 255 } };
const Color Colors::Blue          = { {   0,   0, 255, 255 } };
const Color Colors::Yellow        = { { 255, 255,   0, 255 } };
const Color Colors::Magenta       = { { 255,   0, 255, 255 } };
const Color Colors::Cyan          = { {   0, 255, 255, 255 } };
const Color Colors::Gold          = { { 255, 215,   0, 255 } };
const Color Colors::GoldenRod     = { { 218, 165,  32, 255 } };
const Color Colors::DarkGoldenRod = { { 184, 134,  11, 255 } };
const Color Colors::Orange        = { { 255, 165,   0, 255 } };
const Color Colors::Violet        = { { 238, 130, 238, 255 } };
const Color Colors::LimeGreen     = { {  50, 205,  50, 255 } };
const Color Colors::PaleGreen     = { { 152, 251, 152, 255 } };
const Color Colors::Navy          = { {   0,   0, 128, 255 } };
const Color Colors::RoyalBlue     = { {  65, 105, 225, 255 } };
const Color Colors::NeonBlue      = { {   0, 185, 222, 255 } };

InitializationParameters InitializationParameters::CreateDefault() NN_NOEXCEPT
{
    InitializationParameters params = {};

    params.vertexCountMax = DefaultVertexCountMax;
    params.origin         = OriginPosition::TopLeft;
    params.canvasSize     = DefaultCanvasSize;
    params.baseFontSize   = detail::DefaultFontSize;

    return params;
}

void Initialize() NN_NOEXCEPT
{
    auto params = InitializationParameters::CreateDefault();
    Initialize(params);
}

void Initialize(const InitializationParameters& parameters) NN_NOEXCEPT
{
    SCOPED_LOCK;

    NN_ABORT_UNLESS(!g_IsInitialized, "Already initialized");

    g_FontContext.Initialize();

    g_CurrentShader = ShaderType::Simple;

    g_OriginPosition = parameters.origin;
    g_CanvasSize     = parameters.canvasSize;

    g_pRenderTargetImage = nullptr;

    g_VertexCountMax        = parameters.vertexCountMax;
    g_NextVertexIndex       = 0;
    g_pVertexPointBuffer    = new char[g_VertexCountMax * sizeof(Point3D)];
    g_pVertexColorBuffer    = new char[g_VertexCountMax * sizeof(NormalizedColor)];
    g_pVertexTexCoordBuffer = new char[g_VertexCountMax * sizeof(Point2D)];
    NN_ASSERT_NOT_NULL(g_pVertexPointBuffer);
    NN_ASSERT_NOT_NULL(g_pVertexColorBuffer);
    NN_ASSERT_NOT_NULL(g_pVertexTexCoordBuffer);

    InitializeGraphics();
    InitializeShaders();
    InitializeVertexData();

    auto params = detail::FontInitializationParameters::CreateDefault();
    params.isDynamicMode = true;
    params.baseFontSize  = parameters.baseFontSize;
    detail::InitializeSharedFont(params);

    g_IsInitialized = true;
}

void Finalize() NN_NOEXCEPT
{
    SCOPED_LOCK;

    NN_ABORT_UNLESS(g_IsInitialized, "Not initialized");

    Destroy();
    detail::FinalizeSharedFont();
    FinalizeGraphics();

    delete[] g_pVertexTexCoordBuffer;
    delete[] g_pVertexColorBuffer;
    delete[] g_pVertexPointBuffer;

    g_IsInitialized = false;
}

void BeginRender() NN_NOEXCEPT
{
    SCOPED_LOCK;

    g_NextVertexIndex = 0;
    glClear(GL_COLOR_BUFFER_BIT);

    // ブレンドモードの初期化
    SetBlendMode(BlendMode::Normal);
}

void EndRender() NN_NOEXCEPT
{
    SCOPED_LOCK;

    ResetRenderTarget();
    ChangeShader(ShaderType::None);
    g_GraphicsHelper.SwapBuffers();
}

void SetClearColor(const Color& color) NN_NOEXCEPT
{
    SCOPED_LOCK;

    auto normalColor = NormalizedColor::FromU8Color(color);
    glClearColor(normalColor.r, normalColor.g, normalColor.b, normalColor.a);
}

void SetBlendMode(BlendMode mode) NN_NOEXCEPT
{
    SCOPED_LOCK;

    switch (mode)
    {
    case BlendMode::Normal:
        glBlendEquation(GL_FUNC_ADD);
        glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
        break;

    case BlendMode::Add:
        glBlendEquation(GL_FUNC_ADD);
        glBlendFunc(GL_SRC_ALPHA, GL_ONE);
        break;

    case BlendMode::Subtract:
        glBlendEquation(GL_FUNC_REVERSE_SUBTRACT);
        glBlendFunc(GL_SRC_ALPHA, GL_ONE);
        break;

    case BlendMode::Multiple:
        glBlendEquation(GL_FUNC_ADD);
        glBlendFunc(GL_ZERO, GL_SRC_COLOR);
        break;

    case BlendMode::Screen:
        glBlendEquation(GL_FUNC_ADD);
        glBlendFunc(GL_ONE_MINUS_DST_COLOR, GL_ONE);
        break;

    case BlendMode::Xor:
        glBlendEquation(GL_FUNC_ADD);
        glBlendFunc(GL_ONE_MINUS_DST_COLOR, GL_ONE_MINUS_SRC_COLOR);
        break;

    case BlendMode::Inverse:
        glBlendEquation(GL_FUNC_ADD);
        glBlendFunc(GL_ONE_MINUS_DST_COLOR, GL_ZERO);
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void SetRenderTarget(ImageData* pTargetImage) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTargetImage);

    SCOPED_LOCK;

    if (g_pRenderTargetImage == pTargetImage)
    {
        return;
    }
    else if (g_pRenderTargetImage != nullptr)
    {
        // 別の画像に切り替える前に一旦元に戻す
        ResetRenderTarget();
    }

    // テクスチャへの描画用フレームバッファを生成
    glGenFramebuffers(1, &g_ImageFrameBuffer);
    glBindFramebuffer(GL_FRAMEBUFFER, g_ImageFrameBuffer);
    glFramebufferTexture(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, pTargetImage->textureId, 0);

    // 描画バッファのリストをセット
    GLenum drawBuffers[] = { GL_COLOR_ATTACHMENT0 };
    glDrawBuffers(NN_ARRAY_SIZE(drawBuffers), drawBuffers);

    g_pRenderTargetImage = pTargetImage;
}

void ResetRenderTarget() NN_NOEXCEPT
{
    SCOPED_LOCK;

    if (g_pRenderTargetImage == nullptr)
    {
        return;
    }

    // 標準のフレームバッファに戻す
    glBindFramebuffer(GL_FRAMEBUFFER, 0);
    glDeleteFramebuffers(1, &g_ImageFrameBuffer);

    g_pRenderTargetImage = nullptr;
}

void DrawPoint(float x, float y, const Color& color, float size) NN_NOEXCEPT
{
    SCOPED_LOCK;

    ChangeShader(ShaderType::Simple);

    const int vertexCount = 1;

    Point3D point = { { NormalizeX(x), NormalizeY(y), 0, } };
    glBindBuffer(GL_ARRAY_BUFFER, g_VertexBufferId);
    glBufferSubData(GL_ARRAY_BUFFER, g_NextVertexIndex * sizeof(Point3D), sizeof(point), &point);

    auto colorForGl = NormalizedColor::FromU8Color(color);
    glBindBuffer(GL_ARRAY_BUFFER, g_VertexColorBufferId);
    glBufferSubData(GL_ARRAY_BUFFER, g_NextVertexIndex * sizeof(NormalizedColor), sizeof(colorForGl), &colorForGl);

    glLineWidth(size);
    glDrawArrays(GL_POINTS, g_NextVertexIndex, vertexCount);

    g_NextVertexIndex += vertexCount;
    NN_ASSERT_LESS(g_NextVertexIndex, g_VertexCountMax);

    TouchRenderTarget();
}

void DrawLine(float x1, float y1, float x2, float y2, const Color& color, float lineWidth) NN_NOEXCEPT
{
    DrawGradientLine(x1, y1, x2, y2, color, color, lineWidth);
}

void DrawLine(const Point2D& startPoint, const Point2D& endPoint, const Color& color, float lineWidth) NN_NOEXCEPT
{
    DrawLine(startPoint.x, startPoint.y, endPoint.x, endPoint.y, color, lineWidth);
}

void DrawGradientLine(float x1, float y1, float x2, float y2, const Color& startColor, const Color& endColor, float lineWidth) NN_NOEXCEPT
{
    SCOPED_LOCK;

    ChangeShader(ShaderType::Simple);

    const int vertexCount = 2;

    Point3D points[vertexCount] =
    {
        {{ NormalizeX(x1), NormalizeY(y1), 0 }},
        {{ NormalizeX(x2), NormalizeY(y2), 0 }}
    };
    glBindBuffer(GL_ARRAY_BUFFER, g_VertexBufferId);
    glBufferSubData(GL_ARRAY_BUFFER, g_NextVertexIndex * sizeof(Point3D), sizeof(points), points);

    NormalizedColor colors[vertexCount] =
    {
        NormalizedColor::FromU8Color(startColor),
        NormalizedColor::FromU8Color(endColor)
    };
    glBindBuffer(GL_ARRAY_BUFFER, g_VertexColorBufferId);
    glBufferSubData(GL_ARRAY_BUFFER, g_NextVertexIndex * sizeof(NormalizedColor), sizeof(colors), colors);

    glLineWidth(lineWidth);
    glDrawArrays(GL_LINES, g_NextVertexIndex, vertexCount);

    g_NextVertexIndex += vertexCount;
    NN_ASSERT_LESS(g_NextVertexIndex, g_VertexCountMax);

    TouchRenderTarget();
}

void DrawGradientLine(const Point2D& startPoint, const Point2D& endPoint, const Color& startColor, const Color& endColor, float lineWidth) NN_NOEXCEPT
{
    DrawGradientLine(startPoint.x, startPoint.y, endPoint.x, endPoint.y, startColor, endColor, lineWidth);
}

void DrawRectangle(float x, float y, float width, float height, const Color& color, float lineWidth) NN_NOEXCEPT
{
    Point2D points[] =
    {
        {{ x,         y }},
        {{ x + width, y }},
        {{ x + width, y + height }},
        {{ x,         y + height }}
    };

    DrawPolygon(NN_ARRAY_SIZE(points), points, color, lineWidth);
}

void DrawRectangle(const Rectangle& rect, const Color& color, float lineWidth) NN_NOEXCEPT
{
    DrawRectangle(rect.x, rect.y, rect.width, rect.height, color, lineWidth);
}

void DrawPolygon(int pointCount, const Point2D* points, const Color& color, float lineWidth) NN_NOEXCEPT
{
    NN_ASSERT_LESS_EQUAL(pointCount, TemporaryVertexCountMax);

    SCOPED_LOCK;

    ChangeShader(ShaderType::Simple);

    for (int i = 0; i < pointCount; i++)
    {
        Normalize(&g_TempPointBuffer[i], points[i]);
        g_TempColorBuffer[i].SetU8Color(color);
    }

    glBindBuffer(GL_ARRAY_BUFFER, g_VertexBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(Point3D),
        pointCount * sizeof(Point3D),
        g_TempPointBuffer);

    glBindBuffer(GL_ARRAY_BUFFER, g_VertexColorBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(NormalizedColor),
        pointCount * sizeof(NormalizedColor),
        g_TempColorBuffer);

    glLineWidth(lineWidth);
    glDrawArrays(GL_LINE_LOOP, g_NextVertexIndex, pointCount);

    g_NextVertexIndex += pointCount;
    NN_ASSERT_LESS(g_NextVertexIndex, g_VertexCountMax);

    TouchRenderTarget();
}

void DrawCircle(float centerX, float centerY, float radius, const Color& color, float lineWidth) NN_NOEXCEPT
{
    DrawCircleCommon(centerX, centerY, radius, color, lineWidth, GL_LINE_LOOP);
}

void DrawCircle(const Point2D& center, float radius, const Color& color, float lineWidth) NN_NOEXCEPT
{
    DrawCircle(center.x, center.y, radius, color, lineWidth);
}

void FillRectangle(float x, float y, float width, float height, const Color& color) NN_NOEXCEPT
{
    Point2D points[] =
    {
        {{ x,         y }},
        {{ x + width, y }},
        {{ x + width, y + height }},
        {{ x,         y + height }}
    };

    FillPolygon(NN_ARRAY_SIZE(points), points, color);
}

void FillRectangle(const Rectangle& rectangle, const Color& color) NN_NOEXCEPT
{
    FillRectangle(rectangle.x, rectangle.y, rectangle.width, rectangle.height, color);
}

void FillPolygon(int pointCount, const Point2D* points, const Color& color) NN_NOEXCEPT
{
    NN_ASSERT_LESS_EQUAL(pointCount, TemporaryVertexCountMax);

    SCOPED_LOCK;

    for (int i = 0; i < pointCount; i++)
    {
        g_TempColorBufferU8[i] = color;
    }

    FillGradientPolygon(pointCount, points, g_TempColorBufferU8);
}

void FillGradientPolygon(int pointCount, const Point2D* points, const Color* colors) NN_NOEXCEPT
{
    NN_ASSERT_LESS_EQUAL(pointCount, TemporaryVertexCountMax);

    SCOPED_LOCK;

    ChangeShader(ShaderType::Simple);

    for (int i = 0; i < pointCount; i++)
    {
        Normalize(&g_TempPointBuffer[i], points[i]);
        g_TempColorBuffer[i].SetU8Color(colors[i]);
    }

    glBindBuffer(GL_ARRAY_BUFFER, g_VertexBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(Point3D),
        pointCount * sizeof(Point3D),
        g_TempPointBuffer);

    glBindBuffer(GL_ARRAY_BUFFER, g_VertexColorBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(NormalizedColor),
        pointCount * sizeof(NormalizedColor),
        g_TempColorBuffer);

    glDrawArrays(GL_TRIANGLE_FAN, g_NextVertexIndex, pointCount);

    g_NextVertexIndex += pointCount;
    NN_ASSERT_LESS(g_NextVertexIndex, g_VertexCountMax);

    TouchRenderTarget();
}

void FillCircle(float centerX, float centerY, float radius, const Color& color) NN_NOEXCEPT
{
    DrawCircleCommon(centerX, centerY, radius, color, 1.0f, GL_TRIANGLE_FAN);
}

void FillCircle(const Point2D& center, float radius, const Color& color) NN_NOEXCEPT
{
    FillCircle(center.x, center.y, radius, color);
}

ResultCode LoadImage(ImageData* pOutImage, const char* imageFile) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutImage);
    NN_ASSERT_NOT_NULL(imageFile);

    SCOPED_LOCK;

    auto* pExtension = std::strrchr(imageFile, '.');
    if (pExtension == nullptr)
    {
        // 拡張子がないのでファイル種別判定不能
        return ResultCode::DecodeFailed;
    }

    char* pPixelData;
    {
        ResultCode result;
        if (S2D_STRING_LITERAL_EQUAL(pExtension, ".bmp"))
        {
            result = detail::LoadBitmap(&pOutImage->size, reinterpret_cast<void**>(&pPixelData), imageFile);
        }
        else if (S2D_STRING_LITERAL_EQUAL(pExtension, ".jpg") ||
                 S2D_STRING_LITERAL_EQUAL(pExtension, ".jpeg"))
        {
            result = detail::LoadJpeg(&pOutImage->size, reinterpret_cast<void**>(&pPixelData), imageFile);
        }
        else
        {
            // 非対応の拡張子
            return ResultCode::DecodeFailed;
        }

        if (result != ResultCode::Ok)
        {
            return result;
        }
    }

    {
        auto result = CreateTextureCommon(pOutImage, pPixelData);
        if (result != ResultCode::Ok)
        {
            delete[] pPixelData;
            pOutImage->pPixelData = nullptr;
        }

        return result;
    }
}

void CreateImage(ImageData* pOutImage, const Size& size) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutImage);

    SCOPED_LOCK;

    // 空のテクスチャを作成
    size_t pixelDataSize = size.width * size.height * 4;
    auto pPixelData = new char[pixelDataSize];
    NN_ABORT_UNLESS_NOT_NULL(pPixelData);

    std::memset(pPixelData, 0, pixelDataSize);

    auto result = CreateTextureCommon(pOutImage, pPixelData);

    // 失敗しないはず
    NN_ABORT_UNLESS_EQUAL(result, ResultCode::Ok);
}

void DestroyImage(ImageData* pImage) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pImage);

    SCOPED_LOCK;

    glDeleteTextures(1, &(pImage->textureId));

    if (pImage->pPixelData != nullptr)
    {
        delete[] pImage->pPixelData;
        pImage->pPixelData = nullptr;
    }
}

void DrawImage(const ImageData& image, float x, float y) NN_NOEXCEPT
{
    DrawImage(image, x, y, static_cast<float>(image.size.width), static_cast<float>(image.size.height));
}

void DrawImage(const ImageData& image, const Point2D& point) NN_NOEXCEPT
{
    DrawImage(image, point.x, point.y);
}

void DrawImage(const ImageData& image, float x, float y, float width, float height) NN_NOEXCEPT
{
    Rectangle destRect = { { x, y, width, height } };
    Rectangle srcRect  = { { 0, 0, static_cast<GLfloat>(image.size.width), static_cast<GLfloat>(image.size.height) } };
    DrawImage(image, destRect, srcRect);
}

void DrawImage(const ImageData& image, const Point2D& point, const Size& size) NN_NOEXCEPT
{
    DrawImage(image, point.x, point.y, size.width, size.height);
}

void DrawImage(const ImageData& image, const Point2D& destPoint, const Rectangle& srcRect) NN_NOEXCEPT
{
    Rectangle destRect = { { destPoint.x, destPoint.y, srcRect.width, srcRect.height } };
    DrawImage(image, destRect, srcRect);
}

void DrawImage(const ImageData& image, const Rectangle& destRect, const Rectangle& srcRect) NN_NOEXCEPT
{
    SCOPED_LOCK;

    ChangeShader(ShaderType::Texture);

    const int vertexCount = 4;

    // 頂点座標
    Point3D points[vertexCount] =
    {
        {{ NormalizeX(destRect.x),                  NormalizeY(destRect.y),                   0 }},
        {{ NormalizeX(destRect.x + destRect.width), NormalizeY(destRect.y),                   0 }},
        {{ NormalizeX(destRect.x + destRect.width), NormalizeY(destRect.y + destRect.height), 0 }},
        {{ NormalizeX(destRect.x),                  NormalizeY(destRect.y + destRect.height), 0 }}
    };
    glBindBuffer(GL_ARRAY_BUFFER, g_VertexBufferId);
    glBufferSubData(GL_ARRAY_BUFFER, g_NextVertexIndex * sizeof(Point3D), sizeof(points), points);

    // 頂点色
    const auto vertexColor = NormalizedColor::FromU8Color(Colors::White);
    NormalizedColor colors[vertexCount] =
    {
        vertexColor,
        vertexColor,
        vertexColor,
        vertexColor
    };
    glBindBuffer(GL_ARRAY_BUFFER, g_VertexColorBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(NormalizedColor),
        vertexCount * sizeof(NormalizedColor),
        colors);

    // 描画サイズからテクスチャ座標を計算
    GLfloat x      = srcRect.x      / static_cast<GLfloat>(image.size.width);
    GLfloat y      = srcRect.y      / static_cast<GLfloat>(image.size.height);
    GLfloat width  = srcRect.width  / static_cast<GLfloat>(image.size.width);
    GLfloat height = srcRect.height / static_cast<GLfloat>(image.size.height);
    Point2D texCoords[vertexCount]
    {
        {{ x,         y }},
        {{ x + width, y }},
        {{ x + width, y + height }},
        {{ x,         y + height }}
    };
    glBindBuffer(GL_ARRAY_BUFFER, g_TexCoordBufferId);
    glBufferSubData(GL_ARRAY_BUFFER, g_NextVertexIndex * sizeof(Point2D), sizeof(texCoords), texCoords);

    glBindTexture(GL_TEXTURE_2D, image.textureId);
    glDrawArrays(GL_TRIANGLE_FAN, g_NextVertexIndex, vertexCount);

    g_NextVertexIndex += vertexCount;
    NN_ASSERT_LESS(g_NextVertexIndex, g_VertexCountMax);

    TouchRenderTarget();
}

void SetTextColor(const Color& color) NN_NOEXCEPT
{
    SCOPED_LOCK;

    g_FontContext.color = color;
}

void SetTextScale(nn::util::Float2 scale) NN_NOEXCEPT
{
    SCOPED_LOCK;

    g_FontContext.scaleX = std::max(scale.x, 0.0f);
    g_FontContext.scaleY = std::max(scale.y, 0.0f);
}

bool VConvertUtf8ToUtf32(uint32_t* pOutBuffer, int bufferLength, const char* text, std::va_list args) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutBuffer);
    NN_ASSERT_NOT_NULL(text);

    static char s_Buffer[TemporaryStringBufferSize];

    // 書式指定の展開
    auto length = std::vsnprintf(s_Buffer, sizeof(s_Buffer), text, args);
    if (length < 0)
    {
        // バッファサイズオーバー
        return false;
    }

    // UTF8 -> UTF32 変換
    {
        int lengthUtf32;
        auto result = nn::util::GetLengthOfConvertedStringUtf8ToUtf32(&lengthUtf32, s_Buffer);
        if (result != nn::util::CharacterEncodingResult_Success ||
            lengthUtf32 >= TemporaryStringBufferSize)
        {
            // 変換不能
            return false;
        }
    }

    auto result = nn::util::ConvertStringUtf8ToUtf32(pOutBuffer, TemporaryStringBufferSize, s_Buffer);
    if (result != nn::util::CharacterEncodingResult_Success)
    {
        // 変換失敗
        return false;
    }

    return true;
}

void DrawText(const Point2D& position, const char* format, ...) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(format);

    SCOPED_LOCK;

    static uint32_t s_Utf32Buffer[TemporaryStringBufferSize];

    std::va_list args;
    va_start(args, format);
    auto success = VConvertUtf8ToUtf32(s_Utf32Buffer, TemporaryStringBufferSize, format, args);
    va_end(args);
    if (!success)
    {
        // 変換失敗
        return;
    }

    DrawUtf32Text(position.x, position.y, s_Utf32Buffer);
}

void DrawText(const Point2D& position, const uint32_t* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);

    SCOPED_LOCK;

    DrawUtf32Text(position.x, position.y, text);
}

void DrawText(const Point2D& position, const uint16_t* text, int textLength) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);

    SCOPED_LOCK;

    // UTF16 -> UTF8 変換
    static char s_Utf8Buffer[TemporaryStringBufferSize];

    {
        int lengthUtf8;
        auto result = nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8(&lengthUtf8, text, textLength);
        if (result != nn::util::CharacterEncodingResult_Success ||
            lengthUtf8 >= TemporaryStringBufferSize)
        {
            // 変換不能
            return;
        }
    }

    auto result = nn::util::ConvertStringUtf16NativeToUtf8(s_Utf8Buffer, TemporaryStringBufferSize, text);
    if (result != nn::util::CharacterEncodingResult_Success)
    {
        // 変換失敗
        return;
    }

    DrawText(position, s_Utf8Buffer);
}

void GetTextDrawSize(Size* pOutSize, const char* format, ...) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutSize);
    NN_ASSERT_NOT_NULL(format);

    SCOPED_LOCK;

    static uint32_t s_Utf32Buffer[TemporaryStringBufferSize];

    std::va_list args;
    va_start(args, format);
    auto success = VConvertUtf8ToUtf32(s_Utf32Buffer, TemporaryStringBufferSize, format, args);
    va_end(args);
    if (!success)
    {
        // 変換失敗
        pOutSize->width  = 0;
        pOutSize->height = 0;
        return;
    }

    GetTextDrawSize(pOutSize, s_Utf32Buffer);
}

void GetTextDrawSize(Size* pOutSize, const uint32_t* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutSize);
    NN_ASSERT_NOT_NULL(text);

    SCOPED_LOCK;

    GetUtf32TextSize(pOutSize, text);
}

}  // s2d
