﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <deque>
#include <string>
#include <vector>
#include <nn/nn_Common.h>
#include <nn/util/util_MathTypes.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/hid/system/hid_Npad.h>

#include "ScreenStatics.h"
#include "s2d/Simple2D.h"


/**
 * @brief       Test 内で追加定義されたデバイスタイプ
 */
struct TestDeviceType : nn::hid::system::DeviceType
{
    typedef ::nn::util::BitFlagSet<32, DeviceType>::Flag<30> UsbCompatibleController;      //!< Switch Proコントローラー互換の Usb コントローラー
};

struct RecordableFloat
{
    float value;
    float min;
    float max;
    RecordableFloat& operator+= (const float& input) NN_NOEXCEPT
    {
        max = std::max(max, value + input);
        value += input;
        return *this;
    };
    RecordableFloat& operator-= (const float& input) NN_NOEXCEPT
    {
        min = std::min(min, value - input);
        value -= input;
        return *this;
    };
    RecordableFloat& operator= (const float& input) NN_NOEXCEPT
    {
        max = std::max(max, input);
        min = std::min(min, input);
        value = input;
        return *this;
    };
    void Reset(const float& y) NN_NOEXCEPT
    {
        min = y;
        max = y;
        value = y;
    }
    float Length() NN_NOEXCEPT
    {
        return max - min;
    }
};

struct RecordablePosition
{
    float x;
    RecordableFloat y;

    void Reset(const s2d::Point2D& input) NN_NOEXCEPT
    {
        x = input.x;
        y.Reset(input.y);
    };

    float GetHeight() NN_NOEXCEPT
    {
        return y.Length();
    };

    void NewLine() NN_NOEXCEPT
    {
        y += YSizeLine;
    };

    RecordablePosition& operator= (const s2d::Point2D& input) NN_NOEXCEPT
    {
        x = input.x;
        y = input.y;
        return *this;
    }

    const s2d::Point2D Get() const NN_NOEXCEPT
    {
        return { { x, y.value } };
    }
};

inline float SelectablePrintAndStep(const s2d::Point2D& position, const char* text, bool padding, bool draw) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);

    s2d::Size size;
    s2d::GetTextDrawSize(&size, text);
    if (draw)
    {
        s2d::DrawText(position, text);
    }
    return (size.width + XSizePadding);
}

inline float PrintAndStep(const s2d::Point2D& position, const char* text, bool padding) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);
    return SelectablePrintAndStep(position, text, padding, true);
}

inline float PrintAndStep(const s2d::Point2D& position, const char* text) NN_NOEXCEPT
{
    return PrintAndStep(position, text, false);
}

std::string GetDeviceTypeString(const nn::hid::system::DeviceTypeSet& deviceType) NN_NOEXCEPT;
std::string GetDeviceTypeShort(const nn::hid::system::DeviceTypeSet& deviceType) NN_NOEXCEPT;
void DrawDeviceType(RecordablePosition& position, const nn::hid::system::NpadDeviceTypeSet deviceType) NN_NOEXCEPT;
void DrawIndicator(RecordablePosition& position, ::nn::Bit8 pattern) NN_NOEXCEPT;
void DrawColor(RecordablePosition& position, const nn::util::Color4u8Type& color) NN_NOEXCEPT;
void DrawArrow(const s2d::Point2D& startPoint, const s2d::Point2D& endPoint, const s2d::Color& color, float lineWidth, const float& size) NN_NOEXCEPT;
void DrawInterface(RecordablePosition& position, const nn::hid::system::InterfaceType& interfaceType) NN_NOEXCEPT;
