﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <complex>

#include "SceneSixAxisSensor.h"
#include "ScreenStatics.h"
#include "ScreenUtil.h"
#include "s2d/Simple2D.h"

namespace {

const VectorChartUiParameterConfigList VectorChartUiParameterConfigListAcc =
{
    {
        {
            {200.0f, 600.0f, 20.0f, 1.1f},  // sampleCount
            {2.0f, 16.0f, 0.05f, 1.1f}, // scale
        }
    }
};

const VectorChartUiParameterConfigList VectorChartUiParameterConfigListAngle =
{
    {
        {
            {200.0f, 600.0f, 20.0f, 1.1f},  // sampleCount
            {1.0f, 1.0f, 1.0f, 1.0f}, // scale
        }
    }
};

const LineChartUiParameterConfigList LineChartUiParameterConfigListSensor =
{
    {
        {
            {200.0f, 600.0f, 20.0f, 1.1f},  // xSampleCount
            {2.0f, 16.0f, 0.05f, 1.1f}, // yScale
            {0.0f, 16.0f, -16.0f, 0.0f}   // yShift
        }
    }
};

class ILineChartAccessorSensor : public ILineChartDataSetAccessor
{
protected:
    SixAxisSensorStateFifo* m_pFifo;
public:
    virtual int size() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFifo->size();
    }
    ILineChartAccessorSensor(const std::string label, const s2d::Color& color) NN_NOEXCEPT
    {
        m_Label = label;
        m_Color = color;
    }
    void SetSixAxisSensorFifo(SixAxisSensorStateFifo* pFifo) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pFifo);
        m_pFifo = pFifo;
    }
};

enum SixAxisSensorAxis
{
    SixAxisSensorAxis_X = 0,
    SixAxisSensorAxis_Y = 1,
    SixAxisSensorAxis_Z = 2,
};

class LineChartAccessorAcceleration final : public ILineChartAccessorSensor
{
    SixAxisSensorAxis m_Axis;

public:
    LineChartAccessorAcceleration(const std::string label, const s2d::Color& color, SixAxisSensorAxis axis) NN_NOEXCEPT :
        ILineChartAccessorSensor(label, color),
        m_Axis(axis)
    {
        // 何もしない
    };
    virtual float at(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFifo->at(index).acceleration.v[m_Axis];
    }
    virtual const char* backStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%1.3f", m_pFifo->back().acceleration.v[m_Axis]);
        return buf;
    }
    virtual const char* maxStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%1.3f", m_pFifo->GetMax().acceleration.v[m_Axis]);
        return buf;
    }
    virtual const char* minStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%1.3f", m_pFifo->GetMin().acceleration.v[m_Axis]);
        return buf;
    }
};

class LineChartAccessorAccelerationLength final : public ILineChartAccessorSensor
{
public:
    LineChartAccessorAccelerationLength(const std::string label, const s2d::Color& color) NN_NOEXCEPT :
        ILineChartAccessorSensor(label, color)
    {
        // 何もしない
    };
    virtual float at(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFifo->at(index).accelerationLength3d;
    }
    virtual const char* backStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%1.3f", m_pFifo->back().accelerationLength3d);
        return buf;
    }
    virtual const char* maxStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%1.3f", m_pFifo->GetMax().accelerationLength3d);
        return buf;
    }
    virtual const char* minStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%1.3f", m_pFifo->GetMin().accelerationLength3d);
        return buf;
    }
};
class LineChartAccessorAngularVelocity final : public ILineChartAccessorSensor
{
    SixAxisSensorAxis m_Axis;

public:
    LineChartAccessorAngularVelocity(const std::string label, const s2d::Color& color, SixAxisSensorAxis axis) NN_NOEXCEPT :
        ILineChartAccessorSensor(label, color),
        m_Axis(axis)
    {
        // 何もしない
    };
    virtual float at(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFifo->at(index).angularVelocity.v[m_Axis];
    }
    virtual const char* backStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%1.3f", m_pFifo->back().angularVelocity.v[m_Axis]);
        return buf;
    }
    virtual const char* maxStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%1.3f", m_pFifo->GetMax().angularVelocity.v[m_Axis]);
        return buf;
    }
    virtual const char* minStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%1.3f", m_pFifo->GetMin().angularVelocity.v[m_Axis]);
        return buf;
    }
};

class LineChartAccessorAngle final : public ILineChartAccessorSensor
{
    SixAxisSensorAxis m_Axis;

public:
    LineChartAccessorAngle(const std::string label, const s2d::Color& color, SixAxisSensorAxis axis) NN_NOEXCEPT :
        ILineChartAccessorSensor(label, color),
        m_Axis(axis)
    {
        // 何もしない
    };
    virtual float at(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFifo->at(index).angle.v[m_Axis];
    }
    virtual const char* backStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%1.3f", m_pFifo->back().angle.v[m_Axis]);
        return buf;
    }
    virtual const char* maxStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%1.3f", m_pFifo->GetMax().angle.v[m_Axis]);
        return buf;
    }
    virtual const char* minStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%1.3f", m_pFifo->GetMin().angle.v[m_Axis]);
        return buf;
    }
};

class IVectorChartAccessorSensor : public IVectorChartDataSetAccessor
{
protected:
    SixAxisSensorStateFifo* m_pFifo;
public:
    IVectorChartAccessorSensor(const std::string label, const s2d::Color& color) NN_NOEXCEPT
    {
        m_Label = label;
        m_Color = color;
    }
    virtual int size() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFifo->size();
    }

    void SetSixAxisSensorFifo(SixAxisSensorStateFifo* pFifo) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pFifo);
        m_pFifo = pFifo;
    }
};

class VectorChartAccessorSensorAcceleration : public IVectorChartAccessorSensor
{
private:
    SixAxisSensorAxis m_XAxis;
    SixAxisSensorAxis m_YAxis;

public:
    VectorChartAccessorSensorAcceleration(const std::string label, const s2d::Color& color, SixAxisSensorAxis xAxis, SixAxisSensorAxis yAxis) NN_NOEXCEPT :
        IVectorChartAccessorSensor(label, color),
        m_XAxis(xAxis),
        m_YAxis(yAxis)
    {
        // 何もしない
    }
    virtual VectorType back() NN_NOEXCEPT NN_OVERRIDE
    {
        VectorType output;
        output.x = m_pFifo->back().acceleration.v[m_XAxis];
        output.y = m_pFifo->back().acceleration.v[m_YAxis];
        output.length = m_pFifo->back().GetLength(m_XAxis, m_YAxis);
        return output;
    }
    virtual VectorType at(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        VectorType output;
        output.x = m_pFifo->at(index).acceleration.v[m_XAxis];
        output.y = m_pFifo->at(index).acceleration.v[m_YAxis];
        output.length = m_pFifo->at(index).GetLength(m_XAxis, m_YAxis);
        return output;
    }
    virtual const char* xStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%1.3f", m_pFifo->back().acceleration.v[m_XAxis]);
        return buf;
    }

    virtual const char* yStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%1.3f", m_pFifo->back().acceleration.v[m_YAxis]);
        return buf;
    }

    virtual const char* dirStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%3.2f", m_pFifo->back().GetAngle(m_XAxis, m_YAxis));
        return buf;
    }

    virtual const char* lenStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%3.2f", m_pFifo->back().GetLength(m_XAxis, m_YAxis));
        return buf;
    }

    virtual const char* lenMaxStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%3.2f", m_pFifo->GetMax().GetLength(m_XAxis, m_YAxis));
        return buf;
    }

    virtual const char* lenMinStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%3.2f", m_pFifo->GetMin().GetLength(m_XAxis, m_YAxis));
        return buf;
    }
};

class VectorChartAccessorSensorAngle : public IVectorChartAccessorSensor
{
private:
    SixAxisSensorAxis m_Axis;

public:
    VectorChartAccessorSensorAngle(const std::string label, const s2d::Color& color, SixAxisSensorAxis axis) NN_NOEXCEPT :
        IVectorChartAccessorSensor(label, color),
        m_Axis(axis)
    {
        // 何もしない
    }
    virtual VectorType back() NN_NOEXCEPT NN_OVERRIDE
    {
        VectorType output;
        output.x = std::cos(m_pFifo->back().angle.v[m_Axis] * 2 * nn::util::FloatPi);
        output.y = std::sin(m_pFifo->back().angle.v[m_Axis] * 2 * nn::util::FloatPi);
        output.length = 1.0f;
        return output;
    }
    virtual VectorType at(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        VectorType output;
        output.x = std::cos(m_pFifo->at(index).angle.v[m_Axis] * 2 * nn::util::FloatPi);
        output.y = std::sin(m_pFifo->at(index).angle.v[m_Axis] * 2 * nn::util::FloatPi);
        output.length = 1.0f;
        return output;
    }
    virtual const char* xStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "---");
        return buf;
    }

    virtual const char* yStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "---");
        return buf;
    }

    virtual const char* dirStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%3.2f", m_pFifo->back().angle.v[m_Axis]);
        return buf;
    }

    virtual const char* lenStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "---");
        return buf;
    }

    virtual const char* lenMaxStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "---");
        return buf;
    }

    virtual const char* lenMinStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "---");
        return buf;
    }
};

} // namespace

SceneSixAxisSensor::SceneSixAxisSensor() NN_NOEXCEPT :
    m_LineChartAcceleration(LineChartUiParameterConfigListSensor, true),
    m_LineChartAngularVelocity(LineChartUiParameterConfigListSensor, true),
    m_VectorChartAcceleration(VectorChartUiParameterConfigListAcc, true),
    m_VectorChartAngle(VectorChartUiParameterConfigListAngle, true),
    m_IsMenuLocked(false)
{
}

void SceneSixAxisSensor::OnSceneFocused() NN_NOEXCEPT
{
    m_pControllers->ClearSelectedController();
}

void SceneSixAxisSensor::SetNpadControllerList(NpadControllerList* pControllers) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pControllers);
    m_pControllers = pControllers;
}

void SceneSixAxisSensor::DrawImpl(RecordablePosition& position) NN_NOEXCEPT
{
    float xOrigin = position.x;

    LineChartAccessorAcceleration accXAccessor   = { "AccX", ColorSceneTextStrong01, SixAxisSensorAxis_X };
    LineChartAccessorAcceleration accYAccessor   = { "AccY", ColorSceneTextStrong04, SixAxisSensorAxis_Y };
    LineChartAccessorAcceleration accZAccessor   = { "AccZ", ColorSceneTextStrong03, SixAxisSensorAxis_Z };
    LineChartAccessorAccelerationLength accLenAccessor = { "Len",  ColorSceneTextStrong02 };
    LineChartAccessorAngularVelocity gyroXAccessor   = { "GyroX", ColorSceneTextStrong01, SixAxisSensorAxis_X };
    LineChartAccessorAngularVelocity gyroYAccessor   = { "GyroY", ColorSceneTextStrong04, SixAxisSensorAxis_Y };
    LineChartAccessorAngularVelocity gyroZAccessor   = { "GyroZ", ColorSceneTextStrong03, SixAxisSensorAxis_Z };

    VectorChartAccessorSensorAcceleration accXYAccessor   = { "AccXY", ColorSceneTextStrong01, SixAxisSensorAxis_X, SixAxisSensorAxis_Y };
    VectorChartAccessorSensorAcceleration accXZAccessor   = { "AccXZ", ColorSceneTextStrong04, SixAxisSensorAxis_X, SixAxisSensorAxis_Z };
    VectorChartAccessorSensorAcceleration accZYAccessor   = { "AccZY", ColorSceneTextStrong03, SixAxisSensorAxis_Z, SixAxisSensorAxis_Y };
    VectorChartAccessorSensorAngle vecAngleXAccessor   = { "AngX", ColorSceneTextStrong01, SixAxisSensorAxis_X };
    VectorChartAccessorSensorAngle vecAngleYAccessor   = { "AngY", ColorSceneTextStrong04, SixAxisSensorAxis_Y };
    VectorChartAccessorSensorAngle vecAngleZAccessor   = { "AngZ", ColorSceneTextStrong03, SixAxisSensorAxis_Z };

    ILineChartAccessorSensor* pLineChartSensorAccessors[] = {
        &accXAccessor,
        &accYAccessor,
        &accZAccessor,
        &accLenAccessor,
    };
    ILineChartAccessorSensor* pLineChartAngleAccessors[] = {
        &gyroXAccessor,
        &gyroYAccessor,
        &gyroZAccessor,
    };
    IVectorChartAccessorSensor* pVectorAccelerationAccessors[] = {
        &accXYAccessor,
        &accXZAccessor,
        &accZYAccessor,
    };
    IVectorChartAccessorSensor* pVectorAngleAccessors[] = {
        &vecAngleXAccessor,
        &vecAngleYAccessor,
        &vecAngleZAccessor,
    };
    bool isDeviceAvailable = false;

    NpadStyleHandler* pStyle;
    DeviceIndex deviceIndex;
    if (m_pControllers->GetSelectedStyleHandler(&pStyle) == true &&
        m_pControllers->GetSelectedDeviceIndex(&deviceIndex) == true)
    {
        auto pFifo = pStyle->GetSixAxisSensorFifo(deviceIndex);

        for (auto& pAccessor : pLineChartSensorAccessors)
        {
            pAccessor->SetSixAxisSensorFifo(pFifo);
        }
        for (auto& pAccessor : pLineChartAngleAccessors)
        {
            pAccessor->SetSixAxisSensorFifo(pFifo);
        }
        for (auto& pAccessor : pVectorAngleAccessors)
        {
            pAccessor->SetSixAxisSensorFifo(pFifo);
        }
        for (auto& pAccessor : pVectorAccelerationAccessors)
        {
            pAccessor->SetSixAxisSensorFifo(pFifo);
        }
        isDeviceAvailable = true;
    }

    // ベクトルの描画
    m_VectorChartAcceleration.Draw(position.Get(), SizeSensorVectorChart, "XZ", "YZ", reinterpret_cast<IVectorChartDataSetAccessor**>(&pVectorAccelerationAccessors), (isDeviceAvailable) ? NN_ARRAY_SIZE(pVectorAccelerationAccessors) : 0);
    position.x += SceneSize.width / 2;
    m_VectorChartAngle.Draw(position.Get(), SizeSensorVectorChart, "X", "Y", reinterpret_cast<IVectorChartDataSetAccessor**>(&pVectorAngleAccessors), (isDeviceAvailable) ? NN_ARRAY_SIZE(pVectorAngleAccessors) : 0);

    // グラフの描画
    position.x = xOrigin;
    position.y += SizeSensorVectorChart + YSizeLine * 2;
    const s2d::Size size = { {XSizeLineChartFull, YSizeSensorLineChart} };
    m_LineChartAcceleration.Draw(position.Get(), size, false, reinterpret_cast<ILineChartDataSetAccessor**>(&pLineChartSensorAccessors), (isDeviceAvailable) ? NN_ARRAY_SIZE(pLineChartSensorAccessors) : 0);
    position.y += YSizeSensorLineChartFull + YSizeLine * 2;
    m_LineChartAngularVelocity.Draw(position.Get(), size, false, reinterpret_cast<ILineChartDataSetAccessor**>(&pLineChartAngleAccessors), (isDeviceAvailable) ? NN_ARRAY_SIZE(pLineChartAngleAccessors) : 0);

}

PackedButton SceneSixAxisSensor::Control(const PackedButton& button) NN_NOEXCEPT
{
    const nn::hid::NpadButtonSet LockButtons = nn::hid::NpadButton::Plus::Mask | nn::hid::NpadButton::Minus::Mask;

    if ((button.longTrigger & LockButtons).IsAnyOn())
    {
        m_IsMenuLocked = !m_IsMenuLocked;
    }
    if (m_IsMenuLocked)
    {
        auto triggerWithRepeat = button.trigger | button.repeat;
        if (triggerWithRepeat.Test<nn::hid::NpadButton::Up>())
        {
            m_LineChartAcceleration.Decrement(LineChart::UiParameterType_YScale);
            m_LineChartAngularVelocity.Decrement(LineChart::UiParameterType_YScale);
            m_VectorChartAcceleration.Decrement(VectorChart::UiParameterType_Scale);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::Down>())
        {
            m_LineChartAcceleration.Increment(LineChart::UiParameterType_YScale);
            m_LineChartAngularVelocity.Increment(LineChart::UiParameterType_YScale);
            m_VectorChartAcceleration.Increment(VectorChart::UiParameterType_Scale);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::Left>())
        {
            m_LineChartAcceleration.Decrement(LineChart::UiParameterType_XSampleCount);
            m_LineChartAngularVelocity.Decrement(LineChart::UiParameterType_XSampleCount);
            m_VectorChartAcceleration.Decrement(VectorChart::UiParameterType_SampleCount);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::Right>())
        {
            m_LineChartAcceleration.Increment(LineChart::UiParameterType_XSampleCount);
            m_LineChartAngularVelocity.Increment(LineChart::UiParameterType_XSampleCount);
            m_VectorChartAcceleration.Increment(VectorChart::UiParameterType_SampleCount);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::X>())
        {
            m_LineChartAcceleration.Increment(LineChart::UiParameterType_YShift);
            m_LineChartAngularVelocity.Increment(LineChart::UiParameterType_YShift);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::B>())
        {
            m_LineChartAcceleration.Decrement(LineChart::UiParameterType_YShift);
            m_LineChartAngularVelocity.Decrement(LineChart::UiParameterType_YShift);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::L>() ||
            triggerWithRepeat.Test<nn::hid::NpadButton::R>())
        {
            m_pControllers->ResetSelectedControllerFifo();
        }

        nn::hid::NpadButtonSet value;
        value.Reset();
        return button & value;
    }

    return button & ~LockButtons;
}
