﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <complex>

#include "PacketChart.h"
#include "SceneRxPacketHistory.h"
#include "ScreenStatics.h"
#include "ScreenUtil.h"
#include "s2d/Simple2D.h"

namespace {

const LineChartUiParameterConfigList LineChartUiParameterConfigListPlr =
{
    {
        {
            {100.0f, 600.0f, 20.0f, 1.1f},  // xSampleCount
            {20, 40, 2.0f, 1.1f}, // yScale
            {25, 100, 0, 0.0f}   // yShift
        }
    }
};
const LineChartUiParameterConfigList LineChartUiParameterConfigListReceived =
{
    {
        {
            {100.0f, 600.0f, 20.0f, 1.1f},  // xSampleCount
            {120, 120, 2.0f, 1.1f}, // yScale
            {150, 300, 0, 0.0f}   // yShift
        }
    }
};

class LineChartPlr : public ILineChartDataSetAccessor
{
protected:
    RxPacketHistoryFifo* m_pFifo;
public:
    virtual int size() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFifo->GetPlr().size();
    }
    void SetChartInfo(const std::string label, const s2d::Color& color) NN_NOEXCEPT
    {
        m_Label = label;
        m_Color = color;
    }
    void SetRxPacketHistoryFifo(RxPacketHistoryFifo* pFifo) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pFifo);
        m_pFifo = pFifo;
    }
    virtual float at(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFifo->GetPlr().at(index).plr;
    }
    virtual const char* backStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%3.2f", m_pFifo->GetPlr().back().plr);
        return buf;
    }
    virtual const char* maxStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%3.2f", m_pFifo->GetPlr().GetMax().plr);
        return buf;
    }
    virtual const char* minStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%3.2f", m_pFifo->GetPlr().GetMin().plr);
        return buf;
    }
};

class LineChartReceived : public ILineChartDataSetAccessor
{
protected:
    RxPacketHistoryFifo* m_pFifo;
public:
    virtual int size() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFifo->GetPlr().size();
    }
    void SetChartInfo(const std::string label, const s2d::Color& color) NN_NOEXCEPT
    {
        m_Label = label;
        m_Color = color;
    }
    void SetRxPacketHistoryFifo(RxPacketHistoryFifo* pFifo) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pFifo);
        m_pFifo = pFifo;
    }
    virtual float at(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFifo->GetPlr().at(index).received;
    }
    virtual const char* backStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%d", m_pFifo->GetPlr().back().received);
        return buf;
    }
    virtual const char* maxStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%d", m_pFifo->GetPlr().GetMax().received);
        return buf;
    }
    virtual const char* minStr(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%d", m_pFifo->GetPlr().GetMin().received);
        return buf;
    }
};

class PacketChartAccessor : public IPacketChartDataSetAccessor
{
protected:
    RxPacketHistoryFifo* m_pFifo;
public:
    virtual int size() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pFifo->size();
    }
    void SetChartInfo(const std::string label, const s2d::Color& color) NN_NOEXCEPT
    {
        m_Label = label;
        m_Color = color;
    }
    void SetRxPacketHistoryFifo(RxPacketHistoryFifo* pFifo) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pFifo);
        m_pFifo = pFifo;
    }
    virtual bool at(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        return (m_pFifo->at(index).status == nn::hid::detail::PacketStatus_Received);
    }
    virtual const char* continuousDrop(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%d", m_pFifo->GetContinuouPacketDrop());
        return buf;
    }
    virtual const char* maxDrop(char* buf) NN_NOEXCEPT NN_OVERRIDE
    {
        std::sprintf(buf, "%d", m_pFifo->GetMaxContinuouPacketDrop());
        return buf;
    }
};

} // namespace

SceneRxPacketHistory::SceneRxPacketHistory() NN_NOEXCEPT :
    m_LineChartPlr(LineChartUiParameterConfigListPlr, false),
    m_LineChartReceived(LineChartUiParameterConfigListReceived, false),
    m_IsMenuLocked(false)
{
}

void SceneRxPacketHistory::OnSceneFocused() NN_NOEXCEPT
{
    m_pControllers->ClearSelectedController();
}

void SceneRxPacketHistory::SetUniquePadList(UniquePadList* pControllers) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pControllers);
    m_pControllers = pControllers;
}

void SceneRxPacketHistory::DrawImpl(RecordablePosition& position) NN_NOEXCEPT
{
    float xOrigin = position.x;

    LineChartPlr lineChartPlr[nn::hid::system::UniquePadIdCountMax];
    ILineChartDataSetAccessor* pLineChartAccessorsPlr[nn::hid::system::UniquePadIdCountMax];
    LineChartReceived lineChartReceived[nn::hid::system::UniquePadIdCountMax];
    ILineChartDataSetAccessor* pLineChartAccessorsReceived[nn::hid::system::UniquePadIdCountMax];
    int dataSetCount = 0;
    for (auto& controller : *m_pControllers)
    {
        auto pRxHistory = controller->GetRxPacketHistoryFifo();
        if (pRxHistory->size() > 0)
        {
            char label[10] = {};
            std::sprintf(label, "%02d-%s", controller->GetPlayerNumber(), GetDeviceTypeShort(controller->GetDeviceType()).c_str());
            lineChartPlr[dataSetCount].SetRxPacketHistoryFifo(pRxHistory);
            lineChartPlr[dataSetCount].SetChartInfo(label, ColorSceneGraphLine[dataSetCount]);
            pLineChartAccessorsPlr[dataSetCount] = &lineChartPlr[dataSetCount];

            lineChartReceived[dataSetCount].SetRxPacketHistoryFifo(pRxHistory);
            lineChartReceived[dataSetCount].SetChartInfo(label, ColorSceneGraphLine[dataSetCount]);
            pLineChartAccessorsReceived[dataSetCount] = &lineChartReceived[dataSetCount];
            ++dataSetCount;
        }
    }

    // グラフの描画
    position.x = xOrigin;
    position.y += YSizeLine;
    const s2d::Size size = { {XSizeLineChartFull, YSizeRxMonitorLineChart} };
    m_LineChartPlr.Draw(position.Get(), size, true, pLineChartAccessorsPlr, dataSetCount);
    position.y += YSizeRxMonitorLineChartFull;
    m_LineChartReceived.Draw(position.Get(), size, true, pLineChartAccessorsReceived, dataSetCount);
    position.y += YSizeRxMonitorLineChartFull + YSizeLine;

    dataSetCount = 0;
    for (auto& controller : *m_pControllers)
    {
        auto pRxHistory = controller->GetRxPacketHistoryFifo();
        if (pRxHistory->size() > 0)
        {
            char label[10] = {};
            std::sprintf(label, "%02d-%s", controller->GetPlayerNumber(), GetDeviceTypeShort(controller->GetDeviceType()).c_str());
            PacketChart packetChart;
            PacketChartAccessor packetChartAccessor;
            packetChartAccessor.SetRxPacketHistoryFifo(pRxHistory);
            packetChartAccessor.SetChartInfo(label, ColorSceneGraphLine[dataSetCount]);
            const s2d::Size packetChartSize = { {XSizePacketChartHalf, YSizeRxMonitorPacketChart} };
            packetChart.Draw(150, position.Get(), packetChartSize, &packetChartAccessor);
            if (dataSetCount % 2 == 0)
            {
                position.x += SceneSize.width / 2;
            }
            else
            {
                position.x = xOrigin;
                position.y += YSizeRxMonitorPacketChartFull + YSizeLine;
            }
            ++dataSetCount;
        }
    }
}

PackedButton SceneRxPacketHistory::Control(const PackedButton& button) NN_NOEXCEPT
{
    const nn::hid::NpadButtonSet LockButtons = nn::hid::NpadButton::Plus::Mask | nn::hid::NpadButton::Minus::Mask;

    if ((button.longTrigger & LockButtons).IsAnyOn())
    {
        m_IsMenuLocked = !m_IsMenuLocked;
    }
    if (m_IsMenuLocked)
    {
        auto triggerWithRepeat = button.trigger | button.repeat;
        if (triggerWithRepeat.Test<nn::hid::NpadButton::Up>())
        {
            m_LineChartPlr.Decrement(LineChart::UiParameterType_YScale);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::Down>())
        {
            m_LineChartPlr.Increment(LineChart::UiParameterType_YScale);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::Left>())
        {
            m_LineChartPlr.Decrement(LineChart::UiParameterType_XSampleCount);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::Right>())
        {
            m_LineChartPlr.Increment(LineChart::UiParameterType_XSampleCount);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::X>())
        {
            m_LineChartPlr.Increment(LineChart::UiParameterType_YShift);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::B>())
        {
            m_LineChartPlr.Decrement(LineChart::UiParameterType_YShift);
        }
        if (triggerWithRepeat.Test<nn::hid::NpadButton::L>() ||
            triggerWithRepeat.Test<nn::hid::NpadButton::R>())
        {
            m_pControllers->ResetSelectedControllerFifo();
        }

        nn::hid::NpadButtonSet value;
        value.Reset();
        return button & value;
    }

    return button & ~LockButtons;
}
