﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "PlayReportUsageMonitor.h"
#include "ScenePlayReportUsage.h"
#include "ScreenStatics.h"
#include "ScreenUtil.h"
#include "s2d/Simple2D.h"

nn::Bit8 GetIndicatorPattern(int controllerNumber)
{
    switch (controllerNumber)
    {
    case 1:
        return 0x01;
    case 2:
        return 0x03;
    case 3:
        return 0x07;
    case 4:
        return 0x0f;
    case 5:
        return 0x09;
    case 6:
        return 0x05;
    case 7:
        return 0x0e;
    case 8:
        return 0x06;
    default:
        return 0x00;
    }
}

nn::hid::system::DeviceTypeSet ConvertDeviceType(const nn::hid::system::PlayReportDeviceType& usageDeviceType)
{
    switch (usageDeviceType)
    {
    case nn::hid::system::PlayReportDeviceType_JoyConLeft:
        return nn::hid::system::NpadDeviceType::JoyConLeft::Mask;
    case nn::hid::system::PlayReportDeviceType_JoyConRight:
        return nn::hid::system::NpadDeviceType::JoyConRight::Mask;
    case nn::hid::system::PlayReportDeviceType_SwitchProController:
        return nn::hid::system::NpadDeviceType::SwitchProController::Mask;
    case nn::hid::system::PlayReportDeviceType_UsbController:
        return TestDeviceType::UsbCompatibleController::Mask;
    default:
        return nn::hid::system::NpadDeviceType::Unknown::Mask;
    }
}

void DrawPlayReportStyle(RecordablePosition& position, const nn::hid::system::PlayReportPlayStyle& style)
{
    s2d::DrawText(position.Get(), "(%02x)", style);
    position.x += XSizeIndex2Digit;
    s2d::SetTextColor(ColorSceneTextStrong01);
    switch (style)
    {
    case nn::hid::system::PlayReportPlayStyle_Handheld:
        s2d::DrawText(position.Get(), "Handheld");
        break;
    case nn::hid::system::PlayReportPlayStyle_JoyConDual:
        s2d::DrawText(position.Get(), "Joy-Con Dual");
        break;
    case nn::hid::system::PlayReportPlayStyle_JoyConLeftHorizontal:
        s2d::DrawText(position.Get(), "Joy-Con (L) Hor");
        break;
    case nn::hid::system::PlayReportPlayStyle_JoyConLeftVertical:
        s2d::DrawText(position.Get(), "Joy-Con (L) Ver");
        break;
    case nn::hid::system::PlayReportPlayStyle_JoyConRightHorizontal:
        s2d::DrawText(position.Get(), "Joy-Con (R) Hor");
        break;
    case nn::hid::system::PlayReportPlayStyle_JoyConRightVertical:
        s2d::DrawText(position.Get(), "Joy-Con (R) Ver");
        break;
    case nn::hid::system::PlayReportPlayStyle_SwitchProController:
        s2d::DrawText(position.Get(), "Pro Controller");
        break;
    default:
        s2d::DrawText(position.Get(), "Unknown");
        break;
    }
    s2d::SetTextColor(ColorSceneTextDefault);
    position.x += XSizeDeviceType + XSizePadding * 3;
}

void DrawPlayReportSubXcd(RecordablePosition& position, const uint8_t& subType, const uint8_t (version)[4], const nn::hid::system::InterfaceType& interfaceType)
{
    position.x += PrintAndStep(position.Get(), "Sub:", true);
    s2d::DrawText(position.Get(), "%02x", subType);
    position.x += XSizeSubType;
    position.x += PrintAndStep(position.Get(), "Interface:", true);
    DrawInterface(position, interfaceType);
    s2d::DrawText(position.Get(), "%02x.%02x.%02x.%02x", version[0], version[1], version[2], version[3]);
    position.x += XSizeVersion;
}

void DrawPlayReportSubUsb(RecordablePosition& position, const uint16_t& vid, const uint16_t& pid)
{
    position.x += PrintAndStep(position.Get(), "Vid:", true);
    s2d::DrawText(position.Get(), "%04x", vid);
    position.x += XSizeVidPid;
    position.x += PrintAndStep(position.Get(), "Pid:", true);
    s2d::DrawText(position.Get(), "%04x", pid);
    position.x += XSizeVidPid;
}

void ScenePlayReportUsage::DrawImpl(RecordablePosition& position) NN_NOEXCEPT
{
    const auto& usages = PlayReportUsageMonitor::GetInstance().GetUsages();

    for (auto it = usages.begin();
        it != usages.end();
        ++it)
    {
        position.x = PositionSceneText.x;
        s2d::DrawText(position.Get(), "[%04d/%02d/%02d %02d:%02d:%02d]",
            (*it).time.year, (*it).time.month, (*it).time.day, (*it).time.hour, (*it).time.minute, (*it).time.second);

        if ((*it).count == 0)
        {
            position.x = PositionSceneText.x + XSizeTime;
            s2d::DrawText(position.Get(), "Controller Not In Use");
            position.NewLine();
        }
        else
        {
            for (int i = 0; i < (*it).count; ++i)
            {
                position.x = PositionSceneText.x + XSizeTime;
                auto& usage = (*it).usages[i];
                s2d::DrawText(position.Get(), "%2d:", i + 1);
                position.x += XSizeIndex2Digit;
                DrawIndicator(position, GetIndicatorPattern(usage.controllerNumber));
                s2d::DrawText(position.Get(), "(%02x)", usage.deviceType);
                position.x += XSizeIndex2Digit;
                DrawDeviceType(position, ConvertDeviceType(usage.deviceType));
                DrawPlayReportStyle(position, usage.style);
                position.NewLine();
                position.x = PositionSceneText.x + XSizeTime + XSizeIndex2Digit;
                if (usage.deviceType == nn::hid::system::PlayReportDeviceType_UsbController)
                {
                    uint16_t vid = usage.usbDevice.vidHigh << 8 | usage.usbDevice.vidLow;
                    uint16_t pid = usage.usbDevice.pidHigh << 8 | usage.usbDevice.pidLow;
                    DrawPlayReportSubUsb(position, vid, pid);
                }
                else
                {
                    DrawPlayReportSubXcd(position, usage.nxControllerInfo.subType, usage.nxControllerInfo.version, usage.nxControllerInfo.interfaceType);
                }
                position.NewLine();
            }
        }
        position.x = PositionSceneText.x;
        s2d::Point2D startPoint = position.Get();
        startPoint.x += XSizeControllerBorderIndent;
        startPoint.y += YSizeLine / 2;
        s2d::Point2D endPoint = position.Get();
        endPoint.x += XSizeControllerBorderIndent + SizeControllerBorder;
        endPoint.y = startPoint.y;
        s2d::DrawLine(startPoint, endPoint, ColorSceneBorder, WidthBorderDefault);
        position.y += YSizeLine;
    }
}
