﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>
#include "Scene.h"

namespace
{
const nn::hid::NpadButtonSet ScrollButton =
    nn::hid::NpadButton::StickRUp::Mask |
    nn::hid::NpadButton::StickRDown::Mask;

const nn::hid::NpadButtonSet MenuSelectionButton =
    nn::hid::NpadButton::L::Mask |
    nn::hid::NpadButton::R::Mask |
    nn::hid::NpadButton::ZL::Mask |
    nn::hid::NpadButton::ZR::Mask;
}

void SceneManager::Add(IScene* pScene) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pScene);
    m_pScenes.push_back(pScene);
}

void SceneManager::Draw() NN_NOEXCEPT
{
    this->GetCurrentScene()->Draw();
}

void SceneManager::DrawHeader() NN_NOEXCEPT
{
    this->GetCurrentScene()->DrawHeader();
}

PackedButton SceneManager::Control(const PackedButton& button) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(m_pScenes.size(), 0);

    if (button.hold.Test<nn::hid::NpadButton::StickRUp>())
    {
        this->GetCurrentScene()->Scroll(false);
    }
    if (button.hold.Test<nn::hid::NpadButton::StickRDown>())
    {
        this->GetCurrentScene()->Scroll(true);
    }

    auto maskedButton = this->GetCurrentScene()->Control(button & ~ScrollButton);

    if (maskedButton.trigger.Test<nn::hid::NpadButton::L>() ||
        maskedButton.trigger.Test<nn::hid::NpadButton::ZL>())
    {
        --m_SelectedScene;
        if (m_SelectedScene < 0)
        {
            m_SelectedScene = m_pScenes.size() - 1;
        }
        this->GetCurrentScene()->OnSceneFocused();
    }
    if (maskedButton.trigger.Test<nn::hid::NpadButton::R>() ||
        maskedButton.trigger.Test<nn::hid::NpadButton::ZR>())
    {
        ++m_SelectedScene;
        if (m_SelectedScene >= m_pScenes.size())
        {
            m_SelectedScene = 0;
        }
        this->GetCurrentScene()->OnSceneFocused();
    }

    return maskedButton & ~MenuSelectionButton;
}

IScene* SceneManager::GetCurrentScene() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(m_pScenes.size(), 0);
    return m_pScenes.at(m_SelectedScene);
}

