﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cmath>
#include <string>

#include "PacketChart.h"
#include "ScreenStatics.h"
#include "s2d/Simple2D.h"

namespace {

template <typename T>
inline T Clamp(const T& value, const T& maxRange, const T& minRange) NN_NOEXCEPT
{
    if (value > maxRange)
    {
        return maxRange;
    }
    if (value < minRange)
    {
        return minRange;
    }
    return value;
}

} // namespace

void PacketChart::Draw(int sampleCount,
                 s2d::Point2D position,
                 s2d::Size    size,
                 IPacketChartDataSetAccessor* pDataSet) NN_NOEXCEPT
{
    m_SampleCountMax = sampleCount;
    auto graphPosition = position;
    DrawFrame(graphPosition, size);
    DrawSamples(graphPosition, size, pDataSet);
    DrawValue(graphPosition, size, pDataSet);
}

void PacketChart::DrawFrame(s2d::Point2D position,
                      s2d::Size    size) NN_NOEXCEPT
{
    position.y += YSizeLineSmall;
    s2d::Point2D lineStart = { { position.x, position.y + size.height / 2 } };
    s2d::Point2D lineEnd   = { { position.x + size.width, position.y + size.height / 2 } };
    s2d::DrawLine(lineStart, lineEnd, ColorSceneBorderLight, WidthBorderDefault);
}

void PacketChart::DrawSamples(s2d::Point2D position,
                        s2d::Size    size,
                        IPacketChartDataSetAccessor* pDataSet) NN_NOEXCEPT
{
    int sampleCount = std::min(static_cast<int>(pDataSet->size()), m_SampleCountMax);
    float widthPerSample = (size.width / m_SampleCountMax);
    float xPosition = position.x + size.width * (static_cast<float>(m_SampleCountMax - sampleCount) / (m_SampleCountMax));
    float yBasePosition = position.y + size.height / 2 + YSizeLineSmall;
    s2d::SetTextScale(TextScaleSmall);
    if (sampleCount > 1)
    {
        int startingIndex = (pDataSet->size() > sampleCount) ?
                             pDataSet->size() - sampleCount :
                             0;
        for (int sampleIndex = 0; sampleIndex < sampleCount; ++sampleIndex)
        {
            s2d::Rectangle rectangle = { { xPosition, yBasePosition, widthPerSample * 0.6f, size.height * 0.4f } };
            s2d::Color color;
            if (pDataSet->at(startingIndex + sampleIndex))
            {
                rectangle.y -= size.height * 0.5f;
                color = ColorSceneTextStrong01;
             }
            else
            {
                rectangle.y += size.height * 0.1f;
                color = ColorSceneTextStrong03;
            }
            s2d::FillRectangle(rectangle, color);
            xPosition += widthPerSample;
        }
    }
}

void PacketChart::DrawValue(s2d::Point2D position,
                      s2d::Size    size,
                      IPacketChartDataSetAccessor* pDataSet) NN_NOEXCEPT
{
    s2d::SetTextScale(TextScaleSmall);
    s2d::SetTextColor(pDataSet->color());
    s2d::DrawText(position, "■");
    s2d::SetTextColor(ColorSceneTextDefault);
    s2d::DrawText({ { position.x + XSizeLineChartColorIndex, position.y} }, "%s", pDataSet->label());
    position.x += XSizePacketChartValue;
    char outBuffer[256];
    s2d::DrawText(position, "drop: %s", pDataSet->continuousDrop(outBuffer));
    position.x += XSizePacketChartValue;
    s2d::DrawText(position, "max: %s", pDataSet->maxDrop(outBuffer));
    position.x += XSizePacketChartValue;
    s2d::SetTextScale(TextScaleDefault);

}
