﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <vector>

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/time/time_Api.h>

#include "NpadController.h"
#include "EventMonitor.h"
#include "LogPrint.h"
#include "Menu.h"
#include "PlayReportUsageMonitor.h"
#include "Scene.h"
#include "SceneAnalogStick.h"
#include "SceneAnalogStickDriver.h"
#include "SceneControllerStatus.h"
#include "SceneLog.h"
#include "ScenePlayReportUsage.h"
#include "SceneRegisteredDevice.h"
#include "SceneRxPacketHistory.h"
#include "SceneSixAxisSensor.h"
#include "SceneSixAxisSensorDriver.h"
#include "ScreenStatics.h"
#include "StartupArgument.h"
#include "UniquePadController.h"
#include "s2d/Simple2D.h"

namespace {
NpadControllerList g_Controllers;
UniquePadList g_UniquePads;
SceneManager g_SceneManager;

void Initialize() NN_NOEXCEPT;
void Update() NN_NOEXCEPT;
void Proceed() NN_NOEXCEPT;

} // namespace

extern "C" void nnMain()
{
    s2d::Initialize();
    s2d::SetClearColor(ColorSceneBackground);

    // TIME ライブラリの初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Initialize() );

    Initialize();
    bool runs = true;
    while (NN_STATIC_CONDITION(runs))
    {
        s2d::BeginRender();

        Update();

        Proceed();

        g_SceneManager.Draw();
        DrawMenu();
        g_SceneManager.DrawHeader();

        s2d::EndRender();
    }
}

namespace {

void Initialize() NN_NOEXCEPT
{
    nn::hid::InitializeNpad();
    //使用する操作形態を設定
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(NpadIds, sizeof(NpadIds) / sizeof(NpadIds[0]));

    g_Controllers.Initialize();
    g_UniquePads.Initialize();
    StartupArgument();

    InitializeEventMonitor(g_Controllers);
    InitializeMenu();
    PlayReportUsageMonitor::GetInstance().Update();

    // Scene のセットアップ
    auto pControllerStatus = new SceneControllerStatus();
    pControllerStatus->SetNpadControllerList(&g_Controllers);
    g_SceneManager.Add(pControllerStatus);

    auto pSceneAnalogStick = new SceneAnalogStick();
    pSceneAnalogStick->SetNpadControllerList(&g_Controllers);
    g_SceneManager.Add(pSceneAnalogStick);

    auto pSceneAnalogStickDriver = new SceneAnalogStickDriver();
    pSceneAnalogStickDriver->SetUniquePadList(&g_UniquePads);
    g_SceneManager.Add(pSceneAnalogStickDriver);

    auto pSceneSensor = new SceneSixAxisSensor();
    pSceneSensor->SetNpadControllerList(&g_Controllers);
    g_SceneManager.Add(pSceneSensor);

    auto pSceneSensorDriver = new SceneSixAxisSensorDriver();
    pSceneSensorDriver->SetUniquePadList(&g_UniquePads);
    g_SceneManager.Add(pSceneSensorDriver);

    auto pSceneRxPacketHistory = new SceneRxPacketHistory();
    pSceneRxPacketHistory->SetUniquePadList(&g_UniquePads);
    g_SceneManager.Add(pSceneRxPacketHistory);

    g_SceneManager.Add(new ScenePlayReportUsage());
    g_SceneManager.Add(new SceneRegisteredDevice());
    g_SceneManager.Add(new SceneLog());
}


void Update() NN_NOEXCEPT
{
    g_Controllers.Update();
    g_UniquePads.Update();
}

void Proceed() NN_NOEXCEPT
{
    auto butons = g_SceneManager.Control(g_Controllers.GetControlButtons());
    ControlMenu(butons);
}
} // namespace
