﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cmath>
#include <nn/nn_Assert.h>
#include <nn/hid/hid_AnalogStickState.h>
#include <nn/hid/detail/hid_PadTypes.h>

#include "RingBuffer.h"

const int AnalogStickSampleCountMax = 800;

inline nn::hid::AnalogStickState ConvertAnalogStickState(const nn::hid::detail::AnalogStickDriverState& input) NN_NOEXCEPT
{
    return { input.x, input.y };
}

inline nn::hid::AnalogStickState operator - (const nn::hid::AnalogStickState& input1, const nn::hid::AnalogStickState& input2) NN_NOEXCEPT
{
    return { input1.x - input2.x, input1.y - input2.y };
}

//!< アナログスティックの詳細な入力状態
struct AnalogStickStateEx
{
    nn::hid::AnalogStickState stick;
    float   length;
    float   angle;
};

//!< アナログスティックの入力状態を蓄積する FIFO
class AnalogStickStateExFifo : public RingBuffer<AnalogStickStateEx, 800>
{
private:
    AnalogStickStateEx m_Max;
    AnalogStickStateEx m_Min;
    nn::hid::AnalogStickState m_Center;
    bool m_IsMaxMinSet;

public:
    void Initialize(const nn::hid::AnalogStickState& center) NN_NOEXCEPT
    {
        this->Clear();
        m_Center = center;
    }

    void AddSample(const nn::hid::AnalogStickState& input) NN_NOEXCEPT
    {
        auto shifted = input - m_Center;
        AnalogStickStateEx state = { shifted,  std::sqrt(static_cast<float>(shifted.x * shifted.x + shifted.y * shifted.y)), std::atan2(static_cast<float>(shifted.x), static_cast<float>(shifted.y)) * 180 / nn::util::FloatPi };

        this->push_back(state);
        if (m_IsMaxMinSet == false)
        {
            m_Max = state;
            m_Min = state;
            m_IsMaxMinSet = true;
        }

        m_Max.stick.x = std::max(m_Max.stick.x, state.stick.x);
        m_Max.stick.y = std::max(m_Max.stick.y, state.stick.y);
        m_Max.length = std::max(m_Max.length, state.length);
        m_Min.stick.x = std::min(m_Min.stick.x, state.stick.x);
        m_Min.stick.y = std::min(m_Min.stick.y, state.stick.y);
        m_Min.length = std::min(m_Min.length, state.length);
    }

    void Clear() NN_NOEXCEPT
    {
        m_Max.stick = nn::hid::AnalogStickState();
        m_Max.length = 0.0f;
        m_Min.stick = nn::hid::AnalogStickState();
        m_Min.length = 0.0f;
        m_IsMaxMinSet = false;
        this->clear();
    }

    AnalogStickStateEx GetMax() NN_NOEXCEPT
    {
        return m_Max;
    }

    AnalogStickStateEx GetMin() NN_NOEXCEPT
    {
        return m_Min;
    }

    nn::hid::AnalogStickState GetCenter() NN_NOEXCEPT
    {
        return m_Center;
    }
};
