﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <mutex>
#include <memory>

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/mem.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/hid/system/hid_FirmwareUpdate.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/result/result_HandlingUtility.h>

#include "FirmwareUpdate_Util.h"

namespace
{

// サポートする UniquePadType
const nn::hid::system::UniquePadType SupportedDeviceTypes[] =
{
    nn::hid::system::UniquePadType_FullKeyController,
    nn::hid::system::UniquePadType_RightController,
    nn::hid::system::UniquePadType_LeftController
};

// サポートする UniquePadInterface
const nn::hid::system::UniquePadInterface SupportedInterfaces[] =
{
    nn::hid::system::UniquePadInterface_Rail,
    nn::hid::system::UniquePadInterface_Bluetooth
};

// サポートしているデバイスタイプか判定
bool IsSupportedDeviceType(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT
{
    auto deviceType = nn::hid::system::GetUniquePadType(id);
    for (auto deviceTypeToTest : SupportedDeviceTypes)
    {
        if (deviceType == deviceTypeToTest)
        {
            return true;
        }
    }

    return false;
}

// サポートしている接続インターフェースか判定
bool IsSupportedInterface(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT
{
    nn::hid::system::UniquePadInterface padInterface;
    if (nn::hid::system::GetUniquePadInterface(&padInterface, id).IsFailure())
    {
        return false;
    }

    for (auto interfaceToTest : SupportedInterfaces)
    {
        if (padInterface == interfaceToTest)
        {
            return true;
        }
    }

    return false;
}

}  // anonymous

bool IsSupportedDevice(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT
{
    return IsSupportedDeviceType(id) && IsSupportedInterface(id);
}

const char* GetUniquePadTypeName(nn::hid::system::UniquePadType padType) NN_NOEXCEPT
{
    switch (padType)
    {
    case nn::hid::system::UniquePadType_Embedded:
        return "Embedded";

    case nn::hid::system::UniquePadType_FullKeyController:
        return "FullKey";

    case nn::hid::system::UniquePadType_LeftController:
        return "Left";

    case nn::hid::system::UniquePadType_RightController:
        return "Right";

    default:
        return "???";
    }
}

const char* GetUniquePadInterfaceName(nn::hid::system::UniquePadInterface interface) NN_NOEXCEPT
{
    switch (interface)
    {
    case nn::hid::system::UniquePadInterface_Embedded:
        return "Embedded";

    case nn::hid::system::UniquePadInterface_Rail:
        return "Rail";

    case nn::hid::system::UniquePadInterface_Bluetooth:
        return "Bluetooth";

    case nn::hid::system::UniquePadInterface_Usb:
        return "USB";

    default:
        return "???";
    }
}

const char* GetUpdateRequiredReasonString(
    nn::Result result,
    nn::hid::system::FirmwareUpdateRequiredReason reason) NN_NOEXCEPT
{
    if (nn::hid::system::ResultFirmwareUpdateHardwareError::Includes(result))
    {
        return "HW Error";
    }
    else if (result.IsFailure())
    {
        return "-";
    }

    switch (reason)
    {
    case nn::hid::system::FirmwareUpdateRequiredReason_Nothing:
        return "Nothing";

    case nn::hid::system::FirmwareUpdateRequiredReason_Corrupted:
        return "Corrupted";

    case nn::hid::system::FirmwareUpdateRequiredReason_Hotfix:
        return "Hotfix";

    default:
        return "???";
    }
}

const char* GetUpdateStageString(nn::hid::system::FirmwareUpdateStage stage) NN_NOEXCEPT
{
    switch (stage)
    {
    case nn::hid::system::FirmwareUpdateStage_Preparing:
        return "Preparing...";

    case nn::hid::system::FirmwareUpdateStage_Updating:
        return "Updating firmware";

    case nn::hid::system::FirmwareUpdateStage_Finalizing:
        return "Finalizing...";

    case nn::hid::system::FirmwareUpdateStage_Completed:
        return "Complete!";

    default:
        return "???";
    }
}

void FirmwareVersionReader::SetUniquePadId(const nn::hid::system::UniquePadId& id) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_UniquePadId = id;
    m_IsValid     = true;
    InvalidateVersion();
}

void FirmwareVersionReader::Invalidate() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_IsValid = false;
    InvalidateVersion();
}

bool FirmwareVersionReader::HasValidVersion() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    return m_HasValidVersion;
}

bool FirmwareVersionReader::Get(nn::hid::system::FirmwareVersion* pOutVersion) const NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutVersion);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (m_IsValid && m_HasValidVersion)
    {
        *pOutVersion = m_Version;
        return true;
    }

    return false;
}

void FirmwareVersionReader::Update() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (!m_IsValid)
    {
        return;
    }

    if (m_HasValidVersion)
    {
        return;
    }

    if (nn::hid::system::GetFirmwareVersion(
        &m_Version,
        m_UniquePadId).IsSuccess())
    {
        m_HasValidVersion = true;
    }
}

void FirmwareVersionReader::InvalidateVersion() NN_NOEXCEPT
{
    m_HasValidVersion = false;
}

