﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>

#include <nn/hid/hid_DebugPad.h>

namespace {

    const int DebugPadStateCountMax = 8;

    const int DebugPadButtons[] = {
        nn::hid::DebugPadButton::Up::Index,
        nn::hid::DebugPadButton::Left::Index,
        nn::hid::DebugPadButton::ZR::Index,
        nn::hid::DebugPadButton::X::Index,
        nn::hid::DebugPadButton::A::Index,
        nn::hid::DebugPadButton::Y::Index,
        nn::hid::DebugPadButton::B::Index,
        nn::hid::DebugPadButton::ZL::Index,
        nn::hid::DebugPadButton::R::Index,
        nn::hid::DebugPadButton::Start::Index,
        nn::hid::DebugPadButton::Select::Index,
        nn::hid::DebugPadButton::L::Index,
        nn::hid::DebugPadButton::Down::Index,
        nn::hid::DebugPadButton::Right::Index
    };

    const int DebugPadButtonsCount = sizeof(DebugPadButtons) / sizeof(DebugPadButtons[0]);

    const char* GetStringOfDebugPadButton(int button)
    {
        switch (button)
        {
        case nn::hid::DebugPadButton::Up::Index:
            return "Up";

        case nn::hid::DebugPadButton::Left::Index:
            return "Left";

        case nn::hid::DebugPadButton::ZR::Index:
            return "ZR";

        case nn::hid::DebugPadButton::X::Index:
            return "X";

        case nn::hid::DebugPadButton::A::Index:
            return "A";

        case nn::hid::DebugPadButton::Y::Index:
            return "Y";

        case nn::hid::DebugPadButton::B::Index:
            return "B";

        case nn::hid::DebugPadButton::ZL::Index:
            return "ZL";

        case nn::hid::DebugPadButton::R::Index:
            return "R";

        case nn::hid::DebugPadButton::Start::Index:
            return "Start";

        case nn::hid::DebugPadButton::Select::Index:
            return "Select";

        case nn::hid::DebugPadButton::L::Index:
            return "L";

        case nn::hid::DebugPadButton::Down::Index:
            return "Down";

        case nn::hid::DebugPadButton::Right::Index:
            return "Right";

        default:
            return "Unknown";
        }
    }

    bool DetectsInput(const nn::hid::DebugPadState& state)
    {
        for (int i = 0; i < DebugPadButtonsCount; ++i)
        {
            if (state.buttons.Test(DebugPadButtons[i]))
            {
                return true;
            }
        }

        return false;
    }

    bool DetectsInput(const nn::hid::DebugPadState states[],
                      int                            stateCount)
    {
        NN_ASSERT_NOT_NULL(states);
        NN_ASSERT_RANGE(stateCount, 0, DebugPadStateCountMax + 1);

        for (int i = 0; i < stateCount; ++i)
        {
            if (DetectsInput(states[i]))
            {
                return true;
            }
        }

        return false;
    }

    void PrintDebugPadState(const nn::hid::DebugPadState& state)
    {
        int buttonCount = 0;

        NN_LOG("    Sampling No: %d\n", state.samplingNumber);
        NN_LOG("    Buttons: ");
        for (int i = 0; i < DebugPadButtonsCount; ++i )
        {
            if (state.buttons.Test(DebugPadButtons[i]))
            {
                if (buttonCount > 0)
                {
                    NN_LOG(",");
                }
                NN_LOG(GetStringOfDebugPadButton(DebugPadButtons[i]));
                buttonCount += 1;
            }
        }

        if (buttonCount == 0)
        {
            NN_LOG("None");
        }

        NN_LOG("\n");
        NN_LOG("    AnalogStickR.X: %d\n", state.analogStickR.x);
        NN_LOG("    AnalogStickR.Y: %d\n", state.analogStickR.y);
        NN_LOG("    AnalogStickL.X: %d\n", state.analogStickL.x);
        NN_LOG("    AnalogStickL.Y: %d\n", state.analogStickL.y);
    }

    void PrintDebugPadState(const nn::hid::DebugPadState states[],
                              int                            stateCount)
    {
        NN_ASSERT_NOT_NULL(states);
        NN_ASSERT_RANGE(stateCount, 0, DebugPadStateCountMax + 1);

        for (int i = 0; i < stateCount; ++i)
        {
            NN_LOG("    # %d/%d\n", i + 1, stateCount);
            PrintDebugPadState(states[i]);
        }
    }

    void ExecuteReadSample()
    {
        NN_LOG("*** Read Sample ***\n");
        NN_LOG("Input:\n");

        int64_t indices = 0;

        bool    waits = true;

        while (waits)
        {
            int stateCount = 0;
            nn::hid::DebugPadState states[DebugPadStateCountMax];
            stateCount = nn::hid::GetDebugPadStates(
                          &states[0],
                          DebugPadStateCountMax);

            // 取得済みの状態は無視
            while (0 < stateCount)
            {
                if (states[stateCount - 1].samplingNumber < indices)
                {
                    --stateCount;

                    continue;
                }

                indices = states[0].samplingNumber + 1;

                break;
            }

            // 入力が検出された場合のみ結果を出力
            if (DetectsInput(states, stateCount))
            {
                PrintDebugPadState(states, stateCount);

                // ＋ ボタンが押されたら終了
                for (int j = 0; j < stateCount; ++j)
                {
                    nn::hid::DebugPadButtonSet& buttonState = states[j].buttons;
                    if (buttonState.Test(nn::hid::DebugPadButton::Start::Index))
                    {
                        waits = false;
                    }
                }
            }
        }

        NN_LOG("done.\n");
    }

    void ExecuteReadLatestSample()
    {
        NN_LOG("*** ReadLatest Sample ***\n");
        NN_LOG("Input:\n");

        bool waits = true;

        while (waits)
        {
            nn::hid::DebugPadState state;
            nn::hid::GetDebugPadState(&state);

            // 入力が検出された場合のみ結果を出力
            if (DetectsInput(state))
            {
                PrintDebugPadState(state);

                // － ボタンが押されたら終了
                nn::hid::DebugPadButtonSet& buttonState = state.buttons;
                if (buttonState.Test(nn::hid::DebugPadButton::Select::Index))
                {
                    waits = false;
                }
            }
        }

        NN_LOG("done.\n");
    }
}

extern "C" void nnMain()
{
    NN_LOG("DebugPad Sample starts.\n");
    NN_LOG("\n");

    // デバッグパッドの初期化
    nn::hid::InitializeDebugPad();

    NN_LOG("\n");

    // nn::hid::GetDebugPadStates による状態取得
    ExecuteReadSample();

    NN_LOG("\n");

    // nn::hid::GetDebugPadState による状態取得
    ExecuteReadLatestSample();
}
