﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @brief
    メニューを描画し、メニュー選択制御を行います。
*/
#include "Main_Menu.h"

namespace main{

    // カーソル番号
    int CursolPosition = 0;
    int DesidePosition = 0;

    int CursolWaitCount = 0;
    // 通常メニューリスト(動的配列：外部から取得する想定のメニューリスト)
    std::string* MenuList;
    // 通常メニューリストの数(外部から取得する想定)
    int MenuNum;
    // レガシーメニューリスト(動的配列：外部から取得する想定のメニューリスト)
    std::string* CommonMenuList;
    // レガシーメニューリストの数(外部から取得する想定)
    int CommonMenuNum;

    // 描画を滑らかにする為のカウンタ
    int smoothcnt = 0;
    const int smoothcntNum = 20;

    int32_t TouchPanelPosition;

    Menu::Menustate menuSt;

    // 階層構造になった色々なメニューを表示させることを想定し、メニューリストを外部から取得できるようにします
    void Menu::SetList(std::string* menulist, int menunum, std::string* cmnmenulist, int cmnmenunum) NN_NOEXCEPT
    {
        // メニューリストの動的配列生成
        MenuList = new std::string[menunum];
        CommonMenuList = new std::string[cmnmenunum];

        MenuNum = menunum;
        CommonMenuNum = cmnmenunum;

        // 引数で取得したメニューリストをコピー
        // メニューの数だけ実施
        for (int i = 0; i < MenuNum; i++)
        {
            MenuList[i] = menulist[i];
        }
        for (int i = 0; i < CommonMenuNum; i++)
        {
            CommonMenuList[i] = cmnmenulist[i];
        }
    }

    // 取得したメニューリストを解放します。SetList()で設定したメニューリストは本関数で確実に解放してください。
    void Menu::ResetList() NN_NOEXCEPT
    {
        delete[] MenuList;
        delete[] CommonMenuList;

        MenuNum = 0;
        CommonMenuNum = 0;
    }

    // 現在選択されているメニューを判定します。
    // 判定結果はメニュー番号という形で返り値として出力します。
    // メニュー番号はMenuList[]のリストに対応し、上から0,1,2,・・となります。
    int Menu::Select(frm::hid::NPadState m_BtnSt) NN_NOEXCEPT
    {

        (smoothcnt > 0) ? smoothcnt -= 2 : (smoothcnt < 0 ? smoothcnt += 2 : smoothcnt = 0);


        if (m_BtnSt.trg.Test<nn::hid::NpadButton::Down>()
         || m_BtnSt.trg.Test<nn::hid::NpadButton::StickLDown>()
         || m_BtnSt.beat.Test<nn::hid::NpadButton::Down>()
         || m_BtnSt.beat.Test<nn::hid::NpadButton::StickLDown>()
         || m_BtnSt.trg.Test<nn::hid::NpadButton::StickRDown>()
         || m_BtnSt.beat.Test<nn::hid::NpadButton::StickRDown>()
       ){
            CursolPosition++;
            smoothcnt = smoothcntNum;

            if (CursolPosition > MenuNum - 1)
            {
                CursolPosition = 0;
            }
        }
        else if (m_BtnSt.trg.Test<nn::hid::NpadButton::Up>()
              || m_BtnSt.trg.Test<nn::hid::NpadButton::StickLUp>()
              || m_BtnSt.beat.Test<nn::hid::NpadButton::Up>()
              || m_BtnSt.beat.Test<nn::hid::NpadButton::StickLUp>()
              || m_BtnSt.trg.Test<nn::hid::NpadButton::StickRUp>()
              || m_BtnSt.beat.Test<nn::hid::NpadButton::StickRUp>()
       ){
            CursolPosition--;
            smoothcnt = -smoothcntNum;

            if (CursolPosition < 0)
            {
                CursolPosition = MenuNum - 1;
            }
        }

        if (CursolWaitCount == 0) {
            if (TouchPanelPosition < -30)
            {
                CursolPosition++;
                smoothcnt = smoothcntNum;
                CursolWaitCount = 15;

                if (CursolPosition > MenuNum - 1)
                {
                    CursolPosition = 0;
                }
            }
            else if (TouchPanelPosition > 30)
            {
                CursolPosition--;
                smoothcnt = -smoothcntNum;
                CursolWaitCount = 15;

                if (CursolPosition < 0)
                {
                    CursolPosition = MenuNum - 1;
                }
            }
        }
        else {
            CursolWaitCount--;
        }

        return CursolPosition;
    }

    int Menu::Deside(frm::hid::NPadState m_BtnSt) NN_NOEXCEPT
    {
        if (m_BtnSt.trg.Test<nn::hid::NpadButton::Right>()
            || m_BtnSt.trg.Test<nn::hid::NpadButton::StickLRight>()
            || m_BtnSt.beat.Test<nn::hid::NpadButton::Right>()
            || m_BtnSt.beat.Test<nn::hid::NpadButton::StickLRight>()
            || m_BtnSt.trg.Test<nn::hid::NpadButton::StickRRight>()
            || m_BtnSt.beat.Test<nn::hid::NpadButton::StickRRight>()
            ) {
            DesidePosition++;

            if (DesidePosition >  1)
            {
                DesidePosition = 0;
            }
        }
        else if (m_BtnSt.trg.Test<nn::hid::NpadButton::Left>()
                 || m_BtnSt.trg.Test<nn::hid::NpadButton::StickLLeft>()
                 || m_BtnSt.beat.Test<nn::hid::NpadButton::Left>()
                 || m_BtnSt.beat.Test<nn::hid::NpadButton::StickLLeft>()
                 || m_BtnSt.trg.Test<nn::hid::NpadButton::StickRLeft>()
                 || m_BtnSt.beat.Test<nn::hid::NpadButton::StickRLeft>()
                 ) {
            DesidePosition--;

            if (DesidePosition < 0)
            {
                DesidePosition = 1;
            }
        }

        return DesidePosition;
    }

    //!< レガシーメニューを描画します。
    //!< 引数で描画を開始する地点のX座標、Y座標を指定します。
    //!< レガシーメニューはカーソル選択できません。
    //!< 枠について、縦方向(Y)はメニューの数によって変化しますが、横方向(X)の終端は固定です
    void Menu::DrawLegacy(nn::gfx::util::DebugFontTextWriter* pTextWriter, GraphicsSystem* pGraphicsSystem, int ofsX, int ofsY) NN_NOEXCEPT
    {
        //////////////////////////////////////////
        // レガシーメニュータイトルの描画
        //////////////////////////////////////////
        pTextWriter->SetScale(1.0, 1.0);
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetCursor(ofsX, ofsY - 23);
        pTextWriter->Print("操作方法");

        //////////////////////////////////////////////////
        // レガシーメニュー
        //////////////////////////////////////////////////

        // メニューの数だけ実施
        for (int i = 0; i < CommonMenuNum; i++)
        {
            // 描画カラーの指定
            pTextWriter->SetTextColor(Color::Red);
            // 描画する
            pTextWriter->SetCursor(ofsX, ofsY + (i * 20));
            pTextWriter->Print("%s\n", CommonMenuList[i].c_str());
        }
        //////////////////////////////////////////
        // 枠の描画
        //////////////////////////////////////////
        DrawRect(pGraphicsSystem, ofsX, ofsY, CommonMenuNum + 1);
    }

    //!< メニューをリング形式で描画します。レガシーメニューはリング表示できません。
    //!< 引数で描画を開始する地点のX座標、Y座標を指定します。
    void Menu::DrawRing(nn::gfx::util::DebugFontTextWriter* pTextWriter, GraphicsSystem* pGraphicsSystem, int ofsX, int ofsY) NN_NOEXCEPT
    {
        // リング形式で表示する項目の一度に表示する数
        const int ViewMenuNum = 5;
        // 項目ごとのシフト位置
        int shiftPosY = 25;

        //////////////////////////////////////////
        // メニュータイトルの描画
        //////////////////////////////////////////
        pTextWriter->SetScale(1.0, 1.0);
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetCursor(ofsX - 5, ofsY - 25);
        pTextWriter->Print("MENU LIST");

        //////////////////////////////////////////////////
        // 通常メニュー
        //////////////////////////////////////////////////

        // 描画開始座標を選択
        pTextWriter->SetScale(1.2, 1.2);
        pTextWriter->SetCursor(ofsX, ofsY);

        // 現在選択されているカーソルから2つ前の項目から表示する
        int t_sel = CursolPosition - 2;
        if (t_sel < 0)
        {
            t_sel  = t_sel + MenuNum;
        }

        // 描画する数は5つで固定
        for (int i = 0; i < ViewMenuNum; i++)
        {
            // 描画カラーの指定(指定していないものは薄くする)
            if (i == 0 || i == 4)
            {
                pTextWriter->SetTextColor(ClearColor::White);
            }
            else if (i == 1 || i == 3)
            {
                pTextWriter->SetTextColor(HalfColor::White);
            }
            else
            {
                pTextWriter->SetTextColor(Color::White);
            }

            int posY = ofsY + (i * shiftPosY) + smoothcnt;

            if( (posY >= ofsY) && (posY <= ofsY + (4 * shiftPosY)) )
            {
                // インデックスを描画する
                pTextWriter->SetCursor(ofsX, posY);
                pTextWriter->Print("(%d)", (t_sel) + 1);
                // インデックス分右にずらす
                pTextWriter->SetCursor(ofsX + 50, posY);
                // メニュー内容を描画する
                pTextWriter->Print("%s\n", MenuList[t_sel].c_str());
            }
            t_sel++;

            if (t_sel > MenuNum - 1)
            {
                t_sel = 0;
            }
        }

        //////////////////////////////////////////
        // 枠の描画
        //////////////////////////////////////////
        DrawRect(pGraphicsSystem, ofsX, ofsY, ViewMenuNum);
    }

    void Menu::SetTouchPosition(int32_t TouchPos) NN_NOEXCEPT
    {
        TouchPanelPosition =  TouchPos;
    }

    Menu::Menustate Menu::GetMenuState(frm::hid::NPadState m_BtnSt) NN_NOEXCEPT
    {
        if (menuSt == Menustate_Do)
        {
            menuSt = Menustate_Ready;
        }
        else if (menuSt == Menustate_Set)
        {
            if (m_BtnSt.trg.Test<nn::hid::NpadButton::A>())
            {
                menuSt = Menustate_Ready;
            }
        }
        else if (menuSt == Menustate_Ready)
        {
            if (m_BtnSt.trg.Test<nn::hid::NpadButton::A>())
            {
                menuSt = Menustate_Do;
            }
            else if (m_BtnSt.trg.Test<nn::hid::NpadButton::B>())
            {
                menuSt = Menustate_Set;
            }
            else
            {
                menuSt = Menustate_Ready;
            }
        }
        else
        {
            menuSt = Menustate_Set;
        }
        return menuSt;
    }

    Menu::Menustate Menu::ForceSetMenuState(Menu::Menustate St) NN_NOEXCEPT
    {
        menuSt = St;
        return menuSt;
    }

    void Menu::DrawRect(GraphicsSystem* pGraphicsSystem, float ofsX, float ofsY, float menuNum) NN_NOEXCEPT
    {
        //////////////////////////////////////////
        // 枠の描画
        //////////////////////////////////////////

        // 終端位置の設定
        float ofsXEnd = 400.f;
        float ofsYEnd = menuNum * 26.5f;

        // レンダラの生成
        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        // 線の幅を指定
        pPrimitiveRenderer->SetLineWidth(1.f);
        // 描画カラーの指定
        pPrimitiveRenderer->SetColor(Color::White);
        // 枠を描画する
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DFrame(&pGraphicsSystem->GetCommandBuffer(), ofsX - 10, ofsY - 2, ofsXEnd, ofsYEnd);
        pPrimitiveRenderer->SetColor(ClearColor::Black);
        pGraphicsSystem->GetPrimitiveRenderer().Draw2DRect(&pGraphicsSystem->GetCommandBuffer(), ofsX - 10, ofsY - 2, ofsXEnd, ofsYEnd);


    }
}
// [EOF]

