﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <mutex>

#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>

#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/tmp/hid_ConsoleSixAxisSensor.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/util/util_FormatString.h>

#include "SixAxisSensorLogger.h"

const char* SixAxisSensorLogger::Format = "# TestId, SamplingNumber, DeltaTime, AccX, AccY, AccZ, GyroX, GyroY, GyroZ\n";
const char* SixAxisSensorLogger::MountName = "sd";

SixAxisSensorLogger::SixAxisSensorLogger() NN_NOEXCEPT
    : m_FileHandle()
    , m_FileOffset(0)
    , m_FileName()
    , m_IsSdCardInserted(false)
    , m_StartTick(nn::os::GetSystemTick())
    , m_LineQueue()
{
    // 何もしない
}

void SixAxisSensorLogger::Initialize(const char* pFileName) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pFileName);

    strncpy(m_FileName, MountName, FileNameCountMax);
    strncat(m_FileName, ":/", FileNameCountMax);
    strncat(m_FileName, pFileName, FileNameCountMax);

    nn::Result result;

    NN_LOG("Mount SD card as \"%s:/\"\n", MountName);

    // SD カードをマウント名 MountName でマウントします。
    result = nn::fs::MountSdCardForDebug(MountName);
    if (nn::fs::ResultSdCardAccessFailed::Includes(result))
    {
        // SD カードが挿入されていません。
        NN_LOG("SD card not found.");
        return;
    }

    // 上記以外の失敗は実装ミスのため、必ずアボートする
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // SD カードの挿入を確認
    m_IsSdCardInserted = true;

    // ファイルが存在しない場合は作成します。
    {
        nn::fs::DirectoryEntryType directoryEntryType;
        result = nn::fs::GetEntryType(&directoryEntryType, m_FileName);
        if (nn::fs::ResultPathNotFound().Includes(result))
        {
            // 対象ファイルが存在しません。
            // 対象ファイルを作成します。
            NN_LOG("Create \"%s\"\n", m_FileName);
            result = nn::fs::CreateFile(m_FileName, 0);
            if (nn::fs::ResultPathNotFound::Includes(result))
            {
                // 対象ファイルが存在しません。
                // 親ディレクトリが必ず存在する場合は、このエラーハンドリングは不要です。
            }
            else if (nn::fs::ResultPathAlreadyExists::Includes(result) ||
                     nn::fs::ResultTargetLocked::Includes(result))
            {
                // 対象ファイルが既に存在しています。
                // ファイルが既に存在していても構わない場合は、このエラーハンドリングは不要です。
                // エラーハンドリングしない場合、ファイルのサイズが FileSize である保証が無いことに注意してください。
                // 必要であれば対象ファイルを削除してから再度作成してください。
            }
            else if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
            {
                // 空き領域が不足しています。
                // SD カードの空き領域を確認してください。
                NN_ABORT("Usable space not enough.\n");
                return;
            }
            // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
            // これ以上のエラーハンドリングは不要です。
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。
    }

    // ファイル末尾に追記します。
    {
        NN_LOG("Open \"%s\"\n", m_FileName);
        result = nn::fs::OpenFile(&m_FileHandle,
                                  m_FileName,
                                  nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);

        if (nn::fs::ResultPathNotFound::Includes(result))
        {
            // 対象ファイルが存在しません。
            // 存在するファイルしか開かない場合は、このエラーハンドリングは不要です。
        }
        else if (nn::fs::ResultTargetLocked::Includes(result))
        {
            // 対象ファイルが既にオープンされています。
            // ファイルが既にオープンされている可能性が無い場合は、このエラーハンドリングは不要です。
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。

        // ファイルサイズの取得に失敗した場合はライブラリ内でアボートするため、エラーハンドリングは不要です。
        (void)nn::fs::GetFileSize(&m_FileOffset, m_FileHandle);
    }
}

void SixAxisSensorLogger::Finalize() NN_NOEXCEPT
{
    if (m_IsSdCardInserted == false)
    {
        // SD カードが挿入されていないので何もしません。
        // NN_LOG("Do nothing since sd card isn't inserted.\n");
        return;
    }

    NN_LOG("Close \"%s\"\n", m_FileName);
    nn::fs::CloseFile(m_FileHandle);

    // アンマウントします。
    NN_LOG("Unmount \"%s:/\"\n", MountName);
    nn::fs::Unmount(MountName);
}

void SixAxisSensorLogger::SetTick(const ::nn::os::Tick& tick) NN_NOEXCEPT
{
    m_StartTick = tick;
}

void SixAxisSensorLogger::AppendLineFormat() NN_NOEXCEPT
{
    char writeBuffer[1024] = {'\0'};

    int length = nn::util::SNPrintf(writeBuffer,
                                    sizeof(writeBuffer),
                                    Format);

    AppendLine(writeBuffer, length);
}

void SixAxisSensorLogger::AppendLine(const ::nn::hid::tmp::SixAxisSensorCountState& state, int testId) NN_NOEXCEPT
{
    char writeBuffer[1024] = { '\0' };
    auto pState = reinterpret_cast<const ::nn::hid::tmp::SixAxisSensorCountState*>(&state);

    int length = nn::util::SNPrintf(
        writeBuffer,
        sizeof(writeBuffer),
        "%d, %lld, %lld, %d, %d, %d, %d, %d, %d\n",
        testId,
        pState->samplingNumber,
        (nn::os::Tick(pState->tick) - m_StartTick).ToTimeSpan().GetMicroSeconds(),
        pState->acceleration.x,
        pState->acceleration.y,
        pState->acceleration.z,
        pState->angularVelocity.x,
        pState->angularVelocity.y,
        pState->angularVelocity.z
    );

    AppendLine(writeBuffer, length);
}

void SixAxisSensorLogger::AppendLine(const char* str, int length) NN_NOEXCEPT
{
    if (m_IsSdCardInserted == false)
    {
        // SD カードが挿入されていないので何もしません。
        // NN_LOG("Do nothing since sd card isn't inserted.\n");
        return;
    }

    m_LineQueue.push(str);
}

void SixAxisSensorLogger::Write() NN_NOEXCEPT
{
    if (m_LineQueue.empty())
    {
        // キューが空なので何もしません。
        return;
    }

    // キューに溜まった文字列を抽出します。
    ::std::string str;
    int strLength = 0;

    while (!m_LineQueue.empty())
    {
        auto temp = m_LineQueue.front().c_str();
        str += temp;
        strLength += ::std::strlen(temp);
        m_LineQueue.pop();
    }

    // 書き込み処理
    auto result = nn::fs::WriteFile(m_FileHandle,
                                    m_FileOffset,
                                    str.c_str(),
                                    strLength,
                                    nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));

    if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
    {
        // 空き領域が不足しています。
        // 書き込みサイズが想定範囲内の値となっているか、SD カードの空き領域は十分かを確認してください。
        NN_ABORT("Usable space not enough.\n");
        return;
    }
    // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
    // これ以上のエラーハンドリングは不要です。

    // 確認用にコンソールに SD カードに書き込まれるデータをログ出力します。
    // NN_LOG("%d: %s", strLength, str.c_str());

    m_FileOffset += strLength;
}
