﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid.h>
#include <nn/hid/debug/hid_Xpad.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#if defined(NN_BUILD_CONFIG_TOOLCHAIN_CLANG)
#pragma clang diagnostic ignored "-Wdeprecated-declarations"
#endif

#if defined(NN_BUILD_CONFIG_TOOLCHAIN_GCC)
#pragma GCC diagnostic ignored "-Wdeprecated-declarations"
#endif

#if defined(NN_BUILD_CONFIG_TOOLCHAIN_VC)
#pragma warning(disable:4996)
#endif

#include "../Common/testHid_Xpad.h"

namespace
{

struct BasicXpad
{
    typedef ::nn::hid::BasicXpadId Id;
    typedef ::nn::hid::BasicXpadState State;
    typedef ::nn::hid::BasicXpadAttributeSet AttributeSet;
    typedef ::nn::hid::BasicXpadButtonSet ButtonSet;
    static ::nn::hid::BasicXpadId s_Ids[::nn::hid::XpadIdCountMax];
    static ::nn::hid::BasicXpadState s_States[::nn::hid::XpadStateCountMax];
};

//!< 入力状態の読み出し時にバッファが正しく利用されるかテストします。
template<typename T>
void ReadingStateTest(int xpadIdIndex) NN_NOEXCEPT
{
    const int xpadIdCount = ::nn::hid::GetXpadIds(T::s_Ids,
                                                  ::nn::hid::XpadIdCountMax);
    ASSERT_GE(::nn::hid::XpadIdCountMax, xpadIdCount);
    ASSERT_GT(xpadIdCount, xpadIdIndex);

    const typename T::Id& xpadId = T::s_Ids[xpadIdIndex];

    for (typename T::State& state : T::s_States)
    {
        state = typename T::State();
        state.samplingNumber = 0xDEADBEEF;
        state.attributes.Set(1, true);
        state.buttons.Set(2, true);
        state.analogStickR.x = 0x2222;
        state.analogStickR.y = 0x3333;
        state.analogStickL.x = 0x4444;
        state.analogStickL.y = 0x5555;
    }

    ::nn::hid::InitializeXpad(xpadId);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::XpadStateCountMax / 2 *
            ::nnt::hid::GetXpadSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::GetXpadStates(T::s_States,
                                         ::nn::hid::XpadStateCountMax,
                                         xpadId);

    EXPECT_GE(::nn::hid::XpadStateCountMax, count);

    for (int i = count; i < ::nn::hid::XpadStateCountMax; ++i)
    {
        // 入力状態の書き込まれなかった領域は値を維持する。
        EXPECT_EQ(0xDEADBEEF, T::s_States[i].samplingNumber);
        typename T::AttributeSet attributes = {};
        attributes.Set(1, true);
        typename T::ButtonSet buttons = {};
        buttons.Set(2, true);
        EXPECT_EQ(attributes, T::s_States[i].attributes);
        EXPECT_EQ(buttons, T::s_States[i].buttons);
        EXPECT_EQ(0x2222, T::s_States[i].analogStickR.x);
        EXPECT_EQ(0x3333, T::s_States[i].analogStickR.y);
        EXPECT_EQ(0x4444, T::s_States[i].analogStickL.x);
        EXPECT_EQ(0x5555, T::s_States[i].analogStickL.y);
    }

    ::nn::hid::debug::FinalizeXpad(xpadId);
}

//!< 指定の優先度のスレッドから入力状態の読み出しが可能かテストします。
template<typename T>
void ThreadingTest(int xpadIdIndex, int priority) NN_NOEXCEPT
{
    ::nn::os::ThreadType* pThread = ::nn::os::GetCurrentThread();

    // 指定の優先度に設定する。
    const int original = ::nn::os::ChangeThreadPriority(pThread, priority);

    const int xpadIdCount = ::nn::hid::GetXpadIds(T::s_Ids,
                                                  ::nn::hid::XpadIdCountMax);
    ASSERT_GE(::nn::hid::XpadIdCountMax, xpadIdCount);
    ASSERT_GT(xpadIdCount, xpadIdIndex);

    for (int i = 0; i < xpadIdCount; ++i)
    {
        ::nn::hid::InitializeXpad(T::s_Ids[i]);
    }

    int lastCount = 1;
    int64_t lastSamplingNumber = 0;

    for (int i = 0; i < 60; ++i)
    {
        int count = ::nn::hid::GetXpadStates(T::s_States,
                                             ::nn::hid::XpadStateCountMax,
                                             T::s_Ids[xpadIdIndex]);
        EXPECT_LE(lastCount, count);

        EXPECT_LE(lastSamplingNumber, T::s_States[0].samplingNumber);

        for (int j = 0; j < count - 1; ++j)
        {
            const typename T::State& lhs = T::s_States[j];
            const typename T::State& rhs = T::s_States[j + 1];
            EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
        }

        lastCount = count;
        lastSamplingNumber = T::s_States[0].samplingNumber;

        ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(16));
    }

    EXPECT_EQ(::nn::hid::XpadStateCountMax, lastCount);

    for (int i = 0; i < xpadIdCount; ++i)
    {
        ::nn::hid::debug::FinalizeXpad(T::s_Ids[i]);
    }

    // 優先度を元に戻す。
    ::nn::os::ChangeThreadPriority(pThread, original);
}

} // namespace

class XpadReadingStateSuiteWithParam : public ::testing::TestWithParam<int>
{
};

INSTANTIATE_TEST_CASE_P(XpadReadingStateSuite,
                        XpadReadingStateSuiteWithParam,
                        ::testing::Range(0, 4));

//!< 入力状態の読み出し時にバッファは正しく利用されるか
TEST_P(XpadReadingStateSuiteWithParam, ReadingStateTest1)
{
    ReadingStateTest<BasicXpad>(GetParam());
}

//!< 高優先度のスレッドから入力状態の読み出しが可能か
TEST_P(XpadReadingStateSuiteWithParam, ThreadingTest1)
{
    ThreadingTest<BasicXpad>(GetParam(), ::nn::os::HighestThreadPriority);
}

//!< 低優先度のスレッドから入力状態の読み出しが可能か
TEST_P(XpadReadingStateSuiteWithParam, ThreadingTest2)
{
    ThreadingTest<BasicXpad>(GetParam(), ::nn::os::LowestThreadPriority);
}

namespace {

::nn::hid::BasicXpadId BasicXpad::s_Ids[::nn::hid::XpadIdCountMax];
::nn::hid::BasicXpadState BasicXpad::s_States[::nn::hid::XpadStateCountMax];

} // namespace
