﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/system/hid_NpadSystem.h>
#include <nn/hid/system/hid_NpadSystemExt.h>
#include <nn/hid/hid_Vibration.h>
#include <nn/hid/hid_VibrationGcErm.h>

#include "../Npad/testHid_NpadCommon.h"

#include <nn/nn_SdkLog.h>

namespace {

    const int VibrationDeviceCountMax = 2;

    void VibrationDeviceHandleTest(const nn::hid::NpadIdType& id, nn::hid::NpadStyleSet style, int vibrationDeviceCountMax)
    {
        nn::hid::InitializeNpad();

        // 使用する操作形態を設定する
        nn::hid::SetSupportedNpadStyleSet(style);

        // 使用する Npad を設定する
        nn::hid::SetSupportedNpadIdType(testhid::NpadIds, sizeof(testhid::NpadIds) / sizeof(testhid::NpadIds[0]));

        // 振動子のハンドルを取得する
        nn::hid::VibrationDeviceHandle handles[VibrationDeviceCountMax];

        int m_VibrationDeviceCount = nn::hid::GetVibrationDeviceHandles(
            handles, vibrationDeviceCountMax, id, style);

        // 0 <= m_VibrationDeviceCount <= VibrationDeviceCountMax
        EXPECT_GE(m_VibrationDeviceCount, 0);
        EXPECT_LE(m_VibrationDeviceCount, VibrationDeviceCountMax);

        // m_VibrationDeviceCount <= vibrationDeviceCountMax
        EXPECT_LE(m_VibrationDeviceCount, vibrationDeviceCountMax);

        if (vibrationDeviceCountMax == 1)
        {
            EXPECT_EQ(m_VibrationDeviceCount, 1);
        }
        else if (vibrationDeviceCountMax > 1)
        {
            if (style == nn::hid::NpadStyleJoyLeft::Mask ||
                style == nn::hid::NpadStyleJoyRight::Mask ||
                style == nn::hid::NpadStyleGc::Mask)
            {
                EXPECT_EQ(m_VibrationDeviceCount, 1);
            }
            else
            {
                EXPECT_EQ(m_VibrationDeviceCount, 2);
            }
        }

        for (int i = 0; i < m_VibrationDeviceCount; i++)
        {
            // 振動子を初期化する
            nn::hid::InitializeVibrationDevice(handles[i]);

            nn::hid::VibrationDeviceInfo deviceInfo;

            // 振動子の情報を取得する
            nn::hid::GetVibrationDeviceInfo(&deviceInfo, handles[i]);

            // GcStyle 以外は LRA
            if (style != nn::hid::NpadStyleGc::Mask)
            {
                // 現時点では振動子の位置は必ず左右どちらかになる
                EXPECT_GE(deviceInfo.position, nn::hid::VibrationDevicePosition_Left);
                EXPECT_LE(deviceInfo.position, nn::hid::VibrationDevicePosition_Right);

                if (style == nn::hid::NpadStyleJoyLeft::Mask)
                {
                    EXPECT_EQ(deviceInfo.position, nn::hid::VibrationDevicePosition_Left);
                }

                if (style == nn::hid::NpadStyleJoyRight::Mask)
                {
                    EXPECT_EQ(deviceInfo.position, nn::hid::VibrationDevicePosition_Right);
                }


                // 現時点では振動子の種類は必ずLRAになる
                EXPECT_EQ(deviceInfo.deviceType, nn::hid::VibrationDeviceType_LinearResonantActuator);

                nn::hid::VibrationValue vib = nn::hid::VibrationValue::Make();
                vib.amplitudeLow = 1.0f;
                vib.amplitudeHigh = 1.0f;

                nn::hid::VibrationValue ret = nn::hid::VibrationValue::Make();

                // 振動値を送信します
                nn::hid::SendVibrationValue(handles[i], vib);

                // 現在の振動を取得します
                nn::hid::GetActualVibrationValue(&ret, handles[i]);

                //接続していないときは何を設定しても振幅0が返ってくる
                //EXPECT_FLOAT_EQ(vib.amplitudeHigh, ret.amplitudeHigh);
                //EXPECT_FLOAT_EQ(vib.amplitudeLow, ret.amplitudeLow);
                //EXPECT_FLOAT_EQ(vib.frequencyHigh, ret.frequencyHigh);
                //EXPECT_FLOAT_EQ(vib.frequencyLow, ret.frequencyLow);
                //NN_SDK_LOG("%d %d %d/%d/%d %d %d\n", id, style, i, m_VibrationDeviceCount, vibrationDeviceCountMax, deviceInfo.position, deviceInfo.deviceType);
                //NN_SDK_LOG("%f %f\n", vib.amplitudeHigh, ret.amplitudeHigh);
            }
            else if (style == nn::hid::NpadStyleGc::Mask)
            {
                // Position は None
                EXPECT_EQ(deviceInfo.position, nn::hid::VibrationDevicePosition_None);
                // StyleGc の場合は GcErm になる
                EXPECT_EQ(deviceInfo.deviceType, nn::hid::VibrationDeviceType_GcErm);

                auto vib = nn::hid::VibrationGcErmCommand_Stop;
                nn::hid::VibrationGcErmCommand ret;

                // 振動値を送信します
                nn::hid::SendVibrationGcErmCommand(handles[i], vib);

                // 現在の振動を取得します
                nn::hid::GetActualVibrationGcErmCommand(&ret, handles[i]);

                //接続していないときは何を設定しても Stop が返ってくる
                EXPECT_EQ(ret, nn::hid::VibrationGcErmCommand_Stop);

            }

        }
    }
}


//!< 振動のハンドルを取得できるか（正常系）
TEST(VibrationDeviceApiSuite, SuccessTest1)
{
    int tests[] = { 1,2,3 };

    for (int id = 0; id < testhid::NpadIdCountMax; id++)
    {
        for (int style = 0; style < testhid::NpadStyleCountMax; style++)
        {
            for (auto number : tests)
            {
                if (testhid::NpadStyles[style] != nn::hid::NpadStylePalma::Mask)
                {
                    VibrationDeviceHandleTest(testhid::NpadIds[id], testhid::NpadStyles[style], number);
                }
            }
        }
    }
}

//!< 振動のハンドルを取得できるか（エラー系）
TEST(VibrationDeviceApiSuite, ErrorTest1)
{
    int tests[] = { 0 };

    for (int id = 0; id < testhid::NpadIdMinimalSetCountMax; id++)
    {
        for (int style = 0; style < testhid::NpadStyleCountMax; style++)
        {
            for (auto number : tests)
            {
                EXPECT_DEATH_IF_SUPPORTED(VibrationDeviceHandleTest(testhid::NpadIdsMinimalSet[id], testhid::NpadStyles[style], number), "");
            }
        }
    }
}
