﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/hid/debug/hid_SleepButton.h>
#include <nn/hid/system/hid_SleepButton.h>
#include <nn/os.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nntest.h>

#include "../Common/testHid_SleepButton.h"

namespace
{
::nn::hid::system::SleepButtonState g_State;
} // namespace

//!< スリープボタンの初期化状態は GetSleepButtonStates() の呼び出しに反映されるか
TEST(SleepButtonInitializationSuite, NoInitializationTest1)
{
    // スリープボタンが未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::system::GetSleepButtonStates(&g_State, 1), "");

    ::nn::hid::system::InitializeSleepButton();

    ::nn::hid::system::InitializeSleepButton();

    ::nn::hid::debug::FinalizeSleepButton();

    // スリープボタンが終了しきっていないので成功する。
    ::nn::hid::system::GetSleepButtonStates(&g_State, 1);

    ::nn::hid::debug::FinalizeSleepButton();

    // スリープボタンの終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::system::GetSleepButtonStates(&g_State, 1), "");
}

//!< スリープボタンの初期化状態は SetSleepButtonAutoPilotState() の呼び出しに反映されるか
TEST(SleepButtonInitializationSuite, NoInitializationTest2)
{
    ::nn::hid::debug::SleepButtonAutoPilotState autoPilotState = {};

    // スリープボタンが未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::SetSleepButtonAutoPilotState(autoPilotState), "");

    ::nn::hid::system::InitializeSleepButton();

    ::nn::hid::system::InitializeSleepButton();

    ::nn::hid::debug::FinalizeSleepButton();

    // スリープボタンが終了しきっていないので成功する。
    ::nn::hid::debug::SetSleepButtonAutoPilotState(autoPilotState);

    ::nn::hid::debug::FinalizeSleepButton();

    // スリープボタンの終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::SetSleepButtonAutoPilotState(autoPilotState), "");
}

//!< スリープボタンの初期化状態は UnsetSleepButtonAutoPilotState() の呼び出しに反映されるか
TEST(SleepButtonInitializationSuite, NoInitializationTest3)
{
    // スリープボタンが未初期化の状態で呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::UnsetSleepButtonAutoPilotState(), "");

    ::nn::hid::system::InitializeSleepButton();

    ::nn::hid::system::InitializeSleepButton();

    ::nn::hid::debug::FinalizeSleepButton();

    // スリープボタンが終了しきっていないので成功する。
    ::nn::hid::debug::UnsetSleepButtonAutoPilotState();

    ::nn::hid::debug::FinalizeSleepButton();

    // スリープボタンの終了処理後に呼び出すとアボートする。
    EXPECT_DEATH_IF_SUPPORTED(
        ::nn::hid::debug::UnsetSleepButtonAutoPilotState(), "");
}

//!< スリープボタンの初期化処理は最低一つの入力状態の取得を保証するか
TEST(SleepButtonInitializationSuite, StateReadingTest1)
{
    ::nn::hid::system::InitializeSleepButton();

    // 初期化直後の入力状態の取得に成功する。
    ::nn::hid::system::GetSleepButtonStates(&g_State, 1);
    EXPECT_LE(0, g_State.samplingNumber);
    EXPECT_TRUE(g_State.buttons.IsAllOff());

    ::nn::hid::debug::FinalizeSleepButton();
}

//!< スリープボタンの初期化処理はサンプリングを正しく有効化するか
TEST(SleepButtonInitializationSuite, StateReadingTest2)
{
    ::nn::hid::system::InitializeSleepButton();

    ::nn::hid::system::InitializeSleepButton();

    ::nn::hid::system::GetSleepButtonStates(&g_State, 1);

    int64_t samplingNumber = g_State.samplingNumber;

    ::nn::hid::debug::FinalizeSleepButton();

    // スリープボタンが終了しきっていないのでサンプリングは継続する。
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::SleepButtonStateCountMax *
            ::nnt::hid::GetSleepButtonSamplingInterval().GetMilliSeconds()));

    ::nn::hid::system::GetSleepButtonStates(&g_State, 1);

    EXPECT_LT(samplingNumber + ::nn::hid::system::SleepButtonStateCountMax / 4,
              g_State.samplingNumber);

    ::nn::hid::debug::FinalizeSleepButton();
#if defined(NN_BUILD_CONFIG_OS_WIN)
    samplingNumber = g_State.samplingNumber;

    // スリープボタンが終了しているのでサンプリングは停止する。
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::system::SleepButtonStateCountMax *
            ::nnt::hid::GetSleepButtonSamplingInterval().GetMilliSeconds()));

    ::nn::hid::system::InitializeSleepButton();

    ::nn::hid::system::GetSleepButtonStates(&g_State, 1);

    EXPECT_GT(samplingNumber + ::nn::hid::system::SleepButtonStateCountMax / 4,
              g_State.samplingNumber);

    ::nn::hid::debug::FinalizeSleepButton();
#endif
}
