﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>

#include <nn/hid/hid_NpadFullKey.h>
#include <nn/hid/hid_NpadHandheld.h>
#include <nn/hid/hid_NpadJoyDual.h>
#include <nn/hid/hid_NpadJoyLeft.h>
#include <nn/hid/hid_NpadJoyRight.h>
#include <nn/hid/system/hid_NpadSystem.h>
#include <nn/hid/system/hid_NpadSystemExt.h>

#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/hid/debug/hid_SixAxisSensor.h>

#include <nnt/nntest.h>

namespace {

const ::nn::hid::NpadIdType NpadIds[] = {
    ::nn::hid::NpadId::No1,
    ::nn::hid::NpadId::No2,
    ::nn::hid::NpadId::No3,
    ::nn::hid::NpadId::No4,
    ::nn::hid::NpadId::No5,
    ::nn::hid::NpadId::No6,
    ::nn::hid::NpadId::No7,
    ::nn::hid::NpadId::No8,
    ::nn::hid::NpadId::Handheld,
};

const ::nn::hid::NpadStyleSet NpadStyles[] = {
    ::nn::hid::NpadStyleFullKey::Mask,
    ::nn::hid::NpadStyleHandheld::Mask,
    ::nn::hid::NpadStyleJoyDual::Mask,
    ::nn::hid::NpadStyleJoyLeft::Mask,
    ::nn::hid::NpadStyleJoyRight::Mask,
    ::nn::hid::system::NpadStyleSystem::Mask,
    ::nn::hid::system::NpadStyleSystemExt::Mask,
};

const float InitialAccOrigin = 0.f;
const float InitialAccSensitivity = 1.f;

const float InitialGyroOrigin = 0.f;
const float InitialGyroSensitivity = 1.f;

void CheckCalibrationValues(const ::nn::hid::SixAxisSensorHandle& handle,
                            const float ExpectedAccOrigin,
                            const float ExpectedAccSensitivity,
                            const float ExpectedGyroOrigin,
                            const float ExpectedGyroSensitivity) NN_NOEXCEPT
{
    float origin;
    float sensitivity;

    ::nn::hid::debug::GetShiftAccelerometerCalibrationValue(&origin,
                                                            &sensitivity,
                                                            handle);
    EXPECT_EQ(ExpectedAccOrigin, origin);
    EXPECT_EQ(ExpectedAccSensitivity, sensitivity);

    ::nn::hid::debug::GetShiftGyroscopeCalibrationValue(&origin,
                                                        &sensitivity,
                                                        handle);
    EXPECT_EQ(ExpectedGyroOrigin, origin);
    EXPECT_EQ(ExpectedGyroSensitivity, sensitivity);
}

} // namespace

//!< SixAxisSensor のキャリブレーション値のシフト設定値の初期状態が仕様通りか
TEST(SixAxisSensorShiftCalibrationValue, InitialState)
{
    ::nn::hid::SixAxisSensorHandle handles[::nn::hid::SixAxisSensorStateCountMax] = {};

    for (const auto& id : NpadIds)
    {
        for (const auto& style : NpadStyles)
        {
            int count = ::nn::hid::GetSixAxisSensorHandles(handles,
                                                           ::nn::hid::SixAxisSensorStateCountMax,
                                                           id,
                                                           style);

            for (int i = 0; i < count; i++)
            {
                CheckCalibrationValues(handles[i],
                                       InitialAccOrigin,
                                       InitialAccSensitivity,
                                       InitialGyroOrigin,
                                       InitialGyroSensitivity);
            }
        }
    }
}

//!< SixAxisSensor のキャリブレーション値のシフト設定値を書き換えられるか
TEST(SixAxisSensorShiftCalibrationValue, SetValue)
{
    const float AccOrigin = 0xDEAD;
    const float AccSensitivity = 0xBEEF;
    const float GyroOrigin = 0xFACE;
    const float GyroSensitivity = 0xFEED;

    ::nn::hid::SixAxisSensorHandle handles[::nn::hid::SixAxisSensorStateCountMax] = {};

    for (const auto& id : NpadIds)
    {
        for (const auto& style : NpadStyles)
        {
            int count = ::nn::hid::GetSixAxisSensorHandles(handles,
                                                           ::nn::hid::SixAxisSensorStateCountMax,
                                                           id,
                                                           style);

            for (int i = 0; i < count; i++)
            {
                // 設定値を変更
                ::nn::hid::debug::SetShiftAccelerometerCalibrationValue(handles[i],
                                                                        AccOrigin,
                                                                        AccSensitivity);
                ::nn::hid::debug::SetShiftGyroscopeCalibrationValue(handles[i],
                                                                    GyroOrigin,
                                                                    GyroSensitivity);

                CheckCalibrationValues(handles[i],
                                       AccOrigin,
                                       AccSensitivity,
                                       GyroOrigin,
                                       GyroSensitivity);

                // 初期設定に書き戻す
                ::nn::hid::debug::SetShiftAccelerometerCalibrationValue(handles[i],
                                                                        InitialAccOrigin,
                                                                        InitialAccSensitivity);
                ::nn::hid::debug::SetShiftGyroscopeCalibrationValue(handles[i],
                                                                    InitialGyroOrigin,
                                                                    InitialGyroSensitivity);

                CheckCalibrationValues(handles[i],
                                       InitialAccOrigin,
                                       InitialAccSensitivity,
                                       InitialGyroOrigin,
                                       InitialGyroSensitivity);
            }
        }

    }
}
