﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_TimeSpan.h>
#include <nn/hid.h>
#include <nn/hid/hid_Mouse.h>
#include <nn/hid/debug/hid_Mouse.h>
#include <nn/hid/debug/hid_TouchScreen.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#include "../Common/testHid_Mouse.h"
#include "../Common/testHid_TouchScreen.h"

namespace
{

::nn::hid::MouseState g_MouseStates[::nn::hid::MouseStateCountMax];

::nn::hid::TouchScreenState<::nn::hid::TouchStateCountMax>
    g_TouchScreenStates[::nn::hid::TouchScreenStateCountMax];

} // namespace

//!< Mouse の初期化処理が正しく処理されるか
TEST(MouseAndTouchScreenSuite, InitializationTest1)
{
    ::nn::hid::InitializeTouchScreen();

    ::nn::hid::InitializeMouse();

    ::nn::hid::GetMouseState(&g_MouseStates[0]);

    int64_t samplingNumber = g_MouseStates[0].samplingNumber;

    ::nn::hid::debug::FinalizeTouchScreen();

    // Mouse が終了しきっていないのでサンプリングは継続する。
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    ::nn::hid::GetMouseState(&g_MouseStates[0]);

    EXPECT_LT(samplingNumber + ::nn::hid::MouseStateCountMax / 4,
              g_MouseStates[0].samplingNumber);

    ::nn::hid::debug::FinalizeMouse();
}

//!< TouchScreen の初期化処理が正しく処理されるか
TEST(MouseAndTouchScreenSuite, InitializationTest2)
{
    ::nn::hid::InitializeMouse();

    ::nn::hid::InitializeTouchScreen();

    ::nn::hid::GetTouchScreenState(&g_TouchScreenStates[0]);

    int64_t samplingNumber = g_TouchScreenStates[0].samplingNumber;

    ::nn::hid::debug::FinalizeMouse();

    // TouchScreen が終了しきっていないのでサンプリングは継続する。
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::TouchScreenStateCountMax *
            ::nnt::hid::GetTouchScreenSamplingInterval().GetMilliSeconds()));

    ::nn::hid::GetTouchScreenState(&g_TouchScreenStates[0]);

    EXPECT_LT(samplingNumber + ::nn::hid::TouchScreenStateCountMax / 4,
              g_TouchScreenStates[0].samplingNumber);

    ::nn::hid::debug::FinalizeTouchScreen();
}

//!< Mouse と TouchScreen の入力状態の読み出しが同時に可能か
TEST(MouseAndTouchScreenSuite, ReadingStateTest1)
{
    ::nn::hid::InitializeMouse();
    ::nn::hid::InitializeTouchScreen();

    ::nn::TimeSpan t0 = ::nn::os::ConvertToTimeSpan(::nn::os::GetSystemTick());

    ::nn::hid::GetMouseState(&g_MouseStates[0]);

    ::nn::hid::GetTouchScreenState(&g_TouchScreenStates[0]);

    const int64_t mouseSamplingNumber =
        g_MouseStates[0].samplingNumber;

    const int64_t touchScreenSamplingNumber =
        g_TouchScreenStates[0].samplingNumber;

    ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(24));

    const int mouseStateCount = ::nn::hid::GetMouseStates(
        g_MouseStates,
        ::nn::hid::MouseStateCountMax);

    const int touchScreenStateCount = ::nn::hid::GetTouchScreenStates(
        g_TouchScreenStates,
        ::nn::hid::TouchScreenStateCountMax);

    ::nn::TimeSpan t1 = ::nn::os::ConvertToTimeSpan(::nn::os::GetSystemTick());

    int64_t expectedMouseStateCount =
        (t1 - t0).GetMilliSeconds() /
        ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds();

    int64_t expectedTouchScreenStateCount =
        (t1 - t0).GetMilliSeconds() /
        ::nnt::hid::GetTouchScreenSamplingInterval().GetMilliSeconds();

    if (expectedMouseStateCount > ::nn::hid::MouseStateCountMax)
    {
        expectedMouseStateCount = ::nn::hid::MouseStateCountMax;
    }

    if (expectedTouchScreenStateCount > ::nn::hid::TouchScreenStateCountMax)
    {
        expectedTouchScreenStateCount = ::nn::hid::TouchScreenStateCountMax;
    }

    // Mouse の入力状態は 8 ms 以上の間隔を開けて追加される。
    const int64_t mouseObservedCount =
        g_MouseStates[0].samplingNumber - mouseSamplingNumber + 1;
    EXPECT_LE(expectedMouseStateCount - 2, mouseObservedCount);
    EXPECT_GE(expectedMouseStateCount + 2, mouseObservedCount);

    // TouchScreen の入力状態は 4 ms 以上の間隔を開けて追加される。
    const int64_t touchScreenObservedCount =
        g_TouchScreenStates[0].samplingNumber - touchScreenSamplingNumber + 1;
    EXPECT_LE(expectedTouchScreenStateCount - 2, touchScreenObservedCount);
    EXPECT_GE(expectedTouchScreenStateCount + 2, touchScreenObservedCount);

    for (int i = 0; i < mouseStateCount; ++i)
    {
        const ::nn::hid::MouseState& lhs = g_MouseStates[i];
        EXPECT_LE(0, lhs.samplingNumber);
        EXPECT_EQ(0, lhs.x);
        EXPECT_EQ(0, lhs.y);
        EXPECT_EQ(0, lhs.wheelDelta);
        EXPECT_TRUE(lhs.buttons.IsAllOff());
        ::nn::hid::MouseAttributeSet mouseAttributes = {};
        mouseAttributes.Set<::nn::hid::MouseAttribute::Transferable>(true);
        EXPECT_EQ(mouseAttributes, lhs.attributes);

        if (i + 1 < mouseStateCount)
        {
            const ::nn::hid::MouseState& rhs = g_MouseStates[i + 1];
            EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
        }
    }

    for (int i = 0; i < touchScreenStateCount; ++i)
    {
        const auto& lhs = g_TouchScreenStates[i];
        EXPECT_LE(0, lhs.samplingNumber);
        EXPECT_EQ(0, lhs.count);

        if (i + 1 < touchScreenStateCount)
        {
            const auto& rhs = g_TouchScreenStates[i + 1];
            EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
        }
    }

    ::nn::hid::debug::FinalizeTouchScreen();
    ::nn::hid::debug::FinalizeMouse();
}
