﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/hid/hid_Npad.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/os.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nnt/nntest.h>

#include <nn/nn_Log.h>

#include "testHid_NpadCommon.h"
#include "../Common/testHid_Npad.h"

// #define TESTHID_ENABLE_FULL_COMBINATION

namespace
{

const nn::util::BitPack32 NpadIdTestPattern[] = {
    { 0x0000 }, // NpadId 無し
    { 0x000f }, // 1～4
    { 0x00ff }, // 1～8
    { 0x010f }, // Handheld + 1～4
    { 0x01ff }, // Handheld + 1～8
    { 0x03ff }, // 全部
};
const int NpadIdTestPatternCountMax = sizeof(NpadIdTestPattern) / sizeof(NpadIdTestPattern[0]);

#ifdef TESTHID_ENABLE_FULL_COMBINATION
void TestNpadIdCombination(nn::hid::NpadIdType* pIds, int count, int step)
{
    step++;
    for (int i = 0; i < 2; i++)
    {
        if (i == 1)
        {
            pIds[count] = ::testhid::NpadIds[step - 1];
            count++;
        }

        if (step == ::testhid::NpadIdCountMax)
        {
            nn::hid::SetSupportedNpadIdType(pIds, count);

            for (auto& npadId : ::testhid::NpadIds)
            {
                bool isSupported = false;
                for (int j = 0; j < count; j++)
                {
                    if (npadId == pIds[j])
                    {
                        isSupported = true;
                        break;
                    }
                }

                if (isSupported)
                {
                    ::testhid::TestGetNpadState(npadId, false);
                    ::testhid::TestGetNpadStates(npadId, false);
                }
            }
        }
        else
        {
            TestNpadIdCombination(pIds, count, step);
        }
    }
}
#endif

}

//!< Npad の初期化状態は GetNpadState() の呼び出しに反映されるか
TEST(NpadInitialization, NoInitializationTest1)
{
    {
        SCOPED_TRACE("NpadId::Handheld with HandheldStyle");
        ::testhid::ExpectDeathGetNpadState(nn::hid::NpadId::Handheld, nn::hid::NpadStyleHandheld::Mask);
    }
    {
        SCOPED_TRACE("NpadId::Handheld with FullKeyStyle");
        ::testhid::ExpectDeathGetNpadStates(nn::hid::NpadId::Handheld, nn::hid::NpadStyleFullKey::Mask);
    }
    {
        SCOPED_TRACE("NpadId::No1 with FullKeyStyle");
        ::testhid::ExpectDeathGetNpadState(nn::hid::NpadId::No1, nn::hid::NpadStyleFullKey::Mask);
    }
    {
        SCOPED_TRACE("NpadId::No1 with JoyLeft");
        ::testhid::ExpectDeathGetNpadStates(nn::hid::NpadId::No1, nn::hid::NpadStyleJoyLeft::Mask);
    }
    {
        SCOPED_TRACE("NpadId::No8 with FullKeyStyle");
        ::testhid::ExpectDeathGetNpadState(nn::hid::NpadId::No8, nn::hid::NpadStyleFullKey::Mask);
    }
    {
        SCOPED_TRACE("NpadId::No8 with JoyRightStyle");
        ::testhid::ExpectDeathGetNpadStates(nn::hid::NpadId::No8, nn::hid::NpadStyleJoyRight::Mask);
    }
}

//!< NpadId を有効にする処理はすべての組み合わせで成功するか
TEST(NpadInitialization, NpadIdCombination)
{
    ::nn::hid::InitializeNpad();

    // 全ての Style を有効にする
    ::testhid::EnableStyleAll();

    nn::hid::NpadIdType npadIds[::testhid::NpadIdCountMax];

#ifdef TESTHID_ENABLE_FULL_COMBINATION
    TestNpadIdCombination(npadIds, 0, 0);
#else
    for (int testPatternIndex = 0; testPatternIndex < NpadIdTestPatternCountMax; testPatternIndex++)
    {
        int npadIdCount = 0;

        for (int idIndex = 0; idIndex < ::testhid::NpadIdCountMax; idIndex++)
        {
            if (NpadIdTestPattern[testPatternIndex].GetBit(idIndex) == true)
            {
                npadIds[npadIdCount] = ::testhid::NpadIds[idIndex];
                npadIdCount++;
            }
        }

        nn::hid::SetSupportedNpadIdType(npadIds, npadIdCount);

        for (int idIndex = 0; idIndex < ::testhid::NpadIdCountMax; idIndex++)
        {
            if (NpadIdTestPattern[testPatternIndex].GetBit(idIndex) == true)
            {
                ::testhid::TestGetNpadState(::testhid::NpadIds[idIndex], false);
                ::testhid::TestGetNpadStates(::testhid::NpadIds[idIndex], false);
            }
        }
    }
#endif

    nn::hid::debug::FinalizeNpad();
}
