﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_Mouse.h>
#include <nn/hid/debug/hid_Mouse.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#include "../Common/testHid_Mouse.h"

namespace
{

::nn::hid::MouseState g_States1[::nn::hid::MouseStateCountMax];
::nn::hid::MouseState g_States2[::nn::hid::MouseStateCountMax];

//!< Mouse の自動操作状態をファジングします。
void FuzzMouseAutoPilotState(
    ::nn::hid::debug::MouseAutoPilotState* pState, int seed) NN_NOEXCEPT;

//!< Mouse の入力状態が入力無し状態であることを期待します。
void ExpectDefaultMouseState(const ::nn::hid::MouseState& state) NN_NOEXCEPT;

//!< Mouse の入力状態がファジングされた入力状態であることを期待します。
void ExpectFuzzedMouseState(const ::nn::hid::MouseState& state) NN_NOEXCEPT;

//!< Mouse の入力状態が同じドライバの状態から決定されたことを期待します。
void ExpectSameOriginMouseStates(
    const ::nn::hid::MouseState& lhs,
    const ::nn::hid::MouseState& rhs) NN_NOEXCEPT;

//!< Mouse の入力状態が隣り合っていることを期待します。
void ExpectAdjacentMouseStates(
    const ::nn::hid::MouseState& lhs,
    const ::nn::hid::MouseState& rhs) NN_NOEXCEPT;

} // namespace

//!< Mouse の初期化処理は初期化処理は自動操作状態を正しく管理するか
TEST(MouseAutoPilotSuite, InitializationTest1)
{
    ::nn::hid::debug::MouseAutoPilotState autoPilotState;
    FuzzMouseAutoPilotState(&autoPilotState, 1);

    ::nn::hid::InitializeMouse();

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 2 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    // 自動操作が反映された入力状態が取得される。
    ::nn::hid::GetMouseState(&g_States1[0]);
    {
        SCOPED_TRACE("");
        ExpectFuzzedMouseState(g_States1[0]);
    }

    ::nn::hid::InitializeMouse();

    ::nn::hid::debug::FinalizeMouse();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 2 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    // まだユーザが存在するので自動操作設定は解除されない。
    ::nn::hid::GetMouseState(&g_States1[0]);
    {
        SCOPED_TRACE("");
        ExpectFuzzedMouseState(g_States1[0]);
    }

    ::nnt::hid::FinalizeMouseForAutoPilot();

    ::nn::hid::InitializeMouse();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 2 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    // Mouse が完全に開放されたことで自動操作設定が解除される。
    ::nn::hid::GetMouseState(&g_States1[0]);
    {
        SCOPED_TRACE("");
        ExpectDefaultMouseState(g_States1[0]);
    }

    ::nnt::hid::FinalizeMouseForAutoPilot();
}

//!< Mouse の入力状態は不正な自動操作状態が設定されても仕様を保証するか
TEST(MouseAutoPilotSuite, InvalidArgumentTest1)
{
    ::nn::hid::debug::MouseAutoPilotState autoPilotState = {};

    ::nn::hid::MouseState& state = g_States1[0];

    ::nn::hid::InitializeMouse();

    // 変域を負の方向に超過
    autoPilotState.x = -1;
    autoPilotState.y = -1;

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 2 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    ::nn::hid::GetMouseState(&state);

    // 禁則処理が保証されている
    EXPECT_EQ(0, state.x);
    EXPECT_EQ(0, state.y);

    // 変域を正の方向に超過
    autoPilotState.x = ::nnt::hid::MouseRectangleWidth;
    autoPilotState.y = ::nnt::hid::MouseRectangleHeight;

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 2 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    ::nn::hid::GetMouseState(&state);

    // 禁則処理が保証されている
    EXPECT_EQ(::nnt::hid::MouseRectangleWidth - 1, state.x);
    EXPECT_EQ(::nnt::hid::MouseRectangleHeight - 1, state.y);

    ::nnt::hid::FinalizeMouseForAutoPilot();
}

//!< Mouse の入力状態は自動操作状態の設定を正しく反映するか
TEST(MouseAutoPilotSuite, StateReadingTest1)
{
    ::nn::hid::InitializeMouse();

    ::nn::hid::GetMouseState(&g_States1[0]);
    {
        SCOPED_TRACE("");
        ExpectDefaultMouseState(g_States1[0]);
    }

    ::nn::hid::debug::MouseAutoPilotState autoPilotState;

    for (int i = 0; i <= ::nn::hid::MouseStateCountMax / 3; ++i)
    {
        FuzzMouseAutoPilotState(&autoPilotState, i);
        ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                2 * ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));
    }

    int count = ::nn::hid::GetMouseStates(
        g_States1, ::nn::hid::MouseStateCountMax);
    EXPECT_LE(1, count);

    bool isActive = true;

    // 自動操作状態に応じて Mouse の入力状態は変化する。
    for (int i = 0; i < count; ++i)
    {
        const ::nn::hid::MouseState& lhs = g_States1[i];

        EXPECT_LE(0, lhs.samplingNumber);

        if (lhs.buttons.IsAllOff())
        {
            isActive = false;
        }
        else
        {
            SCOPED_TRACE("");
            EXPECT_TRUE(isActive);
            ExpectFuzzedMouseState(lhs);
        }

        if (i + 1 < count)
        {
            SCOPED_TRACE("");
            ExpectAdjacentMouseStates(lhs, g_States1[i + 1]);
        }
    }

    ::nnt::hid::FinalizeMouseForAutoPilot();
}

//!< Mouse の入力状態は自動操作状態の解除を正しく反映するか
TEST(MouseAutoPilotSuite, StateReadingTest2)
{
    ::nn::hid::InitializeMouse();

    ::nn::hid::debug::MouseAutoPilotState autoPilotState;
    FuzzMouseAutoPilotState(&autoPilotState, 10);

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    // 自動操作状態を解除
    ::nn::hid::debug::UnsetMouseAutoPilotState();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 2 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::GetMouseStates(g_States1,
                                          ::nn::hid::MouseStateCountMax);
    EXPECT_EQ(::nn::hid::MouseStateCountMax, count);

    bool hasChanged = false;

    ::nn::hid::MouseState state = {};

    state.attributes = g_States1[0].attributes;

    // 入力状態列状で自動操作状態は連続的に切り替わる。
    for (int i = 0; i < count; ++i)
    {
        if (i + 1 < count)
        {
            const ::nn::hid::MouseState& lhs = g_States1[i];
            const ::nn::hid::MouseState& rhs = g_States1[i + 1];
            EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
            EXPECT_LE(lhs.x, rhs.x);
            EXPECT_LE(lhs.y, rhs.y);
        }

        if (g_States1[i].x != state.x)
        {
            EXPECT_FALSE(hasChanged);
            hasChanged = true;
            state.x = autoPilotState.x;
            state.y = autoPilotState.y;
            state.wheelDelta = autoPilotState.wheelDelta;
            state.buttons = autoPilotState.buttons;
            state.attributes = autoPilotState.attributes;
        }

        {
            SCOPED_TRACE("");
            ExpectSameOriginMouseStates(state, g_States1[i]);
        }
    }

    EXPECT_TRUE(hasChanged);

    ::nnt::hid::FinalizeMouseForAutoPilot();
}

//!< LIFO の境界を跨いで自動操作状態の設定を正しく反映するか
TEST(MouseAutoPilotSuite, StateReadingTest3)
{
    ::nn::hid::InitializeMouse();

    ::nn::hid::debug::MouseAutoPilotState autoPilotState;

    int lastCount = 0;

    // LIFO の境界を 2 回跨ぐまでサンプリングを継続
    for (int i = 0; i < ::nn::hid::MouseStateCountMax * 3; ++i)
    {
        FuzzMouseAutoPilotState(&autoPilotState, i);
        ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

        // 今回のサンプリング結果を格納するバッファ
        ::nn::hid::MouseState *ptr1 = (i % 2 == 1) ? &g_States1[0]
                                                   : &g_States2[0];

        // 前回のサンプリング結果が格納されているバッファ
        ::nn::hid::MouseState *ptr2 = (i % 2 == 1) ? &g_States2[0]
                                                   : &g_States1[0];

        // 取得される入力状態の数は単調増加する。
        int count = ::nn::hid::GetMouseStates(ptr1,
                                              ::nn::hid::MouseStateCountMax);
        EXPECT_LE(lastCount, count);

        bool isActive = true;

        for (int j = 0; j < count; ++j)
        {
            if (ptr1[j].buttons.IsAllOff())
            {
                isActive = false;
                EXPECT_LE(0, ptr1[i].samplingNumber);
            }
            else
            {
                SCOPED_TRACE("");
                EXPECT_TRUE(isActive);
                ExpectFuzzedMouseState(ptr1[j]);
            }

            if (j + 1 < count)
            {
                SCOPED_TRACE("");
                ExpectAdjacentMouseStates(ptr1[j], ptr1[j + 1]);
            }

            // 前回のサンプリング結果と重複する箇所には同じ値が入る。
            if (lastCount > 0 &&
                ptr1[j].samplingNumber == ptr2[0].samplingNumber)
            {
                for (int k = 0; k < lastCount - j; ++k)
                {
                    SCOPED_TRACE("");
                    EXPECT_EQ(ptr2[k].samplingNumber,
                              ptr1[j + k].samplingNumber);
                    ExpectSameOriginMouseStates(ptr2[k], ptr1[j + k]);
                }

                break;
            }
        }

        lastCount = count;
    }

    EXPECT_EQ(::nn::hid::MouseStateCountMax, lastCount);

    ::nnt::hid::FinalizeMouseForAutoPilot();
}

//!< Mouse の入力状態は自動操作状態のホイールの回転差分を正しく反映するか
TEST(MouseAutoPilotSuite, StateReadingTest4)
{
    ::nn::hid::InitializeMouse();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    ::nn::hid::debug::MouseAutoPilotState autoPilotState = {};

    autoPilotState.wheelDelta = 11;

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 3 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 3 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    ::nn::hid::debug::UnsetMouseAutoPilotState();

    int count = ::nn::hid::GetMouseStates(g_States1,
                                          ::nn::hid::MouseStateCountMax);
    EXPECT_EQ(::nn::hid::MouseStateCountMax, count);

    int deltaCount = 0;

    for (int i = 0; i < count; ++i)
    {
        if (g_States1[i].wheelDelta != 0)
        {
            EXPECT_EQ(autoPilotState.wheelDelta, g_States1[i].wheelDelta);
            ++deltaCount;
        }
    }

    // ホイールの回転差分は入力状態に一回だけ反映される
    EXPECT_EQ(2, deltaCount);

    ::nnt::hid::FinalizeMouseForAutoPilot();
}

//!< Mouse の入力状態はカーソル補足直後の移動差分を正しく反映するか
TEST(MouseAutoPilotSuite, StateReadingTest5)
{
    ::nn::hid::InitializeMouse();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    ::nn::hid::debug::MouseAutoPilotState autoPilotState = {};

    autoPilotState.x = 0;
    autoPilotState.y = 0;

    autoPilotState.deltaX = 100;
    autoPilotState.deltaY = 200;
    autoPilotState.x += autoPilotState.deltaX;
    autoPilotState.y += autoPilotState.deltaY;

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 3 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::GetMouseStates(g_States1,
                                          ::nn::hid::MouseStateCountMax);

    EXPECT_EQ(::nn::hid::MouseStateCountMax, count);

    bool checked = false;

    for (int i = 0; i < count - 1; ++i)
    {
        if (!checked && (g_States1[i].x != g_States1[i + 1].x))
        {
            // カーソルを補足した直後の移動差分は無視される
            EXPECT_EQ(100, g_States1[i].x);
            EXPECT_EQ(200, g_States1[i].y);
            EXPECT_EQ(0, g_States1[i].deltaX);
            EXPECT_EQ(0, g_States1[i].deltaY);

            checked = true;
        }
    }

    EXPECT_TRUE(checked);

    ::nn::hid::debug::UnsetMouseAutoPilotState();

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 3 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    count = ::nn::hid::GetMouseStates(g_States1, ::nn::hid::MouseStateCountMax);

    EXPECT_EQ(::nn::hid::MouseStateCountMax, count);

    checked = false;

    for (int i = 0; i < count - 1; ++i)
    {
        if (!checked && (g_States1[i].x != g_States1[i + 1].x))
        {
            // カーソルを喪失した直後の移動差分は無視される
            EXPECT_EQ(0, g_States1[i].x);
            EXPECT_EQ(0, g_States1[i].y);
            EXPECT_EQ(0, g_States1[i].deltaX);
            EXPECT_EQ(0, g_States1[i].deltaY);

            checked = true;
        }
    }

    EXPECT_TRUE(checked);

    ::nnt::hid::FinalizeMouseForAutoPilot();
}

//!< Mouse の入力状態は変域を跨いだカーソル座標の移動差分を正しく反映するか
TEST(MouseAutoPilotSuite, StateReadingTest6)
{
    ::nn::hid::InitializeMouse();

    ::nn::hid::debug::MouseAutoPilotState autoPilotState = {};
    autoPilotState.attributes.Set<::nn::hid::MouseAttribute::IsConnected>();

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    autoPilotState.x = 1279;
    autoPilotState.y = 719;
    autoPilotState.deltaX = 1380;
    autoPilotState.deltaY = 920;

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 3 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::GetMouseStates(g_States1,
                                          ::nn::hid::MouseStateCountMax);

    EXPECT_EQ(::nn::hid::MouseStateCountMax, count);

    bool checked = false;

    for (int i = 1; i < count - 1; ++i)
    {
        if (!checked && (g_States1[i].x != g_States1[i + 1].x))
        {
            EXPECT_EQ(1279, g_States1[i].x);
            EXPECT_EQ(719, g_States1[i].y);
            EXPECT_EQ(1380, g_States1[i].deltaX);
            EXPECT_EQ(920, g_States1[i].deltaY);

            // 移動差分は入力状態に一回だけ反映される
            EXPECT_EQ(1279, g_States1[i - 1].x);
            EXPECT_EQ(719, g_States1[i - 1].y);
            EXPECT_EQ(0, g_States1[i - 1].deltaX);
            EXPECT_EQ(0, g_States1[i - 1].deltaY);

            checked = true;
        }
    }

    EXPECT_TRUE(checked);

    autoPilotState.x = 0;
    autoPilotState.y = 0;
    autoPilotState.deltaX = -1380;
    autoPilotState.deltaY = -920;

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 3 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    count = ::nn::hid::GetMouseStates(g_States1, ::nn::hid::MouseStateCountMax);

    EXPECT_EQ(::nn::hid::MouseStateCountMax, count);

    checked = false;

    for (int i = 1; i < count - 1; ++i)
    {
        if (!checked && (g_States1[i].x != g_States1[i + 1].x))
        {
            EXPECT_EQ(0, g_States1[i].x);
            EXPECT_EQ(0, g_States1[i].y);
            EXPECT_EQ(-1380, g_States1[i].deltaX);
            EXPECT_EQ(-920, g_States1[i].deltaY);

            // 移動差分は入力状態に一回だけ反映される
            EXPECT_EQ(0, g_States1[i - 1].x);
            EXPECT_EQ(0, g_States1[i - 1].y);
            EXPECT_EQ(0, g_States1[i - 1].deltaX);
            EXPECT_EQ(0, g_States1[i - 1].deltaY);

            checked = true;
        }
    }

    EXPECT_TRUE(checked);

    ::nnt::hid::FinalizeMouseForAutoPilot();
}

//!< Mouse の入力状態は変域境界上限のカーソル座標の移動差分を正しく反映するか
TEST(MouseAutoPilotSuite, StateReadingTest7)
{
    ::nn::hid::InitializeMouse();

    ::nn::hid::debug::MouseAutoPilotState autoPilotState = {};
    autoPilotState.attributes.Set<::nn::hid::MouseAttribute::IsConnected>();

    autoPilotState.x = 1279;
    autoPilotState.y = 719;

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    autoPilotState.buttons.Set<::nn::hid::MouseButton::Middle>();
    autoPilotState.deltaX = 100;
    autoPilotState.deltaY = 200;

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 3 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::GetMouseStates(g_States1,
                                          ::nn::hid::MouseStateCountMax);

    EXPECT_EQ(::nn::hid::MouseStateCountMax, count);

    bool checked = false;

    for (int i = 1; i < count - 1; ++i)
    {
        if (!checked && (g_States1[i].buttons != g_States1[i + 1].buttons))
        {
            EXPECT_EQ(1279, g_States1[i].x);
            EXPECT_EQ(719, g_States1[i].y);
            EXPECT_EQ(100, g_States1[i].deltaX);
            EXPECT_EQ(200, g_States1[i].deltaY);

            checked = true;
        }
    }

    EXPECT_TRUE(checked);

    autoPilotState.buttons.Reset<::nn::hid::MouseButton::Middle>();
    autoPilotState.deltaX = -100;
    autoPilotState.deltaY = -200;

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 3 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    count = ::nn::hid::GetMouseStates(g_States1, ::nn::hid::MouseStateCountMax);

    EXPECT_EQ(::nn::hid::MouseStateCountMax, count);

    checked = false;

    for (int i = 1; i < count - 1; ++i)
    {
        if (!checked && (g_States1[i].buttons != g_States1[i + 1].buttons))
        {
            EXPECT_EQ(1279, g_States1[i].x);
            EXPECT_EQ(719, g_States1[i].y);
            EXPECT_EQ(-100, g_States1[i].deltaX);
            EXPECT_EQ(-200, g_States1[i].deltaY);

            checked = true;
        }
    }

    EXPECT_TRUE(checked);

    ::nnt::hid::FinalizeMouseForAutoPilot();
}

//!< Mouse の入力状態は変域境界下限のカーソル座標の移動差分を正しく反映するか
TEST(MouseAutoPilotSuite, StateReadingTest8)
{
    ::nn::hid::InitializeMouse();

    ::nn::hid::debug::MouseAutoPilotState autoPilotState = {};
    autoPilotState.attributes.Set<::nn::hid::MouseAttribute::IsConnected>();

    autoPilotState.x = 0;
    autoPilotState.y = 0;

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    autoPilotState.buttons.Set<::nn::hid::MouseButton::Middle>();
    autoPilotState.deltaX = -100;
    autoPilotState.deltaY = -200;

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 3 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::GetMouseStates(g_States1,
                                          ::nn::hid::MouseStateCountMax);

    EXPECT_EQ(::nn::hid::MouseStateCountMax, count);

    bool checked = false;

    for (int i = 1; i < count - 1; ++i)
    {
        if (!checked && (g_States1[i].buttons != g_States1[i + 1].buttons))
        {
            EXPECT_EQ(0, g_States1[i].x);
            EXPECT_EQ(0, g_States1[i].y);
            EXPECT_EQ(-100, g_States1[i].deltaX);
            EXPECT_EQ(-200, g_States1[i].deltaY);

            checked = true;
        }
    }

    EXPECT_TRUE(checked);

    autoPilotState.buttons.Reset<::nn::hid::MouseButton::Middle>();
    autoPilotState.deltaX = 100;
    autoPilotState.deltaY = 200;

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 3 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    count = ::nn::hid::GetMouseStates(g_States1, ::nn::hid::MouseStateCountMax);

    EXPECT_EQ(::nn::hid::MouseStateCountMax, count);

    checked = false;

    for (int i = 1; i < count - 1; ++i)
    {
        if (!checked && (g_States1[i].buttons != g_States1[i + 1].buttons))
        {
            EXPECT_EQ(0, g_States1[i].x);
            EXPECT_EQ(0, g_States1[i].y);
            EXPECT_EQ(100, g_States1[i].deltaX);
            EXPECT_EQ(200, g_States1[i].deltaY);

            checked = true;
        }
    }

    EXPECT_TRUE(checked);

    ::nnt::hid::FinalizeMouseForAutoPilot();
}

//!< Mouse の入力状態はカーソル座標の実際よりも大きな移動差分を正しく反映するか
TEST(MouseAutoPilotSuite, StateReadingTest9)
{
    ::nn::hid::InitializeMouse();

    ::nn::hid::debug::MouseAutoPilotState autoPilotState = {};
    autoPilotState.attributes.Set<::nn::hid::MouseAttribute::IsConnected>();

    autoPilotState.x = 100;
    autoPilotState.y = 200;

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    autoPilotState.x = 200;
    autoPilotState.y = 400;
    autoPilotState.deltaX = 300;
    autoPilotState.deltaY = 400;

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 3 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::GetMouseStates(g_States1,
                                          ::nn::hid::MouseStateCountMax);

    EXPECT_EQ(::nn::hid::MouseStateCountMax, count);

    bool checked = false;

    for (int i = 1; i < count - 1; ++i)
    {
        if (!checked && (g_States1[i].x != g_States1[i + 1].x))
        {
            EXPECT_EQ(200, g_States1[i].x);
            EXPECT_EQ(400, g_States1[i].y);
            EXPECT_EQ(100, g_States1[i].deltaX);
            EXPECT_EQ(200, g_States1[i].deltaY);

            checked = true;
        }
    }

    EXPECT_TRUE(checked);

    autoPilotState.x = 50;
    autoPilotState.y = 100;
    autoPilotState.deltaX = -200;
    autoPilotState.deltaY = -400;

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 3 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    count = ::nn::hid::GetMouseStates(g_States1, ::nn::hid::MouseStateCountMax);

    EXPECT_EQ(::nn::hid::MouseStateCountMax, count);

    checked = false;

    for (int i = 1; i < count - 1; ++i)
    {
        if (!checked && (g_States1[i].x != g_States1[i + 1].x))
        {
            EXPECT_EQ(50, g_States1[i].x);
            EXPECT_EQ(100, g_States1[i].y);
            EXPECT_EQ(-150, g_States1[i].deltaX);
            EXPECT_EQ(-300, g_States1[i].deltaY);

            checked = true;
        }
    }

    EXPECT_TRUE(checked);

    ::nnt::hid::FinalizeMouseForAutoPilot();
}

//!< Mouse の入力状態はカーソル座標の実際よりも小さな移動差分を正しく反映するか
TEST(MouseAutoPilotSuite, StateReadingTest10)
{
    ::nn::hid::InitializeMouse();

    ::nn::hid::debug::MouseAutoPilotState autoPilotState = {};
    autoPilotState.attributes.Set<::nn::hid::MouseAttribute::IsConnected>();

    autoPilotState.x = 100;
    autoPilotState.y = 200;

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    autoPilotState.x = 200;
    autoPilotState.y = 400;
    autoPilotState.deltaX = 50;
    autoPilotState.deltaY = 100;

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 3 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    int count = ::nn::hid::GetMouseStates(g_States1,
                                          ::nn::hid::MouseStateCountMax);

    EXPECT_EQ(::nn::hid::MouseStateCountMax, count);

    bool checked = false;

    for (int i = 1; i < count - 1; ++i)
    {
        if (!checked && (g_States1[i].x != g_States1[i + 1].x))
        {
            EXPECT_EQ(200, g_States1[i].x);
            EXPECT_EQ(400, g_States1[i].y);
            EXPECT_EQ(100, g_States1[i].deltaX);
            EXPECT_EQ(200, g_States1[i].deltaY);

            checked = true;
        }
    }

    EXPECT_TRUE(checked);

    autoPilotState.x = 50;
    autoPilotState.y = 100;
    autoPilotState.deltaX = -10;
    autoPilotState.deltaY = -50;

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 3 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    count = ::nn::hid::GetMouseStates(g_States1, ::nn::hid::MouseStateCountMax);

    EXPECT_EQ(::nn::hid::MouseStateCountMax, count);

    checked = false;

    for (int i = 1; i < count - 1; ++i)
    {
        if (!checked && (g_States1[i].x != g_States1[i + 1].x))
        {
            EXPECT_EQ(50, g_States1[i].x);
            EXPECT_EQ(100, g_States1[i].y);
            EXPECT_EQ(-150, g_States1[i].deltaX);
            EXPECT_EQ(-300, g_States1[i].deltaY);

            checked = true;
        }
    }

    EXPECT_TRUE(checked);

    ::nnt::hid::FinalizeMouseForAutoPilot();
}

//!< Mouse の入力状態は切断エミュレーションの設定を正しく反映するか
TEST(MouseAutoPilotSuite, StateReadingTest11)
{
    ::nn::hid::InitializeMouse();

    ::nn::hid::debug::MouseAutoPilotState autoPilotState = {};

    ::nn::hid::debug::SetMouseAutoPilotState(autoPilotState);

    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            ::nn::hid::MouseStateCountMax / 2 *
            ::nnt::hid::GetMouseSamplingInterval().GetMilliSeconds()));

    ::nn::hid::MouseState& state = g_States1[0];

    ::nn::hid::GetMouseState(&state);

    EXPECT_FALSE(
        state.attributes.Test<::nn::hid::MouseAttribute::IsConnected>());

    EXPECT_TRUE(
        state.attributes.Test<::nn::hid::MouseAttribute::Transferable>());

    ::nnt::hid::FinalizeMouseForAutoPilot();
}

namespace
{

void FuzzMouseAutoPilotState(
    ::nn::hid::debug::MouseAutoPilotState* pState, int seed) NN_NOEXCEPT
{
    pState->x = seed * 3;
    pState->y = seed * 5;
    pState->deltaX = 0;
    pState->deltaY = 0;
    pState->wheelDelta = seed * 7;
    pState->buttons.Reset();
    pState->buttons.Set(seed % (::nn::hid::MouseButton::Back::Index + 1), true);
    pState->attributes.Reset();
    pState->attributes.Set<::nn::hid::MouseAttribute::IsConnected>();
    pState->attributes.Set<::nn::hid::MouseAttribute::Transferable>(
        pState->buttons.Test<::nn::hid::MouseButton::Middle>());
}

void ExpectDefaultMouseState(const ::nn::hid::MouseState& state) NN_NOEXCEPT
{
    ::nn::hid::MouseAttributeSet attributes = state.attributes;

    // Transferable は環境依存性が強いのでここでは確認しない
    attributes.Reset<::nn::hid::MouseAttribute::Transferable>();

    // IsConnected は環境依存性が強いのでここでは確認しない
    attributes.Reset<::nn::hid::MouseAttribute::IsConnected>();

    EXPECT_LE(0, state.samplingNumber);
    EXPECT_EQ(0, state.x);
    EXPECT_EQ(0, state.y);
    EXPECT_EQ(0, state.deltaX);
    EXPECT_EQ(0, state.deltaY);
    EXPECT_EQ(0, state.wheelDelta);
    EXPECT_TRUE(state.buttons.IsAllOff());
    EXPECT_TRUE(attributes.IsAllOff());
}

void ExpectFuzzedMouseState(const ::nn::hid::MouseState& state) NN_NOEXCEPT
{
    // deltaX, deltaY は単体では値が確定しないためここでは確認しない

    EXPECT_LE(0, state.samplingNumber);
    EXPECT_EQ(0, state.x % 3);
    EXPECT_EQ(0, state.y % 5);
    EXPECT_EQ(0, state.wheelDelta % 7);
    const int seed = state.x / 3;
    EXPECT_EQ(seed, state.y / 5);
    EXPECT_EQ(state.wheelDelta == 0 ? 0 : seed, state.wheelDelta / 7);
    ::nn::hid::MouseButtonSet buttons = {};
    buttons.Set(seed % (::nn::hid::MouseButton::Back::Index + 1), true);
    EXPECT_EQ(buttons, state.buttons);
    EXPECT_EQ(state.buttons.Test<::nn::hid::MouseButton::Middle>(),
              state.attributes.Test<::nn::hid::MouseAttribute::Transferable>());
    EXPECT_TRUE(
        state.attributes.Test<::nn::hid::MouseAttribute::IsConnected>());
}

void ExpectSameOriginMouseStates(
    const ::nn::hid::MouseState& lhs,
    const ::nn::hid::MouseState& rhs) NN_NOEXCEPT
{
    // deltaX, deltaY は単体では値が確定しないためここでは確認しない

    EXPECT_EQ(lhs.x, rhs.x);
    EXPECT_EQ(lhs.y, rhs.y);
    EXPECT_EQ(rhs.wheelDelta == 0 ? 0 : lhs.wheelDelta, rhs.wheelDelta);
    EXPECT_EQ(lhs.buttons, rhs.buttons);
    EXPECT_EQ(lhs.attributes, rhs.attributes);
}

void ExpectAdjacentMouseStates(
    const ::nn::hid::MouseState& lhs,
    const ::nn::hid::MouseState& rhs) NN_NOEXCEPT
{
    EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
    EXPECT_GE(lhs.x, rhs.x);
    EXPECT_GE(lhs.y, rhs.y);
}

} // namespace
