﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/debug/hid_Npad.h>
#include <nn/hid/system/hid_Npad.h>
#include <nn/hid/debug/hid_InputDetection.h>
#include <nn/hid/system/hid_InputDetection.h>
#include <nn/hid/hid_Result.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#include <nn/nn_Log.h>

#include "../Common/testHid_InputDetector.h"

namespace {

    // Nx でサポートされているすべての NpadId
    const nn::hid::NpadIdType NpadIds[] = {
        nn::hid::NpadId::No1,
        nn::hid::NpadId::No2,
        nn::hid::NpadId::No3,
        nn::hid::NpadId::No4,
        nn::hid::NpadId::No5,
        nn::hid::NpadId::No6,
        nn::hid::NpadId::No7,
        nn::hid::NpadId::No8,
        nn::hid::NpadId::Handheld,
        nn::hid::system::NpadSystemId::Other,
    };

    const int NpadIdCountMax = sizeof(NpadIds) / sizeof(NpadIds[0]);

    void Initialize()
    {
        nn::hid::InitializeNpad();
        nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);
        nn::hid::system::InitializeInputDetector();
    }

    void Finalize()
    {
        nn::hid::debug::FinalizeInputDetector();
        nn::hid::SetSupportedNpadIdType(nullptr, 0);
        nn::hid::debug::FinalizeNpad();
    }
}

//!< 入力状態に更新が無い場合は timestamp は不変
TEST(InputDetector, NoInputDetection)
{
    Initialize();

    for (const auto& id : nnt::hid::InputSourceIdSets)
    {
        ::nn::hid::system::InputSourceState state1;
        ::nn::hid::system::GetInputSourceState(&state1, id);

        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(1)
        );

        ::nn::hid::system::InputSourceState state2;
        ::nn::hid::system::GetInputSourceState(&state2, id);

        NN_LOG("[Id=%x] TimeStampDelta = %lld\n",
            id,
            state2.timestamp - state1.timestamp);

        EXPECT_EQ(state2.timestamp, state1.timestamp);
    }

    Finalize();
}

//!< 入力状態に更新がある場合は timestamp が変わる
TEST(InputDetector, NotifyInputDetection)
{
    const int64_t DeltaTimeMilliSeconds = 1000; // 計測間隔
    const float PermissibleErrorRate = 0.01f;  // 許容誤差

    Initialize();

    for (const auto& id : nnt::hid::InputSourceIdSets)
    {
        ::nn::hid::system::InputSourceState state = {};

        // 入力変化を通知
        ::nn::hid::system::NotifyInputDetector(id);

        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(DeltaTimeMilliSeconds)
        );

        ::nn::hid::system::GetInputSourceState(&state, id);

        const double ActualDeltaTimeMilliSecondsSinceLastInput = 1000.0 * (static_cast<double>(::nn::os::GetSystemTick().GetInt64Value() - state.timestamp) / nn::os::GetSystemTickFrequency());

        NN_LOG("[Id=%x] ActualDeltaTimeMilliSecondsSinceLastInput = %5.2f\n",
            id,
            ActualDeltaTimeMilliSecondsSinceLastInput);

        EXPECT_NEAR(
            ActualDeltaTimeMilliSecondsSinceLastInput,
            static_cast<double>(DeltaTimeMilliSeconds),
            DeltaTimeMilliSeconds * PermissibleErrorRate
        );
    }

    Finalize();
}
