﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/hid.h>
#include <nn/hid/hid_SevenSixAxisSensor.h>
#include <nn/hid/debug/hid_ConsoleSixAxisSensor.h>
#include <nn/hid/debug/hid_SevenSixAxisSensor.h>
#include <nn/os.h>
#include <nnt/nntest.h>

#include "../Common/testHid_ConsoleSixAxisSensor.h"

#define VERBOSE

namespace
{

const int StateCountMax = ::nn::hid::SevenSixAxisSensorStateCountMax;
::nn::hid::SevenSixAxisSensorState g_States[StateCountMax];

const int CountStateCountMax = ::nn::hid::debug::ConsoleSixAxisSensorCountStateCountMax;
::nn::hid::debug::ConsoleSixAxisSensorCountState g_CountStates[CountStateCountMax];


// 最大サイズのバッファを用意しておく
NN_ALIGNAS(0x1000)
static uint8_t s_WorkBuffer[::nn::hid::SevenSixAxisSensorWorkBufferSize];

//!< 指定の優先度のスレッドから入力状態の読み出しが可能かテストします。
void ThreadingTest(int priority);

} // namespace

//!< 入力状態の読み出し時にバッファは正しく利用されるか
TEST(SevenSixAxisSensorReadingStateSuite, ReadingStateTest1)
{
    for (::nn::hid::SevenSixAxisSensorState& state : g_States)
    {
        state = ::nn::hid::SevenSixAxisSensorState();
        state.samplingNumber = 0xDEADBEEF;
    }

    ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                            ::nn::hid::SevenSixAxisSensorWorkBufferSize);

    ::nn::hid::StartSevenSixAxisSensor();

    // バッファの半分だけセンサー値が溜まるまで待つ
    ::nn::os::SleepThread(
        ::nn::TimeSpan::FromMilliSeconds(
            StateCountMax / 2 * ::nnt::hid::GetConsoleSixAxisSensorDriverSamplingInterval().GetMilliSeconds()
        )
    );

    int count = ::nn::hid::GetSevenSixAxisSensorStates(g_States, StateCountMax);
    EXPECT_GE(StateCountMax, count);

    for (int i = count; i < StateCountMax; ++i)
    {
        // 入力状態の書き込まれなかった領域は値を維持する。
        EXPECT_EQ(0xDEADBEEF, g_States[i].samplingNumber);
    }

    ::nn::hid::StopSevenSixAxisSensor();

    ::nn::hid::FinalizeSevenSixAxisSensor();
}

//!< 読みだした入力状態の値は適切か
TEST(SevenSixAxisSensorReadingStateSuite, ReadingStateTest2)
{
    for (::nn::hid::SevenSixAxisSensorState& state : g_States)
    {
        state = ::nn::hid::SevenSixAxisSensorState();
    }

    ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                            ::nn::hid::SevenSixAxisSensorWorkBufferSize);

    ::nn::hid::StartSevenSixAxisSensor();

    ::nnt::hid::SleepThreadForSampleCount(StateCountMax);

    const int ReadTestCount = 5; // 読み出し試行回数

    for (int i = 0; i < ReadTestCount; i++)
    {
        int count = ::nn::hid::GetSevenSixAxisSensorStates(g_States, StateCountMax);

        EXPECT_GE(StateCountMax, count);

        // 値の妥当性チェック
        for (int i = 0; i < count; i++)
        {
            ::nnt::hid::AssertIfNotAtRest<::nn::hid::SevenSixAxisSensorState>(g_States[i]);
        }

#ifdef VERBOSE
        NN_LOG("[%lld] Acc : (%f, %f, %f), Gyro : (%f, %f, %f)\n",
            g_States[0].samplingNumber,
            g_States[0].acceleration.x,
            g_States[0].acceleration.y,
            g_States[0].acceleration.z,
            g_States[0].angularVelocity.x,
            g_States[0].angularVelocity.y,
            g_States[0].angularVelocity.z
        );
#endif // VERBOSE

        auto tempState = g_States[0];
        for (int i = 1; i < count; i++)
        {
            // サンプル番号の連続性
            EXPECT_EQ(tempState.samplingNumber - 1, g_States[i].samplingNumber);

            tempState = g_States[i];
        }

        // 1 周期待つ
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                ::nnt::hid::GetConsoleSixAxisSensorSamplingInterval().GetMilliSeconds()
            )
        );
    }

    ::nn::hid::StopSevenSixAxisSensor();

    ::nn::hid::FinalizeSevenSixAxisSensor();
}

//!< 読みだした入力状態のサンプリング番号の増加量は適切か
TEST(SevenSixAxisSensorReadingStateSuite, ReadingStateTest3)
{
    for (::nn::hid::SevenSixAxisSensorState& state : g_States)
    {
        state = ::nn::hid::SevenSixAxisSensorState();
    }

    ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                            ::nn::hid::SevenSixAxisSensorWorkBufferSize);

    ::nn::hid::StartSevenSixAxisSensor();

    ::nnt::hid::SleepThreadForSampleCount(StateCountMax);

    // 計測開始時点のサンプリング番号を取得
    auto count = ::nn::hid::GetSevenSixAxisSensorStates(g_States, StateCountMax);

    EXPECT_GE(StateCountMax, count);

    const auto SamplingNumber0 = g_States[0].samplingNumber;

    // 所定時間待つ
    const auto ExpectedSamplingNumberDelta = 1000; //!< 期待されるサンプル数の増加分
    ::nnt::hid::SleepThreadForSampleCount(ExpectedSamplingNumberDelta);

    // 計測終了時点のサンプリング番号を取得
    count = ::nn::hid::GetSevenSixAxisSensorStates(g_States, StateCountMax);

    EXPECT_GE(StateCountMax, count);

    const auto ActualSamplingNumberDelta = g_States[0].samplingNumber - SamplingNumber0;

    NN_LOG("ActualSamplingNumberDelta : %lld\n", ActualSamplingNumberDelta);

    EXPECT_NEAR(ActualSamplingNumberDelta,
                ExpectedSamplingNumberDelta,
                ExpectedSamplingNumberDelta * 0.05f);

    ::nn::hid::StopSevenSixAxisSensor();

    ::nn::hid::FinalizeSevenSixAxisSensor();
}

//!< 読みだした入力状態の値は適切か
TEST(SevenSixAxisSensorReadingStateSuite, ReadingCountStateTest1)
{
    for (::nn::hid::debug::ConsoleSixAxisSensorCountState& state : g_CountStates)
    {
        state = ::nn::hid::debug::ConsoleSixAxisSensorCountState();
    }

    ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                            ::nn::hid::SevenSixAxisSensorWorkBufferSize);

    ::nn::hid::StartSevenSixAxisSensor();

    ::nnt::hid::SleepThreadForSampleCount(StateCountMax);

    const int ReadTestCount = 5; // 読み出し試行回数

    for (int i = 0; i < ReadTestCount; i++)
    {
        int count = ::nn::hid::debug::GetConsoleSixAxisSensorCountStates(g_CountStates, CountStateCountMax);
        NN_UNUSED(count);

        NN_LOG("[%lld] count: %d, Acc : (%d, %d, %d), Gyro : (%d, %d, %d)\n",
            g_CountStates[0].samplingNumber,
            g_CountStates[0].acceleration.x,
            g_CountStates[0].acceleration.y,
            g_CountStates[0].acceleration.z,
            g_CountStates[0].angularVelocity.x,
            g_CountStates[0].angularVelocity.y,
            g_CountStates[0].angularVelocity.z
        );

        // 1 周期待つ
        ::nn::os::SleepThread(
            ::nn::TimeSpan::FromMilliSeconds(
                ::nnt::hid::GetConsoleSixAxisSensorSamplingInterval().GetMilliSeconds()
            )
        );
    }

    ::nn::hid::StopSevenSixAxisSensor();

    ::nn::hid::FinalizeSevenSixAxisSensor();
}

//!< 読みだした入力状態の座標系番号の増加は適切か
TEST(SevenSixAxisSensorReadingStateSuite, ReadingCoordinateNumberTest)
{
    for (::nn::hid::SevenSixAxisSensorState& state : g_States)
    {
        state = ::nn::hid::SevenSixAxisSensorState();
    }

    ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                            ::nn::hid::SevenSixAxisSensorWorkBufferSize);

    const int64_t InvalidCoordinateNumber = -1;
    auto coordinateNumber = InvalidCoordinateNumber;

    const auto TestCount = 5;
    for (int i = 0; i < TestCount; i++)
    {
        ::nn::hid::StartSevenSixAxisSensor();

        ::nnt::hid::SleepThreadForSampleCount(StateCountMax);

        auto count = ::nn::hid::GetSevenSixAxisSensorStates(g_States, StateCountMax);
        EXPECT_GE(count, 1);

        // 座標系番号が増加しているかチェック
        EXPECT_TRUE(coordinateNumber + 1 == g_States[0].coordinateNumber ||
                    coordinateNumber == InvalidCoordinateNumber);

        coordinateNumber = g_States[0].coordinateNumber;

        ::nn::hid::StopSevenSixAxisSensor();

        ::nnt::hid::SleepThreadForSampleCount(StateCountMax);
    }

    ::nn::hid::FinalizeSevenSixAxisSensor();
}

//!< サンプリングを開始までにかかる時間は仕様通りか
TEST(SevenSixAxisSensorReadingStateSuite, StartSamplingTest)
{
    for (::nn::hid::SevenSixAxisSensorState& state : g_States)
    {
        state = ::nn::hid::SevenSixAxisSensorState();
    }

    ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                            ::nn::hid::SevenSixAxisSensorWorkBufferSize);

    ::nn::hid::StartSevenSixAxisSensor();

    ::nnt::hid::SleepThreadForSampleCount(StateCountMax);

    ::nn::hid::StopSevenSixAxisSensor();

    ::nnt::hid::SleepThreadForSampleCount(StateCountMax);

    ::nn::hid::StartSevenSixAxisSensor();
    const auto StartSamplingNumber = g_States[0].samplingNumber;
    const auto StartTick = nn::os::GetSystemTick();

    auto count = ::nn::hid::GetSevenSixAxisSensorStates(g_States, 1);
    EXPECT_GE(count, 1);
    EXPECT_GE(g_States[0].samplingNumber, 0);

    const auto TimeoutMilliSeconds = 25;
    int64_t deltaTimeToStartSampling;
    for(;;)
    {
        ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(1));
        auto count = ::nn::hid::GetSevenSixAxisSensorStates(g_States, 1);
        EXPECT_GE(count, 1);

        deltaTimeToStartSampling = (nn::os::GetSystemTick() - StartTick).ToTimeSpan()
                                                                        .GetMilliSeconds();

        if (g_States[0].samplingNumber > StartSamplingNumber)
        {
            NN_LOG("DeltaTimeToStartSampling[ms]: %lld\n", deltaTimeToStartSampling);
            break;
        }

        if (deltaTimeToStartSampling > TimeoutMilliSeconds)
        {
            break;
        }
    }


    // StartSevenSixAxisSensor() 呼び出し後、DeltaTimeToStartSamplingMax 以内にデータ取得ができる
    // (⇔ サンプリング番号が増加し始める)
    EXPECT_LE(deltaTimeToStartSampling, TimeoutMilliSeconds);

    ::nn::hid::FinalizeSevenSixAxisSensor();
}

//!< サンプリングの経過時間は適切か
TEST(SevenSixAxisSensorReadingStateSuite, DeltaTimeBetweenEachSampleTest)
{
    for (::nn::hid::SevenSixAxisSensorState& state : g_States)
    {
        state = ::nn::hid::SevenSixAxisSensorState();
    }

    ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                            ::nn::hid::SevenSixAxisSensorWorkBufferSize);

    ::nn::hid::StartSevenSixAxisSensor();

    ::nnt::hid::SleepThreadForSampleCount(StateCountMax * 2);

    auto count = ::nn::hid::GetSevenSixAxisSensorStates(g_States, StateCountMax);
    EXPECT_EQ(StateCountMax, count);

    const auto ExpectedDeltaTimeMicroSecondsForSeven = 1000;
    int64_t meanActualDeltaTimeMicroSecondsForSeven = 0;
    for (int i = 0; i < StateCountMax - 1; i++)
    {
        meanActualDeltaTimeMicroSecondsForSeven += (g_States[i].timeStamp - g_States[i + 1].timeStamp).GetMicroSeconds();
    }
    meanActualDeltaTimeMicroSecondsForSeven /= (StateCountMax - 1);

    NN_LOG("MeanActualDeltaTimeMicroSecondsForSeven[us] = %lld\n", meanActualDeltaTimeMicroSecondsForSeven);
    EXPECT_NEAR(ExpectedDeltaTimeMicroSecondsForSeven,
                meanActualDeltaTimeMicroSecondsForSeven,
                ExpectedDeltaTimeMicroSecondsForSeven * 0.1f); // ± 10%


    ::nn::hid::StopSevenSixAxisSensor();

    ::nn::hid::FinalizeSevenSixAxisSensor();
}

//!< サンプリングの経過時間は Stop でクリアされるか
TEST(SevenSixAxisSensorReadingStateSuite, ElapsedTimeClearTest)
{
    for (::nn::hid::SevenSixAxisSensorState& state : g_States)
    {
        state = ::nn::hid::SevenSixAxisSensorState();
    }

    ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                            ::nn::hid::SevenSixAxisSensorWorkBufferSize);

    ::nn::hid::StartSevenSixAxisSensor();

    ::nnt::hid::SleepThreadForSampleCount(1000);

    ::nn::hid::StopSevenSixAxisSensor();

    ::nnt::hid::SleepThreadForSampleCount(StateCountMax);

    auto count = ::nn::hid::GetSevenSixAxisSensorStates(g_States, 1);
    EXPECT_EQ(1, count);

    const auto BeforeStopTimestamp = g_States[0].timeStamp;
    const auto SamplingNumber = g_States[0].samplingNumber;

    ::nn::hid::StartSevenSixAxisSensor();

    for (;;)
    {
        count = ::nn::hid::GetSevenSixAxisSensorStates(g_States, 1);
        EXPECT_EQ(1, count);

        if (g_States[0].samplingNumber > SamplingNumber)
        {
            break;
        }
    }

    NN_LOG("%lld < %lld\n", g_States[0].timeStamp.GetMicroSeconds()
                          , BeforeStopTimestamp.GetMicroSeconds());
    EXPECT_LT(g_States[0].timeStamp, BeforeStopTimestamp); // タイムスタンプのクリアチェック

    ::nn::hid::StopSevenSixAxisSensor();

    ::nn::hid::FinalizeSevenSixAxisSensor();
}

TEST(SevenSixAxisSensorReadingStateSuite, IsAtRestTest)
{
    ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                            ::nn::hid::SevenSixAxisSensorWorkBufferSize);

    ::nn::hid::StartSevenSixAxisSensor();

    // 安定するまで 2 sec 待つ
    ::nnt::hid::SleepThreadForSampleCount(2000);

    EXPECT_TRUE(::nn::hid::IsSevenSixAxisSensorAtRest());

    ::nn::hid::StopSevenSixAxisSensor();

    ::nn::hid::FinalizeSevenSixAxisSensor();
}

//!< 高優先度のスレッドから入力状態の読み出しが可能か
TEST(SevenSixAxisSensorReadingStateSuite, ThreadingTest1)
{
    ThreadingTest(::nn::os::HighestThreadPriority);
}

//!< 低優先度のスレッドから入力状態の読み出しが可能か
TEST(SevenSixAxisSensorReadingStateSuite, ThreadingTest2)
{
    ThreadingTest(::nn::os::LowestThreadPriority);
}

namespace
{

void ThreadingTest(int priority)
{
    ::nn::os::ThreadType* pThread = ::nn::os::GetCurrentThread();

    // 指定の優先度に設定する。
    const int original = ::nn::os::ChangeThreadPriority(pThread, priority);

    ::nn::hid::InitializeSevenSixAxisSensor(s_WorkBuffer,
                                            ::nn::hid::SevenSixAxisSensorWorkBufferSize);

    ::nn::hid::StartSevenSixAxisSensor();

    int lastCount = 0;
    int64_t lastSamplingNumber = 0;

    for (int i = 0; i < 60; ++i)
    {
        int count = ::nn::hid::GetSevenSixAxisSensorStates(g_States, StateCountMax);
        EXPECT_LE(lastCount, count);

        EXPECT_LE(lastSamplingNumber, g_States[0].samplingNumber);

        for (int j = 0; j < count - 1; ++j)
        {
            const ::nn::hid::SevenSixAxisSensorState& lhs = g_States[j];
            const ::nn::hid::SevenSixAxisSensorState& rhs = g_States[j + 1];
            EXPECT_EQ(1, lhs.samplingNumber - rhs.samplingNumber);
        }

        lastCount = count;
        lastSamplingNumber = g_States[0].samplingNumber;

        ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(16));
    }

    EXPECT_EQ(StateCountMax, lastCount);

    ::nn::hid::StopSevenSixAxisSensor();

    ::nn::hid::FinalizeSevenSixAxisSensor();

    // 優先度を元に戻す。
    ::nn::os::ChangeThreadPriority(pThread, original);
}

} // namespace
