﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/grc/grc_Api.h>
#include <nn/grc/grc_Application.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/applet/applet_Apis.h>
#include "testGrc_Macro.h"
#include "testGrc_MemoryManagement.h"

static const int Width = 1280;
static const int Height = 720;

#define NNT_GRC_INITIALIZE_FOR_APPLICATION(aruidValue)  \
    {                                                           \
        nn::applet::AppletResourceUserId nntGrcAruid = {static_cast<uint64_t>(aruidValue)};   \
        nn::sf::SharedPointer<nn::grcsrv::IMovieMaker> pMovieMakerProxy; \
        NNT_GRC_EXPECT_SUCCESS(nn::grc::CreateMovieMakerForApplication(&pMovieMakerProxy, nntGrcAruid, {0x010000000000B1C1ull})); \
        NNT_GRC_EXPECT_SUCCESS(nn::grc::InitializeForApplication(std::move(pMovieMakerProxy))); \
    }

#define NNT_GRC_SCOPED_INITIALIZE_FOR_APPLICATION(aruidValue)   \
    NNT_GRC_INITIALIZE_FOR_APPLICATION(aruidValue)              \
    NN_UTIL_SCOPE_EXIT{ nn::grc::FinalizeForApplication(); };


NNT_GRC_TEST_SERVERSTATE(OffscreenLayer_CreateDestroy)
{
    const int RepeatCount = 100;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::InitializeByHipc());
    NN_UTIL_SCOPE_EXIT{ nn::grc::FinalizeByHipc(); };
    nn::sf::SharedPointer<nn::grcsrv::IOffscreenRecorder> pService;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::CreateOffscreenRecorder(&pService, MakeTransferMemory(GetRecorderFirmwareMemory(), GetRecorderFirmwareMemorySize()), GetRecorderFirmwareMemorySize()));

    for(int i = 0; i < RepeatCount; i++)
    {
        NN_LOG("Loop %d/%d\n", i + 1, RepeatCount);
        uint64_t h = 0;
        NN_LOG("[%d]creating offscreen layer\n", i);
        NNT_GRC_EXPECT_SUCCESS(pService->CreateOffscreenLayer(&h, nn::applet::GetAppletResourceUserId()));
        NN_LOG("[%d]  -> handle = %llu\n", i, h);
        NN_LOG("[%d]destroying offscreen layer\n", i);
        NNT_GRC_EXPECT_SUCCESS(pService->DestroyOffscreenLayer(h));
    }
}

NNT_GRC_TEST_SERVERSTATE(OffscreenLayer_CreateDisconnect)
{
    const int RepeatCount = 100;

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::InitializeByHipc());
    NN_UTIL_SCOPE_EXIT{ nn::grc::FinalizeByHipc(); };

    for(int i = 0; i < RepeatCount; i++)
    {
        NN_LOG("Loop %d/%d\n", i + 1, RepeatCount);
        uint64_t h = 0;
        NN_LOG("[%d]connect\n");
        nn::sf::SharedPointer<nn::grcsrv::IOffscreenRecorder> pService;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::CreateOffscreenRecorder(&pService, MakeTransferMemory(GetRecorderFirmwareMemory(), GetRecorderFirmwareMemorySize()), GetRecorderFirmwareMemorySize()));
        NN_LOG("[%d]creating offscreen layer\n", i);
        NNT_GRC_EXPECT_SUCCESS(pService->CreateOffscreenLayer(&h, nn::applet::GetAppletResourceUserId()));
        NN_LOG("[%d]  -> handle = %llu\n", i, h);
        NN_LOG("[%d]disconnect\n");
        pService.Reset();
    }
}

NNT_GRC_TEST_SERVERSTATE(OffscreenLayer_Dev_OpenClose)
{
    const int RepeatCount = 100;
    uint64_t aruid = __LINE__;
    InitializeGraphics();

    uint64_t h = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::InitializeByHipc());
    NN_UTIL_SCOPE_EXIT{ nn::grc::FinalizeByHipc(); };
    nn::sf::SharedPointer<nn::grcsrv::IOffscreenRecorder> pService;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::CreateOffscreenRecorder(&pService, MakeTransferMemory(GetRecorderFirmwareMemory(), GetRecorderFirmwareMemorySize()), GetRecorderFirmwareMemorySize()));
    NNT_GRC_EXPECT_SUCCESS(pService->CreateOffscreenLayer(&h, {aruid}));
    NN_UTIL_SCOPE_EXIT{ NNT_GRC_EXPECT_SUCCESS(pService->DestroyOffscreenLayer(h)); };

    NNT_GRC_SCOPED_INITIALIZE_FOR_APPLICATION(aruid);

    for(int i = 0; i < RepeatCount; i++)
    {
        NN_LOG("Loop %d/%d\n", i + 1, RepeatCount);
        int32_t hProducer = {};
        NNT_GRC_EXPECT_SUCCESS(nn::grc::dev::OpenOffscreenLayer(&hProducer, h));
        nn::grc::dev::CloseOffscreenLayer(h);
    }
}

NNT_GRC_TEST_SERVERSTATE(OffscreenLayer_Dev_RegisterUnregister)
{
    const int RepeatCount = 100;
    uint64_t aruid = __LINE__;
    InitializeGraphics();

    uint64_t h = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::InitializeByHipc());
    NN_UTIL_SCOPE_EXIT{ nn::grc::FinalizeByHipc(); };
    nn::sf::SharedPointer<nn::grcsrv::IOffscreenRecorder> pService;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::CreateOffscreenRecorder(&pService, MakeTransferMemory(GetRecorderFirmwareMemory(), GetRecorderFirmwareMemorySize()), GetRecorderFirmwareMemorySize()));
    NNT_GRC_EXPECT_SUCCESS(pService->CreateOffscreenLayer(&h, {aruid}));
    NN_UTIL_SCOPE_EXIT{ NNT_GRC_EXPECT_SUCCESS(pService->DestroyOffscreenLayer(h)); };

    NNT_GRC_SCOPED_INITIALIZE_FOR_APPLICATION(aruid);

    int32_t hProducer = {};
    NNT_GRC_EXPECT_SUCCESS(nn::grc::dev::OpenOffscreenLayer(&hProducer, h));
    NN_UTIL_SCOPE_EXIT{ nn::grc::dev::CloseOffscreenLayer(h); };


    for(int i = 0; i < RepeatCount; i++)
    {
        NN_LOG("Loop %d/%d\n", i + 1, RepeatCount);

        NNT_GRC_EXPECT_SUCCESS(nn::grc::dev::RegisterOffscreenLayer(h, hProducer, Width, Height));
        nn::grc::dev::UnregisterOffscreenLayer(h);
    }
}

NNT_GRC_TEST_SERVERSTATE(OffscreenLayer_Api_OpenClose)
{
    const int RepeatCount = 100;
    uint64_t aruid = __LINE__;
    InitializeGraphics();

    uint64_t h = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::InitializeByHipc());
    NN_UTIL_SCOPE_EXIT{ nn::grc::FinalizeByHipc(); };
    nn::sf::SharedPointer<nn::grcsrv::IOffscreenRecorder> pService;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::CreateOffscreenRecorder(&pService, MakeTransferMemory(GetRecorderFirmwareMemory(), GetRecorderFirmwareMemorySize()), GetRecorderFirmwareMemorySize()));
    NNT_GRC_EXPECT_SUCCESS(pService->CreateOffscreenLayer(&h, {aruid}));
    NN_UTIL_SCOPE_EXIT{ NNT_GRC_EXPECT_SUCCESS(pService->DestroyOffscreenLayer(h)); };

    NNT_GRC_SCOPED_INITIALIZE_FOR_APPLICATION(aruid);

    for(int i = 0; i < RepeatCount; i++)
    {
        NN_LOG("Loop %d/%d\n", i + 1, RepeatCount);
        nn::grc::MovieLayer* pLayer = nullptr;
        NNT_GRC_EXPECT_SUCCESS(nn::grc::OpenMovieLayer(&pLayer, h, Width, Height));
        nn::grc::CloseMovieLayer(pLayer);
    }
}

NNT_GRC_TEST_SERVERSTATE(OffscreenLayer_Api_OpenFinalize)
{
    const int RepeatCount = 100;
    uint64_t aruid = 0x123123;
    InitializeGraphics();

    uint64_t h = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::InitializeByHipc());
    NN_UTIL_SCOPE_EXIT{ nn::grc::FinalizeByHipc(); };
    nn::sf::SharedPointer<nn::grcsrv::IOffscreenRecorder> pService;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::grc::CreateOffscreenRecorder(&pService, MakeTransferMemory(GetRecorderFirmwareMemory(), GetRecorderFirmwareMemorySize()), GetRecorderFirmwareMemorySize()));
    NNT_GRC_EXPECT_SUCCESS(pService->CreateOffscreenLayer(&h, {aruid}));
    NN_UTIL_SCOPE_EXIT{ NNT_GRC_EXPECT_SUCCESS(pService->DestroyOffscreenLayer(h)); };


    for(int i = 0; i < RepeatCount; i++)
    {
        NN_LOG("Loop %d/%d\n", i + 1, RepeatCount);
        NNT_GRC_INITIALIZE_FOR_APPLICATION(aruid);
        nn::grc::MovieLayer* pLayer = nullptr;
        NNT_GRC_EXPECT_SUCCESS(nn::grc::OpenMovieLayer(&pLayer, h, Width, Height));
        nn::grc::FinalizeForApplication();
    }
}

