﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// 負荷計測機能を有効にします。
#define NN_PERF_PROFILE_ENABLED

#include <cstdlib>

#include <nn/gfx.h>

#include <nn/vi.h>

#include <nn/nn_Assert.h>

#include <nn/ui2d.h>
#include <nn/font.h>
#include <nn/font/font_ScalableFontUtil.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nn/util/util_Vector.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_FormatString.h>
#include <nn/mem.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/time/time_CalendarTime.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_LocationName.h>
#include <nn/oe.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#endif

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/init.h>
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nv/nv_MemoryManagement.h>
#endif

#include "testUi2d_SimpleAllocator.h"

#include <nn/perf.h>

#include <nnt.h>
#include <nnt/graphics/testGraphics_Path.h>
#include <nnt/graphics/testGraphics_GetHostExecutableFilepath.h>
#include <nnt/graphics/testGraphics_CreateDirectories.h>
#include <nnt/graphics/testGraphics_InitializeAllocatorFunctionForStandardAllocator.h>
#include <nnt/graphics/testGraphics_PerformanceProfileData.h>

#include "testUi2d_Dataset.h"

//------------------------------------------------------------------------------
//  変数の宣言
//------------------------------------------------------------------------------
// アプリケーションヒープとして確保されるメモリ領域の大きさ(byte)
static const size_t ApplicationHeapSize = 384 * 1024 * 1024;
// 固定コンスタントバッファのサイズ
static const uint32_t ConstantBufferSize = 256 * 1024 * 1024;
// アプリケーションヒープ
static SimpleAllocator  g_Allocator;

// 画面サイズ
static const int DisplayWidth = 1280;
static const int DisplayHeight = 720;

// ui2d レイアウト
static nn::ui2d::Layout** g_pLayouts = NULL;
// ui2d アーカイブリソースアクセッサ
static nn::ui2d::ArcResourceAccessor* g_pArcResourceAccessor = NULL;
// レイアウトアーカイブバイナリデータ
static void* g_pLayoutArchiveBinary;
// ui2d が描画に使用するグラフィックスリソース
static nn::ui2d::GraphicsResource g_GraphicsResource;

// ui2d 描画の際に使用するコンスタントバッファ
static nn::font::GpuBuffer  g_Ui2dConstantBuffer;
// フォントの描画に使用するコンスタントバッファ
static nn::font::GpuBuffer  g_Ui2dFontConstantBuffer;
// ui2d のコンスタントバッファのダブルバッファリング用インデックス
static uint32_t     g_Ui2dConstantBufferIndex = 0;
// ui2d 描画情報
static nn::ui2d::DrawInfo g_DrawInfo;

// アニメーションコントロール用の ui2d アニメータ
static nn::ui2d::GroupAnimator* g_pAnimator;
// プロファイラで使用するメモリ
static void*  g_pMeterMemory;

// グラフィックスフレームワーク
nns::gfx::GraphicsFramework                 g_GfxFramework;
ptrdiff_t                                   g_offsetToUi2dDrawMemoryPool;
ptrdiff_t                                   g_offsetToLoadMeterMemoryPool;

static const char* LayoutArchivePath = "Contents:/Resources/testUi2d_PerformanceProfile.arc";
static const char* FontDirectoryPath = "Contents:/Resources";

static nnt::ui2d::Dataset g_Dataset;

static const int MaxNumberOfFonts = 256;
int g_NumberOfFonts = 0;
nn::font::ResFont* g_Fonts[MaxNumberOfFonts];

static const int MaxNumberOfFcpxs = 256;
int g_NumberOfFcpxs = 0;
int g_FontFaceHead[MaxNumberOfFcpxs];
nn::font::Font* g_Fcpxs[MaxNumberOfFcpxs];
nn::font::TextureCache::InitializeArg* g_pTextureCacheInitializeArg = NULL;
nn::font::TextureCache* g_pTextureCache = NULL;

static const int g_FileCount = 4;
static const char* g_FileNames[g_FileCount] =
{
    "sample.bffnt",
    "nintendo_NTLG-DB_002.bfttf",
    "color.bffnt",
    "IconFont-Regular.ttf",
};
static size_t g_FontBinarySize[g_FileCount] = { 0 };
static void* g_pFontBinary[g_FileCount] = { NULL };
static void* g_pFontRawResource[g_FileCount] = { NULL };

// ユーザー負荷メーターの種類
enum UserLoadMeterType
{
    UserLoadMeterType_Frame = 0,        //< １フレームの CPU 負荷計測用メーター
    UserLoadMeterType_Initialization,   //< レイアウトデータの初期化処理計測メーター
    UserLoadMeterType_Max
};

//------------------------------------------------------------------------------
//  アプリケーションヒープからメモリを確保する
//------------------------------------------------------------------------------
static void* AllocateFromApplicationHeap(size_t size)
{
    return g_Allocator.AllocateMemory(size, 4);
}

static void* AllocateFromApplicationHeap(size_t size, size_t alignment)
{
    return g_Allocator.AllocateMemory(size, alignment);
}

//------------------------------------------------------------------------------
//  アプリケーションヒープから確保したメモリを開放する
//------------------------------------------------------------------------------
static void DeallocateApplicationHeap(void* ptr)
{
    if (ptr == NULL)
    {
        return ;
    }

    g_Allocator.FreeMemory(ptr);
}

//------------------------------------------------------------------------------
//  計測データ処理のためのメモリ確保関数
//------------------------------------------------------------------------------
static void* AllocateProfilingMemory(size_t size)
{
    return g_Allocator.AllocateMemoryWithoutLogging(size, 4);
}

static void DeallocateProfilingMemory(void* ptr, size_t size)
{
    NN_UNUSED(size);

    g_Allocator.FreeMemoryWithoutLogging(ptr);
}


template <typename TObject>
TObject* AllocAndConstruct()
{
    return new(AllocateFromApplicationHeap(sizeof(TObject))) TObject();
}

template <typename TObject>
void DestructAndFree(TObject* ptr)
{
    ptr->TObject::~TObject();
    DeallocateApplicationHeap(ptr);
}

//------------------------------------------------------------------------------
// ui2d 用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* Ui2dAllocateFunction(size_t size, size_t alignment, void* pUserData)
{
    NN_UNUSED(pUserData);
    return AllocateFromApplicationHeap(size, alignment);
}

static void Ui2dDeallocateFunction(void* ptr, void* pUserData)
{
    NN_UNUSED(pUserData);
    DeallocateApplicationHeap(ptr);
}

//------------------------------------------------------------------------------
// nn::fs用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* FsAllocateFunction( size_t size )
{
    return malloc(size);
}

static void FsDeallocateFunction( void* ptr, size_t size )
{
    NN_UNUSED( size );
    free(ptr);
}

//------------------------------------------------------------------------------

bool RegisterSlotForTexture_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);

        const int slotIndex = graphicsFramework->AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1 );
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);

        pPool->BeginUpdate();
        {
            pPool->SetTextureView(slotIndex, &textureView);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(pDstSlot, slotIndex);
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------

bool RegisterSlotForSampler_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);

        const int slotIndex = graphicsFramework->AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, 1 );
        if (slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex)
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);

        pPool->BeginUpdate();
        {
            pPool->SetSampler(slotIndex, &sampler);
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot(pDstSlot, slotIndex);
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------

void UnregisterSlotForSampler_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    NN_UNUSED(sampler);
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler);
        const int slotIndex = pPool->GetDescriptorSlotIndex(*pDstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, slotIndex);
    }

}

//------------------------------------------------------------------------------

void UnregisterSlotForTexture_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    NN_UNUSED(textureView);
    if (pUserData)
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>(pUserData);
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView);
        int slotIndex = pPool->GetDescriptorSlotIndex(*pDstSlot);
        graphicsFramework->FreeDescriptorSlot(nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, slotIndex);
    }
}

//------------------------------------------------------------------------------
// メモリを確保してファイルを読み込みます
//------------------------------------------------------------------------------
void* ReadFileWithAllocate(const char* pFileName, size_t alignment)
{
    nn::Result result;
    nn::fs::FileHandle fileHandle;
    int64_t fileSize;

    NN_LOG("Loading file: %s ...", pFileName);

    result = nn::fs::OpenFile(&fileHandle, pFileName, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::fs::GetFileSize(&fileSize, fileHandle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    //void* binary = AllocateFromApplicationHeap(static_cast<size_t>(fileSize), alignment);
    void* binary = g_Allocator.AllocateMemoryWithoutLogging(static_cast<size_t>(fileSize), alignment);
    size_t readSize;
    result = nn::fs::ReadFile(&readSize, fileHandle, 0, binary, static_cast<size_t>(fileSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::fs::CloseFile(fileHandle);

    NN_LOG("ok\n", pFileName);

    return binary;
}

//-----------------------------------------------------------------------------
// ファイルシステムの初期化
//-----------------------------------------------------------------------------
void InitializeFileSystem()
{
    nn::fs::SetAllocator( FsAllocateFunction, FsDeallocateFunction );

    nn::Result result;

    nnt::graphics::Path contentsPath(nnt::graphics::GetHostExecutableFilepath());
    contentsPath.MakeParent();
    contentsPath.MakeParent();
    contentsPath.CombineAssign("Contents");

#if defined(NN_BUILD_APISET_NX)
    contentsPath.CombineAssign("NX");
#elif defined(NN_BUILD_CONFIG_SPEC_GENERIC)
    contentsPath.CombineAssign("Generic");
#else
    NN_ABORT("Unexpected spec.");
#endif
    NN_LOG("Mount Contents: %s\n", contentsPath.GetString());
    result = nn::fs::MountHost("Contents", contentsPath.GetString());
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nnt::graphics::Path outputsPath(nnt::graphics::GetHostExecutableFilepath());
    outputsPath.MakeParent();
    NN_LOG("Mount Outputs: %s\n", outputsPath.GetString());
    result = nn::fs::MountHost("Outputs", outputsPath.GetString());
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    result = nnt::graphics::CreateDirectories("Outputs:/Outputs");
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    NN_UNUSED(result);
}

//------------------------------------------------------------------------------
// フォントの初期化処理
//------------------------------------------------------------------------------
static void RegisterDatasetFonts(const char* fontname, void* param)
{
    NN_UNUSED(param);
    NN_ASSERT(g_NumberOfFonts + 1 < MaxNumberOfFonts);

    nn::gfx::Device*    pDevice = g_GfxFramework.GetDevice();

    // フォントの初期化
    nn::font::ResFont* pFont = AllocAndConstruct<nn::font::ResFont>();

    // フォントの読み込み
    nnt::graphics::Path fontPath(FontDirectoryPath);
    fontPath.CombineAssign(fontname);

    void* pFontBinary = ReadFileWithAllocate(fontPath.GetString(), nn::font::ResourceAlignment);
    bool    result = pFont->SetResource(pDevice, pFontBinary);
    NN_ASSERT(result);
    pFont->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture_, &g_GfxFramework);

    g_pArcResourceAccessor->RegisterFont(fontname, pFont);

    g_Fonts[g_NumberOfFonts] = pFont;
    g_NumberOfFonts++;
}

//------------------------------------------------------------------------------
// フォント名からフォントデータを探すコールバック関数
//------------------------------------------------------------------------------
static void* AcquireFontFunctionForComplexFont(size_t* pOutFontDataSize, const char* pFontName, nn::ui2d::ResType resType, void* pUserData)
{
    for (int i = 0; i < g_FileCount; i++)
    {
        // フォントを直接ロードする場合
        if (strcmp(pFontName, g_FileNames[i]) == 0)
        {
            if (g_pFontBinary[i] != NULL)
            {
                *pOutFontDataSize = g_FontBinarySize[i];
                return g_pFontBinary[i];
            }
            else
            {
                nn::Result result;
                nn::fs::FileHandle fileHandle;
                int64_t fileSize;

                char path[256];
                strcpy(path, "Contents:/Resources/");
                strcat(path, g_FileNames[i]);
                result = nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                result = nn::fs::GetFileSize(&fileSize, fileHandle);
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                void* pFontFileImage = Ui2dAllocateFunction(static_cast<size_t>(fileSize), nn::font::ResourceAlignment, NULL);
                g_pFontRawResource[i] = pFontFileImage;

                size_t readSize;
                result = nn::fs::ReadFile(&readSize, fileHandle, 0, pFontFileImage, static_cast<size_t>(fileSize));
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                g_FontBinarySize[i] = static_cast<size_t>(fileSize);
                g_pFontBinary[i] = pFontFileImage;

                nn::fs::CloseFile(fileHandle);

                NN_ASSERT_NOT_NULL(g_pFontBinary[i]);
                *pOutFontDataSize = g_FontBinarySize[i];
                return g_pFontBinary[i];
            }
        }
    }

    // リソース内からデータを探す場合
    {
        nn::ui2d::ResourceAccessor* pResAccsessor = reinterpret_cast<nn::ui2d::ResourceAccessor*>(pUserData);
        void* pFont = pResAccsessor->FindResourceByName(pOutFontDataSize, resType, pFontName);
        NN_ASSERT(pFont != NULL, "The font resource [%s] was not found.\n", pFontName);
        return pFont;
    }
}

//------------------------------------------------------------------------------
// fcpx の読み込み
//------------------------------------------------------------------------------
static void LoadDatasetFcpxs(const char* fcpxname, void* param)
{
    NN_UNUSED(param);
    NN_ASSERT(g_NumberOfFcpxs + 1 < MaxNumberOfFcpxs);

    char name[256];
    strcpy(name, fcpxname);
    strcat(name, ".bfcpx");
    const void* pFcpxData = g_pArcResourceAccessor->FindResourceByName(nn::ui2d::ResourceTypeComplexFont, name);

    if (g_pTextureCacheInitializeArg == NULL)
    {
        // fcpx が 1 つでもあれば InitializeArg を初期化する
        g_pTextureCacheInitializeArg = static_cast<nn::font::TextureCache::InitializeArg*>(nn::ui2d::Layout::AllocateMemory(sizeof(nn::font::TextureCache::InitializeArg)));
        g_pTextureCacheInitializeArg->SetDefault();
        g_pTextureCacheInitializeArg->workMemorySize = 1024 * 1024;
        g_pTextureCacheInitializeArg->noPlotWorkMemorySize = 1024 * 1024;
        g_pTextureCacheInitializeArg->allocateFunction = nn::ui2d::Layout::GetAllocateFunction();
        g_pTextureCacheInitializeArg->textureCacheWidth = 1024;
        g_pTextureCacheInitializeArg->textureCacheHeight = 1024;
    }

    // ユーティリティ関数を使って bfcpx から TextureCacheArg を構築する
    g_FontFaceHead[g_NumberOfFcpxs] = nn::ui2d::ComplexFontHelper::SetupTextureCacheArg(
        g_pTextureCacheInitializeArg,
        AcquireFontFunctionForComplexFont,
        g_pArcResourceAccessor,
        pFcpxData);
    g_NumberOfFcpxs++;
}

//------------------------------------------------------------------------------
// TextureCache の初期化
//------------------------------------------------------------------------------
static void InitializeTextureCache()
{
    if (g_pTextureCacheInitializeArg == NULL)
    {
        // fcpx が 1 つもない場合は TextureCache の初期化をスキップする
        return;
    }

    nn::gfx::Device*    pDevice = g_GfxFramework.GetDevice();

    g_pTextureCache = AllocAndConstruct<nn::font::TextureCache>();
    g_pTextureCache->Initialize(pDevice, *g_pTextureCacheInitializeArg);
    g_pTextureCache->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture_, &g_GfxFramework);

    nn::ui2d::Layout::FreeMemory(g_pTextureCacheInitializeArg);
    g_pTextureCacheInitializeArg = NULL;
    g_NumberOfFcpxs = 0;
}

//------------------------------------------------------------------------------
// fcpx の登録
//------------------------------------------------------------------------------
static void RegisterDatasetFcpxs(const char* fcpxname, void* param)
{
    NN_UNUSED(param);
    NN_ASSERT(g_NumberOfFcpxs + 1 < MaxNumberOfFcpxs);

    nn::gfx::Device*    pDevice = g_GfxFramework.GetDevice();

    char name[256];
    strcpy(name, fcpxname);
    strcat(name, ".bfcpx");
    const void* pFcpxData = g_pArcResourceAccessor->FindResourceByName(nn::ui2d::ResourceTypeComplexFont, name);

    g_Fcpxs[g_NumberOfFcpxs] = nn::ui2d::ComplexFontHelper::InitializeComplexFontTree(
        pDevice,
        RegisterSlotForTexture_,
        &g_GfxFramework,
        g_pTextureCache,
        g_FontFaceHead[g_NumberOfFcpxs],
        AcquireFontFunctionForComplexFont,
        g_pArcResourceAccessor,
        pFcpxData);

    strcpy(name, fcpxname);
    strcat(name, ".fcpx");
    g_pArcResourceAccessor->RegisterFont(name, g_Fcpxs[g_NumberOfFcpxs]);

    g_NumberOfFcpxs++;
}

//------------------------------------------------------------------------------
// TextureCache のアップデート
//------------------------------------------------------------------------------
static void UpdateTextureCache(uint32_t drawCount)
{
    if (g_pTextureCache == NULL)
    {
        return;
    }
    for (uint32_t i = 0; i < drawCount; i++)
    {
        nn::font::RegisterGlyphFromTextBoxRecursive(g_pLayouts[i]->GetRootPane());
    }
    g_pTextureCache->UpdateTextureCache();
    g_pTextureCache->CompleteTextureCache();
}

//---------------------------------------------------------------
// 負荷メーターの初期化
//---------------------------------------------------------------
void InitializeLoadMeter()
{
    nn::gfx::Device*    pDevice = g_GfxFramework.GetDevice();

    nn::perf::LoadMeterCenterInfo info;
    info.SetCoreCount(1);
    info.SetUserMeterCount(UserLoadMeterType_Max);
    info.SetCpuBufferCount(2);
    info.SetGpuBufferCount(2);
    info.SetCpuSectionCountMax(32 * 1024);
    info.SetGpuSectionCountMax(64);

    size_t memorySize = NN_PERF_GET_BUFFER_SIZE(info);
    size_t memoryAlignment = NN_PERF_GET_BUFFER_ALIGNMENT();
    g_pMeterMemory = g_Allocator.AllocateMemoryWithoutLogging(memorySize, memoryAlignment);
    size_t memoryPoolSize = NN_PERF_GET_MEMORY_POOL_SIZE(pDevice, info);

    nn::gfx::MemoryPoolInfo memPoolInfo;
    memPoolInfo.SetDefault();
    memPoolInfo.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached );

    const nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType_Data;

    nn::gfx::MemoryPool* pMemoryPool = g_GfxFramework.GetMemoryPool(memoryPoolType);
    g_offsetToLoadMeterMemoryPool    = g_GfxFramework.AllocatePoolMemory(memoryPoolType, memoryPoolSize, NN_PERF_GET_MEMORY_POOL_ALIGNMENT(pDevice, info));

    NN_PERF_INITIALIZE_METER(pDevice, info,
        g_pMeterMemory, memorySize,
        pMemoryPool, g_offsetToLoadMeterMemoryPool, memoryPoolSize);
}

//---------------------------------------------------------------
// 負荷メーターの破棄
//---------------------------------------------------------------
void FinalizeLoadMeter()
{
    nn::gfx::Device*    pDevice = g_GfxFramework.GetDevice();

    g_GfxFramework.FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_Data, g_offsetToLoadMeterMemoryPool);

    NN_PERF_FINALIZE_METER(pDevice);
    g_Allocator.FreeMemoryWithoutLogging(g_pMeterMemory);
}

//------------------------------------------------------------------------------
// ui2d 描画のためのバッファ関連初期化処理
//------------------------------------------------------------------------------
void InitializeUi2dBuffers(nn::ui2d::DrawInfo& drawInfo, const nn::ui2d::BuildResultInformation& buildResultInfomation)
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // バッファの再セットアップ
    // ui2d 描画に使用する各種 gfx バッファのためのメモリプールを作成
    nn::gfx::MemoryPoolInfo memPoolInfo;
    memPoolInfo.SetDefault();
    memPoolInfo.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached);

    // BuildWithName でレイアウトデータをセットアップした際に BuildResultInformation に収集した各種バッファのサイズを満たすメモリプールを作成する。
    const size_t constantBufferTotalSize = buildResultInfomation.requiredUi2dConstantBufferSize + buildResultInfomation.requiredFontConstantBufferSize;
    const size_t memoryPoolGranularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(pDevice, memPoolInfo);
    const size_t memoryPoolSize = nn::util::align_up(constantBufferTotalSize, memoryPoolGranularity);
    const size_t MemoryPoolAlignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, memPoolInfo);

    if (memoryPoolSize > 0)
    {
        const nns::gfx::GraphicsFramework::MemoryPoolType constantBufferType = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;
        const size_t totalMemoryPoolSize = memoryPoolSize * nn::ui2d::ConstantBufferCount;

        nn::gfx::MemoryPool* pMemoryPool = g_GfxFramework.GetMemoryPool(constantBufferType);
        g_offsetToUi2dDrawMemoryPool = g_GfxFramework.AllocatePoolMemory(constantBufferType, totalMemoryPoolSize, MemoryPoolAlignment);

        size_t  bufferOffset = g_offsetToUi2dDrawMemoryPool;

        // ui2d 用のコンスタントバッファを作成する。
        if (buildResultInfomation.requiredUi2dConstantBufferSize > 0)
        {
            nn::font::GpuBuffer::InitializeArg  arg;

            arg.SetBufferSize(buildResultInfomation.requiredUi2dConstantBufferSize);
            arg.SetBufferCount(nn::ui2d::ConstantBufferCount);
            arg.SetGpuAccessFlag(nn::gfx::GpuAccess_ConstantBuffer);
            arg.SetMemoryPool(pMemoryPool);
            arg.SetMemoryPoolOffset(bufferOffset);
            arg.SetAllocator(Ui2dAllocateFunction, NULL);
            arg.SetAllocateSyncFlag(false);

            // フォント描画のためのコンスタントバッファを作成
            g_Ui2dConstantBuffer.Initialize(pDevice, arg);
            drawInfo.SetUi2dConstantBuffer(&g_Ui2dConstantBuffer);

            bufferOffset += buildResultInfomation.requiredUi2dConstantBufferSize * nn::ui2d::ConstantBufferCount;

        }
        else
        {
            drawInfo.SetUi2dConstantBuffer(NULL);
        }

        // ui2d 内で使用される font 用のコンスタントバッファを作成する。
        if (buildResultInfomation.requiredFontConstantBufferSize > 0)
        {
            nn::font::GpuBuffer::InitializeArg  arg;

            arg.SetBufferSize(buildResultInfomation.requiredFontConstantBufferSize);
            arg.SetBufferCount(nn::ui2d::ConstantBufferCount);
            arg.SetGpuAccessFlag(nn::gfx::GpuAccess_ConstantBuffer);
            arg.SetMemoryPool(pMemoryPool);
            arg.SetMemoryPoolOffset(bufferOffset);
            arg.SetAllocator(Ui2dAllocateFunction, NULL);
            arg.SetAllocateSyncFlag(false);

            g_Ui2dFontConstantBuffer.Initialize(pDevice, arg);
            drawInfo.SetFontConstantBuffer(&g_Ui2dFontConstantBuffer);

            bufferOffset += buildResultInfomation.requiredFontConstantBufferSize * nn::ui2d::ConstantBufferCount;
        }
        else
        {
            drawInfo.SetFontConstantBuffer(NULL);
        }
    }
}

//------------------------------------------------------------------------------
// ui2d 関連の初期化処理
//------------------------------------------------------------------------------
void InitializeUi2dRuntime()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // レイアウトライブラリの初期化
    nn::ui2d::Initialize(Ui2dAllocateFunction, Ui2dDeallocateFunction, NULL);

    // リソースアクセサの初期化
    g_pArcResourceAccessor = AllocAndConstruct<nn::ui2d::ArcResourceAccessor>();

    // レイアウトアーカイブの読み込み
    g_pLayoutArchiveBinary = ReadFileWithAllocate(LayoutArchivePath, nn::ui2d::ArchiveResourceAlignment);
    {
        bool    result = g_pArcResourceAccessor->Attach(g_pLayoutArchiveBinary, ".");
        NN_ASSERT(result);
    }

    // フォントの読み込み
    g_Dataset.ForEachFont(RegisterDatasetFonts, NULL);

    // fcpx の読み込み
    g_Dataset.ForEachFcpx(LoadDatasetFcpxs, NULL);
    InitializeTextureCache();
    g_Dataset.ForEachFcpx(RegisterDatasetFcpxs, NULL);

    uint32_t    drawCount = g_Dataset.GetDrawCount();
    g_pLayouts = static_cast<nn::ui2d::Layout**>(AllocateFromApplicationHeap(sizeof(nn::ui2d::Layout*) * drawCount));

    nn::ui2d::BuildResultInformation   buildResultInfomation;
    buildResultInfomation.SetDefault();

    NN_PERF_BEGIN_MEASURE_INDEX(UserLoadMeterType_Initialization);
    for (uint32_t i = 0; i < drawCount;i++ )
    {
        // レイアウトの初期化
        g_pLayouts[i] = AllocAndConstruct<nn::ui2d::Layout>();

        nn::ui2d::Layout::BuildOption   opt;
        opt.SetDefault();

        if (!g_pLayouts[i]->BuildWithName(&buildResultInfomation, pDevice, g_pArcResourceAccessor, NULL, NULL, opt, g_Dataset.GetLayoutName()))
        {
            NN_LOG("Failed to build the layout data[%s].", g_Dataset.GetLayoutName());
        }
    }

    nn::font::Rectangle rect = g_pLayouts[0]->GetLayoutRect();

    // グラフィックスリソースの設定
    g_GraphicsResource.Setup(pDevice, 512);
    g_GraphicsResource.RegisterCommonSamplerSlot(RegisterSlotForSampler_, &g_GfxFramework);

    // Ui2d の描画に使用される各種バッファを初期化して DrawInfo へ設定する
    InitializeUi2dBuffers(g_DrawInfo, buildResultInfomation);

    // 描画に使用する情報の設定
    {
        nn::util::MatrixT4x4fType   projection;
        nn::util::MatrixOrthographicOffCenterRightHanded(
            &projection,
            rect.left,
            rect.right,
            rect.bottom,
            rect.top,
            0.0f,
            300.0f);
        nn::util::MatrixT4x3fType   view;
        nn::util::Vector3fType  pos;
        nn::util::Vector3fType  up;
        nn::util::Vector3fType  target;
        nn::util::VectorSet(&pos, 0.0f, 0.0f, 1.0f);
        nn::util::VectorSet(&up, 0.0f, 1.0f, 0.0f);
        nn::util::VectorSet(&target, 0.0f, 0.0f, 0.0f);
        nn::util::MatrixLookAtRightHanded(
            &view,
            pos,
            target,
            up);

        g_DrawInfo.SetGraphicsResource(&g_GraphicsResource);
        g_DrawInfo.SetProjectionMtx(projection);
        g_DrawInfo.SetViewMtx(view);
    }
    NN_PERF_END_MEASURE_INDEX(UserLoadMeterType_Initialization);

    // アニメーションの作成と再生
    if(g_Dataset.IsAnimationData())
    {
        for (uint32_t i = 0; i < drawCount; i++)
        {
            g_pAnimator = g_pLayouts[i]->CreateGroupAnimator(pDevice, g_Dataset.GetAnimatorTag());
            g_pAnimator->PlayAuto(1.0f);
        }
    }

    // テクスチャパターンアニメーションに利用されるテクスチャが正しく登録されるように、アニメーションの初期化後に実行します。
    g_pArcResourceAccessor->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture_, &g_GfxFramework);

    // TextureCache のアップデート
    UpdateTextureCache(drawCount);
}

//------------------------------------------------------------------------------
// ui2d 関連の終了処理
//------------------------------------------------------------------------------
void FinalizeUi2dRuntime()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    g_Ui2dConstantBuffer.Finalize(pDevice, Ui2dDeallocateFunction, NULL);
    g_Ui2dFontConstantBuffer.Finalize(pDevice, Ui2dDeallocateFunction, NULL);

    g_GraphicsResource.UnregisterCommonSamplerSlot(UnregisterSlotForSampler_, &g_GfxFramework);
    g_GraphicsResource.Finalize(pDevice);
    uint32_t    drawCount = g_Dataset.GetDrawCount();
    for (uint32_t i = 0; i < drawCount;i++)
    {
        g_pLayouts[i]->Finalize(pDevice);
        DestructAndFree<nn::ui2d::Layout>(g_pLayouts[i]);
    }
    DeallocateApplicationHeap(g_pLayouts);

    g_pArcResourceAccessor->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture_, &g_GfxFramework);
    g_pArcResourceAccessor->Detach();
    g_pArcResourceAccessor->Finalize(pDevice);
    DestructAndFree<nn::ui2d::ArcResourceAccessor>(g_pArcResourceAccessor);

    for(int i = 0; i < g_NumberOfFonts; i++)
    {
        auto& pFont = g_Fonts[i];
        pFont->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture_, &g_GfxFramework);
        void* pFontResource = pFont->RemoveResource(pDevice);
        pFont->Finalize(pDevice);
        g_Allocator.FreeMemoryWithoutLogging(pFontResource);

        DestructAndFree<nn::font::ResFont>(pFont);
    }
    g_NumberOfFonts = 0;

    for (int i = 0; i < g_NumberOfFcpxs; i++)
    {
        nn::ui2d::ComplexFontHelper::FinalizeComplexFontTree(
            pDevice,
            g_Fcpxs[i],
            UnregisterSlotForTexture_,
            &g_GfxFramework);
        g_Fcpxs[i] = NULL;
    }
    g_NumberOfFcpxs = 0;

    if (g_pTextureCache != NULL)
    {
        g_pTextureCache->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture_, &g_GfxFramework);
        g_pTextureCache->Finalize(pDevice, nn::ui2d::Layout::GetFreeFunction(), NULL);
        DestructAndFree<nn::font::TextureCache>(g_pTextureCache);
        g_pTextureCache = NULL;
    }

    for (int i = 0; i < g_FileCount; i++)
    {
        Ui2dDeallocateFunction(g_pFontRawResource[i], NULL);

        g_FontBinarySize[i] = 0;
        g_pFontBinary[i] = NULL;
        g_pFontRawResource[i] = NULL;
    }

    // ArcResourceAccessor の Finalize でアクセスされるため最後に開放する。
    g_Allocator.FreeMemoryWithoutLogging(g_pLayoutArchiveBinary);
}

//------------------------------------------------------------------------------
// メインループ を中断するかどうかを判定
//------------------------------------------------------------------------------
bool CheckToBreakMainLoop()
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    // WindowsMessege の処理
    {
        MSG  msg;
        if ( PeekMessage( &msg, NULL, 0, 0, PM_REMOVE) )
        {
            TranslateMessage(&msg);
            if ( msg.message == WM_QUIT)
            {
                return true;
            }
            DispatchMessage(&msg);
        }
    }
#elif !defined(NN_BUILD_CONFIG_OS_HORIZON)
    if(g_Pad.IsHold(g_Pad.MASK_L|g_Pad.MASK_R))
    {
        return true;
    }
#endif

    return false;
};

//------------------------------------------------------------------------------
// メインループ (ループを中断する際は true を返します)
//------------------------------------------------------------------------------
bool MainLoop(int frame)
{
    NN_UNUSED(frame);

    // ループを中断する必要があるかどうか。
    if(CheckToBreakMainLoop())
    {
        return true;
    }

    // キャッシュを汚す目的で CPU の L3 キャッシュサイズ分適当にメモリを読み書きする。
    char*   pSrc = static_cast<char*>(AllocateFromApplicationHeap(2 * 1024 * 1024));
    char*   pDest = static_cast<char*>(AllocateFromApplicationHeap(2 * 1024 * 1024));

    for (int write = 0; write < 2 * 1024 * 1024; ++write)
    {
        pDest[write] = pSrc[write];
    }

    // テクスチャ取得と vsync 待ち
    NN_PERF_BEGIN_FRAME();

    const int commandBufferIndex = 0;

    // 最初のフレームだけ例外的にこれから書き込むテクスチャーをまず取得する。
    if (frame == 0)
    {
        g_GfxFramework.AcquireTexture(commandBufferIndex);
        g_GfxFramework.WaitDisplaySync(commandBufferIndex, nn::TimeSpan::FromSeconds(2));
    }

    NN_PERF_BEGIN_MEASURE_INDEX(UserLoadMeterType_Frame);

    // レイアウトの更新処理
    uint32_t    drawCount = g_Dataset.GetDrawCount();

    g_DrawInfo.Map(g_Ui2dConstantBufferIndex);

    for (uint32_t i = 0; i < drawCount; i++)
    {
        g_pLayouts[i]->AnimateAndUpdateAnimFrame();
        g_pLayouts[i]->Calculate(g_DrawInfo);
    }
    g_DrawInfo.Unmap();
    g_DrawInfo.SetGpuAccessBufferIndex(g_Ui2dConstantBufferIndex);

    ++g_Ui2dConstantBufferIndex;
    g_Ui2dConstantBufferIndex %= nn::ui2d::ConstantBufferCount;

    // 描画処理
    g_GfxFramework.BeginFrame(commandBufferIndex);
    {
        nn::gfx::CommandBuffer* pCommandBuffer = g_GfxFramework.GetRootCommandBuffer(commandBufferIndex);
        NN_PERF_BEGIN_MEASURE_GPU(pCommandBuffer);

        {
            nn::gfx::ColorTargetView* pTarget = g_GfxFramework.GetColorTargetView();
            nn::gfx::DepthStencilView* pDepthStencilView = g_GfxFramework.GetDepthStencilView();
            // バッファのクリア
            pCommandBuffer->ClearColor(pTarget, 0.0f, 0.0f, 0.5f, 1.0f, NULL);
            pCommandBuffer->ClearDepthStencil(
                pDepthStencilView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL);

            // レンダーステートの設定
            pCommandBuffer->SetRenderTargets(1, &pTarget, pDepthStencilView);
            pCommandBuffer->SetViewportScissorState(g_GfxFramework.GetViewportScissorState());
            pCommandBuffer->SetBlendState(g_GfxFramework.GetBlendState(
                nns::gfx::GraphicsFramework::BlendStateType_Alpha));
            pCommandBuffer->SetDepthStencilState(g_GfxFramework.GetDepthStencilState(
                nns::gfx::GraphicsFramework::DepthStencilStateType_Disabled));
            pCommandBuffer->SetRasterizerState(g_GfxFramework.GetRasterizerState(
                nns::gfx::GraphicsFramework::RasterizerStateType_FillSolid_CullNone));
        }

       // Draw
        for (uint32_t i = 0; i < drawCount; i++)
        {
            g_pLayouts[i]->Draw(g_DrawInfo, *pCommandBuffer);
        }

        NN_PERF_END_MEASURE_GPU(pCommandBuffer);
    }

    g_GfxFramework.EndFrame(commandBufferIndex);

    // Queue 関連の処理は nvservices で実行され、計測結果のぶれが発生する原因となり得るため
    // Queue 関連の処理が呼び出される直前で計測を終了する。
    NN_PERF_END_MEASURE_INDEX(UserLoadMeterType_Frame);

    // コマンドの実行
    // 結果の表示
    g_GfxFramework.ExecuteCommand(commandBufferIndex);

    g_GfxFramework.QueuePresentTexture(1);

    // 次のフレームに書き込む対象を取得する
    g_GfxFramework.AcquireTexture(commandBufferIndex);
    g_GfxFramework.WaitDisplaySync(commandBufferIndex, nn::TimeSpan::FromSeconds(2));
    // Gpuコマンドの完了待ちをします。
    g_GfxFramework.WaitGpuSync(commandBufferIndex, nn::TimeSpan::FromSeconds(2));

    NN_PERF_END_FRAME();

    DeallocateApplicationHeap(pSrc);
    DeallocateApplicationHeap(pDest);

    if(!g_Dataset.IsAnimationData())
    {
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
// アプリケーションの初期化処理
//------------------------------------------------------------------------------
void InitializeApp_()
{
    // アプリケーションで使用するヒープ領域を初期化
    g_Allocator.Initialize(ApplicationHeapSize);

    // ファイルシステムの初期化(gfx の初期化より前に実行します)
    InitializeFileSystem();

    // グラフィックスフレームワークの初期化
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(1024 * 1024 * 32);
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayWidth(DisplayWidth);
    fwInfo.SetDisplayHeight(DisplayHeight);
    fwInfo.SetMemoryPoolSize(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, ConstantBufferSize);
    g_GfxFramework.Initialize(fwInfo);
}

//------------------------------------------------------------------------------
// アプリケーションの終了処理
//------------------------------------------------------------------------------
void FinalizeApp_()
{
    // ファイルシステム終了処理。
    nn::fs::Unmount("Contents");

    // グラフィックスフレームワーク 終了処理
    g_GfxFramework.FreePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, g_offsetToUi2dDrawMemoryPool);
    g_GfxFramework.Finalize();

    nn::time::Finalize();

    // ヒープメモリーの終了処理
    g_Allocator.Finalize();
}



//------------------------------------------------------------------------------
//  メイン関数
//------------------------------------------------------------------------------
TEST(PerformanceProfile, AllData)
{
    // スレッドの動作しているコア番号を取得する API が重く
    // パフォーマンス計測の結果に大きな影響が出ている。
    // 暫定回避策としてメインスレッドを 0 番に固定して
    // コア番号取得 API が呼び出されないように nn::perf を使用する。
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);
    nn::time::Initialize();

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    // NX 実機のみ
    // 性能を携帯モード、据え置きモードで同じ設定に固定する。
    nn::oe::Initialize();
    nn::oe::SetPerformanceConfiguration(nn::oe::PerformanceConfiguration_Cpu1020MhzGpu307MhzEmc1331Mhz);
#endif

    // 初期化
    {
        InitializeApp_();

        // 負荷計測機能を初期化
        InitializeLoadMeter();
    }

    //-----------------------
    // テスト用のループ
    //-----------------------
    g_Dataset.Initialize("Contents:/Resources/dataset.txt");
    g_Dataset.ReadConfig();

    NN_LOG("Start dataset.\n");

    nnt::graphics::PerformanceProfileData  profile;
    nn::time::PosixTime posixTime;
    nn::time::CalendarTime  calendarTime;

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#if defined(NN_BUILD_CONFIG_ADDRESS_64)
    const char* platformString = "NXFP2-a64";
#else
    const char* platformString = "NXFP2-a32";
#endif
#else
    const char* platformString = "PC";
#endif

    bool isNextDataExist = true;
    LOOP_TESTDATA:
    isNextDataExist = g_Dataset.NextData();
    if(!isNextDataExist)
    {
        goto LOOP_TESTDATA_END;
    }
    if(g_Dataset.IsAnimationData())
    {
        g_Dataset.NextKeyFrame();
    }

    profile.Initialize(1024, AllocateProfilingMemory, DeallocateProfilingMemory);
    profile.SetName(g_Dataset.GetLayoutName());

    nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime);
    nn::time::ToCalendarTime(&calendarTime, NULL, posixTime);

    char    timeString[32];

    nn::util::SNPrintf(timeString, 32,
        "%04d-%02d-%02d %02d:%02d:%02d",
        calendarTime.year, calendarTime.month, calendarTime.day,
        calendarTime.hour, calendarTime.minute, calendarTime.second);

    // 計測した日付を設定。
    profile.SetDate(timeString);
    // 計測した環境を設定。
    profile.SetEnvironmentText(platformString);

    // 最大使用量をリセットする
    g_Allocator.ResetMaxSize();

    // ui2d ランタイムの初期化処理
    InitializeUi2dRuntime();

    // メインループ
    NN_LOG("Start demo.\n");

    for(int frame = 0; ; frame++)
    {
        if(g_Dataset.IsAnimationData() && frame > g_Dataset.GetKeyFrame() && !g_Dataset.NextKeyFrame())
        {
            break;
        }

        bool needToBreak = MainLoop(frame);

        nn::perf::CpuMeter*   pCpuMeter = NN_PERF_GET_USER_METER(UserLoadMeterType_Frame);
        nn::perf::GpuMeter*   pGpuMeter = NN_PERF_GET_GPU_METER();

        if (pCpuMeter != NULL &&
            pGpuMeter != NULL)
        {
            profile.SetCpuValue(frame, pCpuMeter->GetLastTotalSpan().GetMicroSeconds());
            profile.SetGpuValue(frame, pGpuMeter->GetLastTotalSpan().GetMicroSeconds());
            profile.SetMemoryUsageValue(frame, g_Allocator.GetMaxSize());
        }

        // 初期化時の負荷情報は 0 フレーム目のみ書き出す。
        if (frame == 0)
        {
            nn::perf::CpuMeter*   pInitializationCpuMeter = NN_PERF_GET_USER_METER(UserLoadMeterType_Initialization);
            if (pInitializationCpuMeter != NULL)
            {
                profile.SetInitializationLoadValue(pInitializationCpuMeter->GetLastTotalSpan().GetMicroSeconds());
            }
        }
        if(needToBreak)
        {
            break;
        }
    }

    // ファイルにデータを保存
    {
        char    filePath[1024];

        nn::util::SNPrintf(timeString, 32,
            "%04d%02d%02d_%02d%02d%02d",
            calendarTime.year, calendarTime.month, calendarTime.day,
            calendarTime.hour, calendarTime.minute, calendarTime.second);

        nn::util::SNPrintf(filePath, 1024,
            "Outputs:/Outputs/%s_%s.json", g_Dataset.GetDataName(), timeString);
        profile.Write(filePath);
    }

    profile.Finalize();

    FinalizeUi2dRuntime();
    // ループの終端
    goto LOOP_TESTDATA;
LOOP_TESTDATA_END:

    NN_LOG("Complete dataset\n");
    g_Dataset.Finalize();
    nn::fs::Unmount("Outputs");

    // 終了処理
    {
        FinalizeLoadMeter();

        FinalizeApp_();
    }

    NN_ASSERT(g_Allocator.GetOccupiedSize() == 0);
    SUCCEED();
} // NOLINT(impl/function_size)
