﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <iostream>
#include <fstream>
#include <string>
#include <unordered_set>
#include <unordered_map>
#include "IntermediateFileManager.h"
#include "Texpkg_Helper.h"

int main(int argc, char* argv[])
{
    if(argc >= 2 && argv[1] == std::string("-help"))
    {
        std::cout << "Help Menu\n";
        std::cout << "----------------------------------------------------------------------------\n";
        std::cout << "-c  <filename>: Specify config file with shader/texture inputs\n";
        std::cout << "                See \"sampleConfig.txt\" for example\n";
        std::cout << "-o  <filename>: Desired name for output file\n";
        std::cout << "-ip <path>:     Path to the folder containing the raw asset files\n";
        std::cout << "-op <path>:     Path to the folder where output files should be written\n";
        std::cout << "-db <bool>:     Whether to output debug GLSLC files for shader debugging\n";
        system("pause");
        return 0;
    }
    else if(argc < 3)
    {
        std::cout << "No config file specified, type \"-help\" for command list.\n";
        system("pause");
        return 1;
    }

    std::string outputFileName = "default.out";

    std::string inPathName;
    std::string outPathName;

    std::string configFileName;
    bool outputDebugGlslcFiles = false;
    bool setShaderIncludePath = false;

    int argCounter = 1;
    bool output = false;
    bool config = false;
    bool outPath = false;
    bool inPath = false;
    bool outputDebug = false;
    bool includes = false;

    IntermediateFileManager fileManager;

    while(argCounter < argc)
    {
        if(std::string(argv[argCounter]) == "-o")
        {
            output = true;
            config = false;
            outPath = false;
            inPath = false;
            outputDebug = false;
            includes = false;
        }
        else if(std::string(argv[argCounter]) == "-c")
        {
            output = false;
            config = true;
            outPath = false;
            inPath = false;
            outputDebug = false;
            includes = false;
        }
        else if (std::string(argv[argCounter]) == "-ip")
        {
            output = false;
            config = false;
            outPath = false;
            inPath = true;
            outputDebug = false;
            includes = false;
        }
        else if (std::string(argv[argCounter]) == "-op")
        {
            output = false;
            config = false;
            outPath = true;
            inPath = false;
            outputDebug = false;
            includes = false;
        }
        else if (std::string(argv[argCounter]) == "-db")
        {
            output = false;
            config = false;
            outPath = false;
            inPath = false;
            outputDebug = true;
            includes = false;
        }
        else if (std::string(argv[argCounter]) == "-in")
        {
            output = false;
            config = false;
            outPath = false;
            inPath = false;
            outputDebug = false;
            includes = true;
        }
        else if(output)
        {
            outputFileName = argv[argCounter];
        }
        else if(config)
        {
            configFileName = argv[argCounter];

            std::cout << "Loading Config File: " << argv[argCounter] << "\n";

            if(!fileManager.LoadConfigFile(argv[argCounter]))
            {
                std::cout << "Failed to load config file: " << argv[argCounter] << "\n";
                system("pause");
                return 1;
            }
        }
        else if (inPath)
        {
            inPathName = argv[argCounter];
            char check = inPathName.back();
            if(check != '\\' && check != '/')
            {
                inPathName = inPathName + "\\";
            }
        }
        else if (outPath)
        {
            outPathName = argv[argCounter];
            char check = outPathName.back();
            if(check != '\\' && check != '/')
            {
                outPathName = outPathName + "\\";
            }
        }
        else if (outputDebug)
        {
            std::string debugCheck = argv[argCounter];
            if(debugCheck == "true")
            {
                outputDebugGlslcFiles = true;
            }
        }
        else if (includes)
        {
            std::string includeCheck = argv[argCounter];
            if (includeCheck == "true")
            {
                setShaderIncludePath = true;
            }
        }
        else
        {
            std::cout << "Invalid command line input: " << argv[argCounter] << ", type \"-help\" for command list.\n";
            system("pause");
            return 1;
        }

        ++argCounter;
    }

    if (!configFileName.size())
    {
        std::cout << "No config file specified, type \"-help\" for command list.\n";
        system("pause");
        return 1;
    }

    std::cout << "Loading Texpkg\n";
#ifdef _WIN64
    const wchar_t* nvnImageDllName = L"NVNImage.dll";
    const wchar_t* nvnTexPkgDllName = L"NvnTexpkg.dll";
#else
    const wchar_t* nvnImageDllName = L"NVNImage32.dll";
    const wchar_t* nvnTexPkgDllName = L"NvnTexpkg32.dll";
#endif
    if (!TexpkgHelperLoadLibraries(nvnImageDllName, nvnTexPkgDllName))
    {
        std::cout << "Texpkg failed to load\n";
        system("pause");
        return 1;
    }

    std::cout << "\nLoading shader source\n";
    if(!fileManager.LoadShaderPrograms(inPathName))
    {
        std::cout << "Failed to load shader file\n";
        system("pause");
        return 1;
    }

    std::cout << "Compiling shaders\n";
    if(!fileManager.CompileShaderPrograms(setShaderIncludePath, inPathName))
    {
        std::cout << "Shader compilation failed\n";
        system("pause");
        return 1;
    }

    std::cout << "Writing shader reflection headers/source:\n";
    if(!fileManager.OutputUniformBlockHeaderFiles(outPathName))
    {
        std::cout << "Failed to write header file\n";
        system("pause");
        return 1;
    }

    std::cout << "\nLoading texture data\n";
    if(!fileManager.LoadTextureData(inPathName))
    {
        std::cout << "Failed to load texture file\n";
        system("pause");
        return 1;
    }

    std::cout << "Converting textures\n";
    if(!fileManager.ConvertTextures())
    {
        std::cout << "Failed to convert texture data\n";
        system("pause");
        return 1;
    }

    std::cout << "\nWriting output asset file\n";
    if(!fileManager.WriteOutputFile(outPathName, outputFileName, outputDebugGlslcFiles))
    {
        std::cout << "Failed to write output file\n";
        system("pause");
        return 1;
    }

    std::cout << "\nAsset file conversion successful\n\n";
    system("pause");
}//NOLINT(impl/function_size)
