﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{IntermediateFileManager.cpp,PageSampleNvnTutorial02}
 *
 * @brief
 *  This file defines a class that handles reading config
 *  text files, loading raw asset data, and writing binary
 *  asset files for the tutorial. The config files are used
 *  to define what texture files need to loaded/converted,
 *  shader source to be compiled into programs, and model
 *  data to be output in the asset file.
 */

#include "IntermediateFileManager.h"
#include "Texpkg_Helper.h"
#include <cassert>
#include <iostream>
#include <fstream>
#include "OutputFileHeaders.h"
#include <vector>
#include <string>
#include <set>
#include <cstdio>


static const float g_OutputFileVersion = 1.0f;

IntermediateFileManager::IntermediateFileManager()
{
}

/*
 * IntermediateFileManager::~IntermediateFileManager
 * -------------------------------------------------
 * Cleans up GLSLC and texpkg objects and frees any dynamically
 * allocated memory.
 */
IntermediateFileManager::~IntermediateFileManager()
{
        /* Finalize the glslc compile objects for each shader program. */
    for(std::unordered_map<std::string, ShaderCompileData>::iterator itr = m_ShaderPrograms.begin(); itr != m_ShaderPrograms.end(); ++itr)
    {
        glslcFinalize(&(itr->second.m_CompileObject));
    }

        /* Release the raw texture data. */
    for(std::vector<TextureData>::iterator itr = m_RawTextureData.begin(); itr != m_RawTextureData.end(); ++itr)
    {
        itr->m_pRawImage->Release();
    }

        /* Release the converted texture data. */
    for(std::unordered_map<std::string, nvnTool::texpkg::NVNHWTexture>::iterator itr = m_ConvertedTextureData.begin(); itr != m_ConvertedTextureData.end(); ++itr)
    {
        g_HwTextureExporter->ReleaseTextureData(&itr->second);
    }

        /* Frees the loaded model data. */
    for(uint32_t i = 0; i < m_RawModelData.size(); ++i)
    {
        free(m_RawModelData[i].m_IndexData.m_pData);

        for(uint32_t j = 0; j < m_RawModelData[i].m_VertexAttributes.size(); ++j)
        {
            free(m_RawModelData[i].m_VertexAttributes[j].m_pData);
        }

    }
}

/*
 * IntermediateFileManager::LoadShaderPrograms
 * -------------------------------------------
 * Loads the shader source needed for each stage of the shader program.
 */
bool IntermediateFileManager::LoadShaderPrograms(const std::string& path)
{
    for (std::unordered_map<std::string, ShaderCompileData>::iterator itr = m_ShaderPrograms.begin(); itr != m_ShaderPrograms.end(); ++itr)
    {
        std::string* temp = reinterpret_cast<std::string*>(&(itr->second));

        for(uint32_t i = 0; i < 6; ++i)
        {
                /* If the shader stage exists for the program, load the source. */
            if (temp[i].size())
            {
                if (!LoadShaderSource(temp[i], path))
                {
                    return false;
                }
            }
        }
    }

    return true;
}

/*
 * IntermediateFileManager::LoadShaderSource
 * -----------------------------------------
 * Loads the shader source specified by the given file name. If the source for
 * that file name has already been loaded, it just returns.
 */
bool IntermediateFileManager::LoadShaderSource(const std::string& filename, const std::string& path)
{
        /* If the sour has been loaded already, return true. */
    if (m_RawShaderSource.find(filename) != m_RawShaderSource.end())
    {
        return true;
    }

    std::ifstream shaderFile;
    shaderFile.open(path + filename);

        /* Return false if the file is not found. */
    if (!shaderFile.is_open())
    {
        return false;
    }

    while(!shaderFile.eof())
    {
        std::string temp;
        std::getline(shaderFile, temp);
        temp += "\n";
        m_RawShaderSource[filename] += temp;
    }

    return true;
}

/*
 * IntermediateFileManager::CompileShaderPrograms
 * ----------------------------------------------
 * Compile the shader sources for each stage with a default set of
 * shader compile options.
 */
bool IntermediateFileManager::CompileShaderPrograms(bool setShaderIncludePath, const std::string& path)
{
    for(std::unordered_map<std::string, ShaderCompileData>::iterator itr = m_ShaderPrograms.begin(); itr != m_ShaderPrograms.end(); ++itr)
    {
        std::vector<const char *> shaderSources;
        std::vector<NVNshaderStage> shaderStages;

            /*
             * Checks each stage to see if there is shader source for it. If
             * there is, then the source and stage value are added to their
             * respective lists.
             */
        if(itr->second.m_Comp.size())
        {
            shaderSources.push_back(m_RawShaderSource.at(itr->second.m_Comp).c_str());
            shaderStages.push_back(NVNshaderStage::NVN_SHADER_STAGE_COMPUTE);
        }

        if(itr->second.m_Frag.size())
        {
            shaderSources.push_back(m_RawShaderSource.at(itr->second.m_Frag).c_str());
            shaderStages.push_back(NVNshaderStage::NVN_SHADER_STAGE_FRAGMENT);
        }

        if(itr->second.m_Geom.size())
        {
            shaderSources.push_back(m_RawShaderSource.at(itr->second.m_Geom).c_str());
            shaderStages.push_back(NVNshaderStage::NVN_SHADER_STAGE_GEOMETRY);
        }

        if(itr->second.m_TessCont.size())
        {
            shaderSources.push_back(m_RawShaderSource.at(itr->second.m_TessCont).c_str());
            shaderStages.push_back(NVNshaderStage::NVN_SHADER_STAGE_TESS_CONTROL);
        }

        if(itr->second.m_TessEval.size())
        {
            shaderSources.push_back(m_RawShaderSource.at(itr->second.m_TessEval).c_str());
            shaderStages.push_back(NVNshaderStage::NVN_SHADER_STAGE_TESS_EVALUATION);
        }

        if(itr->second.m_Vert.size())
        {
            shaderSources.push_back(m_RawShaderSource.at(itr->second.m_Vert).c_str());
            shaderStages.push_back(NVNshaderStage::NVN_SHADER_STAGE_VERTEX);
        }

            /*
             * GLSLC Compile Object
             * --------------------
             * The GLSLCcompileObject holds the options used in the shader compile,
             * the output from the compile, the shader compile status/log, and shader
             * reflection data. The same compile object can be used for multiple compiles.
             * The lastCompiledResults pointer will be freed on subsequent compiles so
             * it is up to the application to copy the necessary data before using the
             * compile object again. The compile object needs to be initialize before
             * being used and should be finalized once the application is done with it.
             */
        if (!glslcInitialize(&(itr->second.m_CompileObject)))
        {
            std::cout << "Failed to initialize glslc compile object.\n";
            return false;
        }

            /*
             * GLSLC Options
             * -------------
             * The GLSLCoptions portion of the GLSLCcompile object has a
             * GLSLCoptionFlags struct that contains a number of options
             * that can be set for the next shader compile. Options:
             *
             * glslSeparable                - Treat the GLSLC object as separable. Can be
             *                                used to compile singular shader stages.
             *
             * outputAssembly               - Include an assembly dump section in the
             *                                GLSLCoutput.
             *
             * outputGpuBinaries            - Output individual program binary sections
             *                                in the GLSLCoutput. Outputs one binary per
             *                                stage.
             *
             * outputPerfStats              - Output individual perf statistics for each
             *                                compiled binary.
             *
             * outputShaderReflection       - Output shader reflection data. Allows the
             *                                application to grab data about the shader like
             *                                uniform block sizes, names, locations, etc.
             *
             * language                     - Specify the language of the input shaders.
             *
             * outputDebugInfo              - Level of debug information to generate.
             *
             * spillControl                 - Control amount of spilling. Can be set to
             *                                DEFAULT_SPILL or NO_SPILL.
             *
             *
             * outputThinGpuBinaries        - Controls whether the resulting output contains a
             *                                GPU program that is a thin binary (only contains
             *                                program code for NX platforms).
             *
             * tessellationAndPassthroughGS - Compiles tessellation evaluation and passthrough
             *                                geometry shaders (NV_geometry_shader_passthrough)
             *                                to work together as a pair.
             *
             * These additional options are available in the GLSLCoptions struct proper:
             * includeInfo                  - Used to set include paths for shader headers.
             *
             * xfbVaryingInfo               - Used to specify transform feedback information.
             *
             * forceIncludeStdHeader        - Include standard header file. If this is NULL,
             *                                no standard header will be included.
             */
        GLSLCoptions * options = &(itr->second.m_CompileObject.options);

        options->optionFlags.outputAssembly = true;

        // NOTE: This must be set to true for now.
        options->optionFlags.outputGpuBinaries = true;

        options->optionFlags.glslSeparable = shaderStages.size() < 2;
        options->optionFlags.outputPerfStats = true;
        options->optionFlags.outputShaderReflection = true;
        options->optionFlags.outputDebugInfo = GLSLC_DEBUG_LEVEL_G2;

        if (setShaderIncludePath)
        {
            const char * includePaths = path.c_str();

            options->includeInfo.numPaths = 1;
            options->includeInfo.paths = &includePaths;
        }

            /*
             * GLSLC Input
             * -------------
             * This structure defines the input data for the compilation.
             * The input is a list of pointers to shader source with a
             * corresponding list of NVNshaderStage values.
             */
        itr->second.m_CompileObject.input.sources = &shaderSources[0];
        itr->second.m_CompileObject.input.stages = &shaderStages[0];
        itr->second.m_CompileObject.input.count = static_cast<uint8_t>(shaderSources.size());

            /* Compiles the shader program with the given input and options. */
        if(!glslcCompile(&(itr->second.m_CompileObject)))
        {
            std::cout << "Compilation failed.  Info log:\n" << itr->second.m_CompileObject.lastCompiledResults->compilationStatus->infoLog << "\n";
            return false;
        }

            /* Set these pointers to NULL before the things they point at go out of scope */
        itr->second.m_CompileObject.input.sources = NULL;
        itr->second.m_CompileObject.input.stages = NULL;
        itr->second.m_CompileObject.input.count = 0;
    }

    return true;
}

/*
 * Align
 * -----
 * Aligns a given value up to the given alignment
 */
size_t IntermediateFileManager::Align(size_t size, size_t alignment, size_t& bytesAlignedUp)
{
    bytesAlignedUp = (alignment - (size % alignment)) % alignment;

    return bytesAlignedUp + size;
}

/*
 * PadFileBuffer
 * -------------
 * Inserts padding bytes into the file to maintain alignment of following
 * entries.
 */
size_t IntermediateFileManager::PadFileBuffer(FILE* file, size_t padSize)
{
    if (padSize == 0)
    {
        return 0;
    }

    static std::vector<char> padBuffer;
    if (padSize > padBuffer.size())
    {
        padBuffer.resize(padSize, 0);
    }

    return fwrite(&padBuffer[0], sizeof(char), padSize, file);
}

/*
 * IntermediateFileManager::WriteOutputFile
 * ----------------------------------------
 * Takes the accumulated shader compile data, converted texture
 * data, and model data and writes it all out into one large
 * binary file. The general structure of the file is laid out
 * by the structs defined in OutputFileHeaders.h.
 */
bool IntermediateFileManager::WriteOutputFile(const std::string& outPath, const std::string& fileName, bool outputDebugGlslcFiles)
{
    std::string filePath = outPath + fileName;
    FILE* outputFile;
    fopen_s(&outputFile, filePath.c_str(), "wb");

    if (!outputFile)
    {
        return false;
    }

    OutputFileHeader outputFileHeader;

    ShaderBlockHeader shaderBlockHeader;
    memset(&shaderBlockHeader, 0, sizeof(ShaderBlockHeader));
    std::vector<std::pair<ShaderProgramHeader, std::vector<ShaderStageHeader> > > shaderProgramHeaders;

    TextureBlockHeader textureBlockHeader;
    memset(&textureBlockHeader, 0, sizeof(TextureBlockHeader));
    std::vector<TextureDataHeader> textureDataHeaders;

    ModelBlockHeader modelBlockHeader;
    memset(&modelBlockHeader, 0, sizeof(ModelBlockHeader));

    std::vector<std::pair<ModelHeader, std::pair<std::vector<VertexAttributeHeader>, IndexBufferHeader> > > modelHeaders;

    outputFileHeader.m_Version = g_OutputFileVersion;

        /* Keep track of the current offset. */
    size_t currentOffset = sizeof(OutputFileHeader);

        /* Keep track of instances of padding */
    std::vector<size_t> padding;
    size_t currentPadding = 0;

        /* Check for shader programs*/
    if (!m_ShaderPrograms.size())
    {
        outputFileHeader.m_ShaderBlockOffset = 0;
    }
    else
    {
        outputFileHeader.m_ShaderBlockOffset = sizeof(OutputFileHeader);

            /* Create directory along side assets that glslc debug output files will go in */
        std::string glslcOutputDirectory(outPath + "GlslcDbg\\");
        wchar_t ws[256];
        memset(ws, 0, 256 * sizeof(wchar_t));
        swprintf(ws, glslcOutputDirectory.size(), L"%hs", glslcOutputDirectory.c_str());
        CreateDirectory(ws, NULL);

            /* Accumulates data about each shader program to be written out. */
        for (std::unordered_map<std::string, ShaderCompileData>::iterator itr = m_ShaderPrograms.begin(); itr != m_ShaderPrograms.end(); ++itr)
        {
            shaderProgramHeaders.push_back(std::pair<ShaderProgramHeader, std::vector<ShaderStageHeader> >());

            ShaderProgramHeader& programHeader = shaderProgramHeaders.back().first;
            std::vector<ShaderStageHeader>& stageHeaders = shaderProgramHeaders.back().second;

            programHeader.m_pProgramName = itr->first.c_str();
            programHeader.m_ProgramNameLength = static_cast<uint32_t>(itr->first.size());
            programHeader.m_NumShaderStages = 0;

                /*
                 * GLSLCoutput
                 * -----------
                 * Holds the various sections output by the shader compile.
                 * Offsets to each section are from the beginning of the
                 * output struct. Offsets defined within a section are from
                 * the beginning of that section. These sections are:
                 *
                 * GLSLC_SECTION_TYPE_GPU_CODE   - Holds the compiled binary and
                 *                                 information for a shader stage.
                 *
                 * GLSLC_SECTION_TYPE_ASM_DUMP   - Holds the assembly dump for the
                 *                                 compiled shader stage.
                 *
                 * GLSLC_SECTION_TYPE_PERF_STATS - Holds the performance stats for
                 *                                 the shader stage.
                 *
                 * GLSLC_SECTION_TYPE_REFLECTION - Holds the shader reflection data
                 *                                 for the shader stage.
                 *
                 * GLSLC_SECTION_TYPE_DEBUG_INFO - Holds debug information for the
                 *                                 shader stage.
                 */
            GLSLCoutput* compileOutput = itr->second.m_CompileObject.lastCompiledResults->glslcOutput;

            if(outputDebugGlslcFiles)
            {
                    /*
                     * Write GLSLC output to file for NVN Graphics Debugger to use.
                     * To use these files, capture a frame in NVNGD, click on a
                     * draw event in the Events View window, go to one of the shader
                     * tabs in the API Inspector View, and click on the link to
                     * view shader source.  Set the folder below as the directory
                     * to look for the debug files in.
                     */
                FILE* glslcOutputFileHandle = NULL;

                std::string glslcOutputFileName(outPath + "GlslcDbg\\" + fileName + ".glslc");
                if(fopen_s(&glslcOutputFileHandle, glslcOutputFileName.c_str(), "wb") != 0)
                {
                    assert(false);
                }
                if(!glslcOutputFileHandle)
                {
                    assert(false);
                }

                fwrite(reinterpret_cast<char*>(compileOutput), compileOutput->size, 1, glslcOutputFileHandle);
                fclose(glslcOutputFileHandle);
            }

            for (uint32_t i = 0; i < compileOutput->numSections; ++i)
            {
                GLSLCsectionTypeEnum type = compileOutput->headers[i].genericHeader.common.type;

                if (type == GLSLC_SECTION_TYPE_GPU_CODE)
                {
                        /*
                         * GLSLCgpuCodeHeader
                         * ------------------
                         * This structure is a section header that holds information used
                         * to access compiled shader code for an individual shader stage.
                         *
                         * stage                      - Shader stage the gpu code correspons to.
                         *
                         * controlOffset              - Offset in bytes to the control section
                         *                              from the start of the data section of the
                         *                              header.
                         *
                         * controlSize                - Size of the control section in bytes.
                         *
                         * dataOffset                 - Offset to the shader data section in bytes
                         *                              fom the start of the header's data section.
                         *
                         * dataSize                   - Size of the shader data section.
                         *
                         * scratchMemBytesPerWarp     - Amount of local memory required for the stage
                         *                              in bytes per warp.
                         *
                         * scratchMemBytesRecommended - Recommended amount of local memory on the target.
                         *
                         * asmDumpSectionIdx          - Index of the assembly dump section for this shader
                         *                              stage (if option was turned on).
                         *
                         * perfStatsSectionNdx        - Index of the performance stats section for this
                         *                              shader stage (if option was turned on).
                         *
                         * subroutineLinkageMapOffset - Offset to subroutine linkage map offset in data section
                         *
                         * subroutineLinkageMapSize   - Size of subroutine linkage map size
                         */
                    GLSLCgpuCodeHeader * gpuHeader = &(compileOutput->headers[i].gpuCodeHeader);

                        /*
                         * Grab a pointer to the data section of the header. The control offset and
                         * data offset from the gpu code header are offsets from this pointer.
                         */
                    void * data = ((char*)compileOutput) + gpuHeader->common.dataOffset;

                    ShaderStageHeader stageHeader;
                    stageHeader.m_ShaderStage = gpuHeader->stage;
                    stageHeader.m_ShaderControlSize = gpuHeader->controlSize;
                    stageHeader.m_ShaderDataSize = gpuHeader->dataSize;
                    stageHeader.m_ShaderControlOffset = 0; // Retrieve this offset later
                    stageHeader.m_pShaderData = ((char*)data + gpuHeader->dataOffset);
                    stageHeader.m_pShaderControl = ((char*)data + gpuHeader->controlOffset);

                    ++programHeader.m_NumShaderStages;

                    stageHeaders.push_back(stageHeader);
                }
            }

            programHeader.m_pShaderStageOffsets = (uint64_t*)malloc(sizeof(uint64_t) * programHeader.m_NumShaderStages);
        }

        shaderBlockHeader.m_NumShaderPrograms = static_cast<uint32_t>(shaderProgramHeaders.size());
        uint32_t shaderProgramOffsetsSize = sizeof(uint64_t) * shaderBlockHeader.m_NumShaderPrograms;
        shaderBlockHeader.m_pShaderProgramOffsets = (uint64_t*)malloc(shaderProgramOffsetsSize);

            /* Based on the accumulated data the currentOffset is updated. */
        currentOffset += sizeof(ShaderBlockHeader);
        currentOffset -= sizeof(shaderBlockHeader.m_pShaderProgramOffsets);
        currentOffset += shaderProgramOffsetsSize;

        for (uint32_t i = 0; i < shaderProgramHeaders.size(); ++i)
        {
                /*
                 * Align shader program header struct to account for
                 * variable length of shader name
                 */
            currentOffset = Align(currentOffset, 8, currentPadding);
            padding.push_back(currentPadding);
            shaderBlockHeader.m_pShaderProgramOffsets[i] = currentOffset;

            currentOffset += sizeof(ShaderProgramHeader);
            currentOffset -= sizeof(shaderProgramHeaders[i].first.m_pShaderStageOffsets);
            currentOffset -= sizeof(shaderProgramHeaders[i].first.m_pProgramName);
            currentOffset += sizeof(char) * shaderProgramHeaders[i].first.m_ProgramNameLength;
            currentOffset += sizeof(uint64_t) * shaderProgramHeaders[i].first.m_NumShaderStages;

            for (uint32_t j = 0; j < shaderProgramHeaders[i].first.m_NumShaderStages; ++j)
            {
                    /*
                     * Align shader stage header struct to account for
                     * variable length of shader and control data
                     */
                currentOffset = Align(currentOffset, 8, currentPadding);
                padding.push_back(currentPadding);
                shaderProgramHeaders[i].first.m_pShaderStageOffsets[j] = currentOffset;

                currentOffset += sizeof(ShaderStageHeader);
                currentOffset -= sizeof(shaderProgramHeaders[i].second[j].m_pShaderData);
                currentOffset -= sizeof(shaderProgramHeaders[i].second[j].m_pShaderControl);
                currentOffset += shaderProgramHeaders[i].second[j].m_ShaderDataSize;

                    /*
                     * Offset of control data is set now to account for the variable
                     * size of the shader data
                     */
                currentOffset = Align(currentOffset, 8, currentPadding);
                padding.push_back(currentPadding);
                shaderProgramHeaders[i].second[j].m_ShaderControlOffset = static_cast<uint32_t>(currentOffset);

                currentOffset += shaderProgramHeaders[i].second[j].m_ShaderControlSize;
            }
        }
    }

        /* Check for textures */
    if (!m_ConvertedTextureData.size())
    {
        outputFileHeader.m_TextureBlockOffset = 0;
    }
    else
    {
        currentOffset = Align(currentOffset, 8, currentPadding);
        padding.push_back(currentPadding);
        outputFileHeader.m_TextureBlockOffset = currentOffset;

        textureBlockHeader.m_NumTextures = static_cast<uint32_t>(m_ConvertedTextureData.size());
        textureBlockHeader.m_pTextureOffsets = (uint64_t*)malloc(sizeof(uint64_t) * textureBlockHeader.m_NumTextures);

            /* Accumulate data to be written out for the textures. */
        for (std::unordered_map<std::string, nvnTool::texpkg::NVNHWTexture>::iterator itr = m_ConvertedTextureData.begin(); itr != m_ConvertedTextureData.end(); ++itr)
        {
            textureDataHeaders.push_back(TextureDataHeader());

            TextureDataHeader& header = textureDataHeaders.back();

            header.m_TextureDataSize = itr->second.header.dataSize;
            header.m_GpuVersion = 0;
            header.m_Alignment = itr->second.header.align;
            header.m_Width = itr->second.header.width;
            header.m_Height = itr->second.header.height;
            header.m_Depth = itr->second.header.depth;
            header.m_NvnTextureTarget = itr->second.header.target;
            header.m_NvnFormat = itr->second.header.format;
            header.m_MipLevels = itr->second.header.mipmapLevels;
            header.m_TextureNameLength = static_cast<uint32_t>(itr->first.size());
            header.m_TextureDataOffset = 0; // Retrieve offset later
            header.m_pTextureName = itr->first.c_str();
            header.m_pTextureData = itr->second.data;
        }

            /* Update currentOffset */
        currentOffset += sizeof(TextureBlockHeader);
        currentOffset -= sizeof(textureBlockHeader.m_pTextureOffsets);
        currentOffset += sizeof(uint64_t) * textureBlockHeader.m_NumTextures;

        for (uint32_t i = 0; i < textureBlockHeader.m_NumTextures; ++i)
        {
            currentOffset = Align(currentOffset, 8, currentPadding);
            padding.push_back(currentPadding);
            textureBlockHeader.m_pTextureOffsets[i] = currentOffset;

            currentOffset += sizeof(TextureDataHeader);
            currentOffset -= sizeof(textureDataHeaders[i].m_pTextureData);
            currentOffset -= sizeof(textureDataHeaders[i].m_pTextureName);
            currentOffset += sizeof(char) * textureDataHeaders[i].m_TextureNameLength;

                /*
                 * Offset of texture data is set now to account for the variable
                 * size of the texture name
                 */
            currentOffset = Align(currentOffset, 8, currentPadding);
            padding.push_back(currentPadding);
            textureDataHeaders[i].m_TextureDataOffset = static_cast<uint32_t>(currentOffset);

            currentOffset += static_cast<size_t>(textureDataHeaders[i].m_TextureDataSize);
        }
    }

        /* Check for models. */
    if (!m_RawModelData.size())
    {
        outputFileHeader.m_ModelBlockOffset = 0;
    }
    else
    {
        currentOffset = Align(currentOffset, 8, currentPadding);
        padding.push_back(currentPadding);
        outputFileHeader.m_ModelBlockOffset = currentOffset;

            /* Accumulate model data in the header structs. */
        for (std::vector<Model>::iterator itr = m_RawModelData.begin(); itr != m_RawModelData.end(); ++itr)
        {
            modelHeaders.push_back(std::pair<ModelHeader, std::pair<std::vector<VertexAttributeHeader>, IndexBufferHeader> >());

            ModelHeader& modelHeader = modelHeaders.back().first;
            std::vector<VertexAttributeHeader>& vertexBufferHeader = modelHeaders.back().second.first;
            IndexBufferHeader& indexBufferHeader = modelHeaders.back().second.second;

            modelHeader.m_NumPrimitives = itr->m_NumPrimitives;
            modelHeader.m_NvnDrawPrimitiveType = itr->m_NvnDrawPrimitiveType;
            modelHeader.m_NumVertexAttributes = itr->m_NumVertexAttributes;
            modelHeader.m_ModelNameLength = static_cast<uint32_t>(itr->m_Name.size());
            modelHeader.m_IndexBufferOffset = 0; // Retrieve offset later
            modelHeader.m_pModelName = itr->m_Name.c_str();

            indexBufferHeader.m_NumIndices = itr->m_IndexData.m_DataSize / itr->m_IndexData.m_Stride;
            indexBufferHeader.m_IndexBufferSize = itr->m_IndexData.m_DataSize;
            indexBufferHeader.m_pIndexData = itr->m_IndexData.m_pData;
            indexBufferHeader.m_IndexType = itr->m_IndexData.m_IndexType;
            indexBufferHeader.m_MagicPadding = 0;

            for (uint32_t i = 0; i < itr->m_VertexAttributes.size(); ++i)
            {
                VertexAttributeHeader attributeHeader;
                attributeHeader.m_AttributeStride = itr->m_VertexAttributes[i].m_Stride;
                attributeHeader.m_NvnFormat = itr->m_VertexAttributes[i].m_NvnFormat;
                attributeHeader.m_AttributeDataSize = itr->m_VertexAttributes[i].m_DataSize;
                attributeHeader.m_AttributeNameLength = static_cast<uint32_t>(itr->m_VertexAttributes[i].m_Name.size());
                attributeHeader.m_AttributeDataOffset = 0; // Retrieve offset later
                attributeHeader.m_MagicPadding = 0;
                attributeHeader.m_pAttributeName = itr->m_VertexAttributes[i].m_Name.c_str();
                attributeHeader.m_AttributeData = itr->m_VertexAttributes[i].m_pData;

                vertexBufferHeader.push_back(attributeHeader);
            }
        }

        modelBlockHeader.m_NumModels = static_cast<uint32_t>(m_RawModelData.size());
        modelBlockHeader.m_pModelOffsets = (uint64_t*)malloc(sizeof(uint64_t) * modelBlockHeader.m_NumModels);

            /* Update currentOffset. */
        currentOffset += sizeof(ModelBlockHeader);
        currentOffset -= sizeof(modelBlockHeader.m_pModelOffsets);
        currentOffset += sizeof(uint64_t) * modelBlockHeader.m_NumModels;

        for (uint32_t i = 0; i < modelHeaders.size(); ++i)
        {
            currentOffset = Align(currentOffset, 8, currentPadding);
            padding.push_back(currentPadding);
            modelBlockHeader.m_pModelOffsets[i] = currentOffset;

            currentOffset += sizeof(ModelHeader);
            currentOffset -= sizeof(modelHeaders[i].first.m_pVertexAttributeOffsets);
            currentOffset -= sizeof(modelHeaders[i].first.m_pModelName);
            currentOffset += sizeof(uint64_t) * modelHeaders[i].second.first.size();
            currentOffset += sizeof(char) * modelHeaders[i].first.m_ModelNameLength;

            modelHeaders[i].first.m_pVertexAttributeOffsets = (uint64_t*)malloc(sizeof(uint64_t) * modelHeaders[i].second.first.size());

            for (uint32_t j = 0; j < modelHeaders[i].second.first.size(); ++j)
            {
                currentOffset = Align(currentOffset, 8, currentPadding);
                padding.push_back(currentPadding);
                modelHeaders[i].first.m_pVertexAttributeOffsets[j] = currentOffset;

                currentOffset += sizeof(VertexAttributeHeader);
                currentOffset -= sizeof(modelHeaders[i].second.first[j].m_pAttributeName);
                currentOffset -= sizeof(modelHeaders[i].second.first[j].m_AttributeData);
                currentOffset += sizeof(char) * modelHeaders[i].second.first[j].m_AttributeNameLength;

                    /*
                     * Offset of vertex attribute data is set now to account for the variable
                     * size of the attribute's name
                     */
                currentOffset = Align(currentOffset, 8, currentPadding);
                padding.push_back(currentPadding);
                modelHeaders[i].second.first[j].m_AttributeDataOffset = static_cast<uint32_t>(currentOffset);

                currentOffset += modelHeaders[i].second.first[j].m_AttributeDataSize;
            }

            currentOffset = Align(currentOffset, 8, currentPadding);
            padding.push_back(currentPadding);
            modelHeaders[i].first.m_IndexBufferOffset = currentOffset;

            currentOffset += sizeof(IndexBufferHeader);
            currentOffset -= sizeof(modelHeaders[i].second.second.m_pIndexData);
            currentOffset += modelHeaders[i].second.second.m_IndexBufferSize;
        }
    }

        /* Write the output file based on the setup file section headers. */
    uint64_t writtenBytes = 0;
    uint32_t magicPadding = 0;
    uint32_t currentPaddingIndex = 0;

        // File Header
    fwrite(&outputFileHeader, sizeof(outputFileHeader), 1, outputFile);
    writtenBytes += sizeof(outputFileHeader);

    if (outputFileHeader.m_ShaderBlockOffset)
    {
            // Shader Block Header
        fwrite(&shaderBlockHeader.m_NumShaderPrograms, sizeof(shaderBlockHeader.m_NumShaderPrograms), 1, outputFile);
        writtenBytes += sizeof(shaderBlockHeader.m_NumShaderPrograms);

        fwrite(&magicPadding, sizeof(uint32_t), 1, outputFile);
        writtenBytes += sizeof(uint32_t);

        fwrite(shaderBlockHeader.m_pShaderProgramOffsets, sizeof(shaderBlockHeader.m_pShaderProgramOffsets[0]), shaderBlockHeader.m_NumShaderPrograms, outputFile);
        writtenBytes += sizeof(shaderBlockHeader.m_pShaderProgramOffsets[0]);

            // Shader Program Headers
        for (uint32_t i = 0; i < shaderProgramHeaders.size(); ++i)
        {
                // Insert padding
            writtenBytes += PadFileBuffer(outputFile, padding[currentPaddingIndex++]);

            fwrite(&shaderProgramHeaders[i].first.m_NumShaderStages, sizeof(shaderProgramHeaders[i].first.m_NumShaderStages), 1, outputFile);
            writtenBytes += sizeof(shaderProgramHeaders[i].first.m_NumShaderStages);

            fwrite(&shaderProgramHeaders[i].first.m_ProgramNameLength, sizeof(shaderProgramHeaders[i].first.m_ProgramNameLength), 1, outputFile);
            writtenBytes += sizeof(shaderProgramHeaders[i].first.m_ProgramNameLength);

            fwrite(shaderProgramHeaders[i].first.m_pShaderStageOffsets, sizeof(uint64_t), shaderProgramHeaders[i].first.m_NumShaderStages, outputFile);
            writtenBytes += sizeof(uint64_t) * shaderProgramHeaders[i].first.m_NumShaderStages;

            fwrite(shaderProgramHeaders[i].first.m_pProgramName, sizeof(char), shaderProgramHeaders[i].first.m_ProgramNameLength, outputFile);
            writtenBytes += sizeof(char) * shaderProgramHeaders[i].first.m_ProgramNameLength;

                // Shader Stage Headers/Data
            for (uint32_t j = 0; j < shaderProgramHeaders[i].first.m_NumShaderStages; ++j)
            {
                    // Insert padding
                writtenBytes += PadFileBuffer(outputFile, padding[currentPaddingIndex++]);

                fwrite(&shaderProgramHeaders[i].second[j].m_ShaderStage, sizeof(shaderProgramHeaders[i].second[j].m_ShaderStage), 1, outputFile);
                writtenBytes += sizeof(shaderProgramHeaders[i].second[j].m_ShaderStage);

                fwrite(&shaderProgramHeaders[i].second[j].m_ShaderDataSize, sizeof(shaderProgramHeaders[i].second[j].m_ShaderDataSize), 1, outputFile);
                writtenBytes += sizeof(shaderProgramHeaders[i].second[j].m_ShaderDataSize);

                fwrite(&shaderProgramHeaders[i].second[j].m_ShaderControlSize, sizeof(shaderProgramHeaders[i].second[j].m_ShaderControlSize), 1, outputFile);
                writtenBytes += sizeof(shaderProgramHeaders[i].second[j].m_ShaderControlSize);

                fwrite(&shaderProgramHeaders[i].second[j].m_ShaderControlOffset, sizeof(shaderProgramHeaders[i].second[j].m_ShaderControlOffset), 1, outputFile);
                writtenBytes += sizeof(shaderProgramHeaders[i].second[j].m_ShaderControlOffset);

                fwrite(shaderProgramHeaders[i].second[j].m_pShaderData, sizeof(char), shaderProgramHeaders[i].second[j].m_ShaderDataSize, outputFile);
                writtenBytes += sizeof(char) * shaderProgramHeaders[i].second[j].m_ShaderDataSize;

                    // Insert padding
                writtenBytes += PadFileBuffer(outputFile, padding[currentPaddingIndex++]);

                fwrite(shaderProgramHeaders[i].second[j].m_pShaderControl, sizeof(char), shaderProgramHeaders[i].second[j].m_ShaderControlSize, outputFile);
                writtenBytes += sizeof(char) * shaderProgramHeaders[i].second[j].m_ShaderControlSize;
            }
        }
    }

    if (outputFileHeader.m_TextureBlockOffset)
    {
            // Insert padding
        writtenBytes += PadFileBuffer(outputFile, padding[currentPaddingIndex++]);

            // Texture Block Header
        fwrite(&textureBlockHeader.m_NumTextures, sizeof(textureBlockHeader.m_NumTextures), 1, outputFile);
        writtenBytes += sizeof(textureBlockHeader.m_NumTextures);

        fwrite(&textureBlockHeader.m_MagicPadding, sizeof(textureBlockHeader.m_MagicPadding), 1, outputFile);
        writtenBytes += sizeof(textureBlockHeader.m_MagicPadding);

        fwrite(textureBlockHeader.m_pTextureOffsets, sizeof(uint64_t), textureBlockHeader.m_NumTextures, outputFile);
        writtenBytes += sizeof(uint64_t) * textureBlockHeader.m_NumTextures;

        for (uint32_t i = 0; i < textureDataHeaders.size(); ++i)
        {
                // Insert padding
            writtenBytes += PadFileBuffer(outputFile, padding[currentPaddingIndex++]);

            fwrite(&textureDataHeaders[i].m_TextureDataSize, sizeof(textureDataHeaders[i].m_TextureDataSize), 1, outputFile);
            writtenBytes += sizeof(textureDataHeaders[i].m_TextureDataSize);

            fwrite(&textureDataHeaders[i].m_GpuVersion, sizeof(textureDataHeaders[i].m_GpuVersion), 1, outputFile);
            writtenBytes += sizeof(textureDataHeaders[i].m_GpuVersion);

            fwrite(&textureDataHeaders[i].m_Alignment, sizeof(textureDataHeaders[i].m_Alignment), 1, outputFile);
            writtenBytes += sizeof(textureDataHeaders[i].m_Alignment);

            fwrite(&textureDataHeaders[i].m_Width, sizeof(textureDataHeaders[i].m_Width), 1, outputFile);
            writtenBytes += sizeof(textureDataHeaders[i].m_Width);

            fwrite(&textureDataHeaders[i].m_Height, sizeof(textureDataHeaders[i].m_Height), 1, outputFile);
            writtenBytes += sizeof(textureDataHeaders[i].m_Height);

            fwrite(&textureDataHeaders[i].m_Depth, sizeof(textureDataHeaders[i].m_Depth), 1, outputFile);
            writtenBytes += sizeof(textureDataHeaders[i].m_Depth);

            fwrite(&textureDataHeaders[i].m_NvnTextureTarget, sizeof(textureDataHeaders[i].m_NvnTextureTarget), 1, outputFile);
            writtenBytes += sizeof(textureDataHeaders[i].m_NvnTextureTarget);

            fwrite(&textureDataHeaders[i].m_NvnFormat, sizeof(textureDataHeaders[i].m_NvnFormat), 1, outputFile);
            writtenBytes += sizeof(textureDataHeaders[i].m_NvnFormat);

            fwrite(&textureDataHeaders[i].m_MipLevels, sizeof(textureDataHeaders[i].m_MipLevels), 1, outputFile);
            writtenBytes += sizeof(textureDataHeaders[i].m_MipLevels);

            fwrite(&textureDataHeaders[i].m_TextureNameLength, sizeof(textureDataHeaders[i].m_TextureNameLength), 1, outputFile);
            writtenBytes += sizeof(textureDataHeaders[i].m_TextureNameLength);

            fwrite(&textureDataHeaders[i].m_TextureDataOffset, sizeof(textureDataHeaders[i].m_TextureDataOffset), 1, outputFile);
            writtenBytes += sizeof(textureDataHeaders[i].m_TextureDataOffset);

            fwrite(textureDataHeaders[i].m_pTextureName, sizeof(char), textureDataHeaders[i].m_TextureNameLength, outputFile);
            writtenBytes += sizeof(char) * textureDataHeaders[i].m_TextureNameLength;

                // Insert padding
            writtenBytes += PadFileBuffer(outputFile, padding[currentPaddingIndex++]);

            fwrite(textureDataHeaders[i].m_pTextureData, sizeof(char), static_cast<size_t>(textureDataHeaders[i].m_TextureDataSize), outputFile);
            writtenBytes += sizeof(char) * textureDataHeaders[i].m_TextureDataSize;
        }
    }

    if (outputFileHeader.m_ModelBlockOffset)
    {
            // Insert padding
        writtenBytes += PadFileBuffer(outputFile, padding[currentPaddingIndex++]);

            // Model Block Header
        fwrite(&modelBlockHeader.m_NumModels, sizeof(modelBlockHeader.m_NumModels), 1, outputFile);
        writtenBytes += sizeof(modelBlockHeader.m_NumModels);

        fwrite(&modelBlockHeader.m_MagicPadding, sizeof(modelBlockHeader.m_MagicPadding), 1, outputFile);
        writtenBytes += sizeof(modelBlockHeader.m_MagicPadding);

        fwrite(modelBlockHeader.m_pModelOffsets, sizeof(uint64_t), modelBlockHeader.m_NumModels, outputFile);
        writtenBytes += sizeof(uint64_t) * modelBlockHeader.m_NumModels;

        for (uint32_t i = 0; i < modelHeaders.size(); ++i)
        {
                // Insert padding
            writtenBytes += PadFileBuffer(outputFile, padding[currentPaddingIndex++]);

                // Model Header
            fwrite(&modelHeaders[i].first.m_NumPrimitives, sizeof(modelHeaders[i].first.m_NumPrimitives), 1, outputFile);
            writtenBytes += sizeof(modelHeaders[i].first.m_NumPrimitives);

            fwrite(&modelHeaders[i].first.m_NvnDrawPrimitiveType, sizeof(modelHeaders[i].first.m_NvnDrawPrimitiveType), 1, outputFile);
            writtenBytes += sizeof(modelHeaders[i].first.m_NvnDrawPrimitiveType);

            fwrite(&modelHeaders[i].first.m_NumVertexAttributes, sizeof(modelHeaders[i].first.m_NumVertexAttributes), 1, outputFile);
            writtenBytes += sizeof(modelHeaders[i].first.m_NumVertexAttributes);

            fwrite(&modelHeaders[i].first.m_ModelNameLength, sizeof(modelHeaders[i].first.m_ModelNameLength), 1, outputFile);
            writtenBytes += sizeof(modelHeaders[i].first.m_ModelNameLength);

            fwrite(&modelHeaders[i].first.m_IndexBufferOffset, sizeof(modelHeaders[i].first.m_IndexBufferOffset), 1, outputFile);
            writtenBytes += sizeof(modelHeaders[i].first.m_IndexBufferOffset);

            fwrite(modelHeaders[i].first.m_pVertexAttributeOffsets, sizeof(uint64_t), modelHeaders[i].first.m_NumVertexAttributes, outputFile);
            writtenBytes += sizeof(uint64_t) * modelHeaders[i].first.m_NumVertexAttributes;

            fwrite(modelHeaders[i].first.m_pModelName, sizeof(char), modelHeaders[i].first.m_ModelNameLength, outputFile);
            writtenBytes += sizeof(char) * modelHeaders[i].first.m_ModelNameLength;

            std::vector<VertexAttributeHeader>& vertexAttributes = modelHeaders[i].second.first;

                // Vertex Attributes
            for (uint32_t j = 0; j < vertexAttributes.size(); ++j)
            {
                    // Insert padding
                writtenBytes += PadFileBuffer(outputFile, padding[currentPaddingIndex++]);

                fwrite(&vertexAttributes[j].m_AttributeStride, sizeof(vertexAttributes[j].m_AttributeStride), 1, outputFile);
                writtenBytes += sizeof(vertexAttributes[j].m_AttributeStride);

                fwrite(&vertexAttributes[j].m_NvnFormat, sizeof(vertexAttributes[j].m_NvnFormat), 1, outputFile);
                writtenBytes += sizeof(vertexAttributes[j].m_NvnFormat);

                fwrite(&vertexAttributes[j].m_AttributeDataSize, sizeof(vertexAttributes[j].m_AttributeDataSize), 1, outputFile);
                writtenBytes += sizeof(vertexAttributes[j].m_AttributeDataSize);

                fwrite(&vertexAttributes[j].m_AttributeNameLength, sizeof(vertexAttributes[j].m_AttributeNameLength), 1, outputFile);
                writtenBytes += sizeof(vertexAttributes[j].m_AttributeNameLength);

                fwrite(&vertexAttributes[j].m_AttributeDataOffset, sizeof(vertexAttributes[j].m_AttributeDataOffset), 1, outputFile);
                writtenBytes += sizeof(vertexAttributes[j].m_AttributeDataOffset);

                fwrite(&vertexAttributes[j].m_MagicPadding, sizeof(vertexAttributes[j].m_MagicPadding), 1, outputFile);
                writtenBytes += sizeof(vertexAttributes[j].m_MagicPadding);

                fwrite(vertexAttributes[j].m_pAttributeName, sizeof(char), vertexAttributes[j].m_AttributeNameLength, outputFile);
                writtenBytes += sizeof(char) * vertexAttributes[j].m_AttributeNameLength;

                    // Insert padding
                writtenBytes += PadFileBuffer(outputFile, padding[currentPaddingIndex++]);

                fwrite(vertexAttributes[j].m_AttributeData, sizeof(char), vertexAttributes[j].m_AttributeDataSize, outputFile);
                writtenBytes += sizeof(char) * vertexAttributes[j].m_AttributeDataSize;
            }

                // Index Buffer Header/Data
            fwrite(&modelHeaders[i].second.second.m_IndexBufferSize, sizeof(modelHeaders[i].second.second.m_IndexBufferSize), 1, outputFile);
            writtenBytes += sizeof(modelHeaders[i].second.second.m_IndexBufferSize);

            fwrite(&modelHeaders[i].second.second.m_NumIndices, sizeof(modelHeaders[i].second.second.m_NumIndices), 1, outputFile);
            writtenBytes += sizeof(modelHeaders[i].second.second.m_NumIndices);

            fwrite(&modelHeaders[i].second.second.m_IndexType, sizeof(modelHeaders[i].second.second.m_IndexType), 1, outputFile);
            writtenBytes += sizeof(modelHeaders[i].second.second.m_IndexType);

            fwrite(&modelHeaders[i].second.second.m_MagicPadding, sizeof(modelHeaders[i].second.second.m_MagicPadding), 1, outputFile);
            writtenBytes += sizeof(modelHeaders[i].second.second.m_MagicPadding);

            fwrite(modelHeaders[i].second.second.m_pIndexData, sizeof(char), modelHeaders[i].second.second.m_IndexBufferSize, outputFile);
            writtenBytes += sizeof(char) * modelHeaders[i].second.second.m_IndexBufferSize;
        }
    }

        // Check if number of bytes written to file does not match size of data generated
    assert(writtenBytes == currentOffset);

    fclose(outputFile);

    if (outputFileHeader.m_ShaderBlockOffset)
    {
        free(shaderBlockHeader.m_pShaderProgramOffsets);

        for (uint32_t i = 0; i < shaderProgramHeaders.size(); ++i)
        {
            free(shaderProgramHeaders[i].first.m_pShaderStageOffsets);
        }
    }

    if (outputFileHeader.m_TextureBlockOffset)
    {
        free(textureBlockHeader.m_pTextureOffsets);
    }

    if (outputFileHeader.m_ModelBlockOffset)
    {
        free(modelBlockHeader.m_pModelOffsets);
        for (uint32_t i = 0; i < modelHeaders.size(); ++i)
        {
            free(modelHeaders[i].first.m_pVertexAttributeOffsets);
        }
    }

    return true;
}//NOLINT(impl/function_size)

/*
 * IntermediateFileManager::OutputUniformBlockHeaderFiles
 * ------------------------------------------------------
 * This function takes the shader reflection data and the shader performance
 * data from the GLSLC compile object and outputs a C++ header/source file pair.
 * These files define some classes and static data for the vertex attributes
 * and uniform blocks to help with setting up the data to be passed to the
 * shader and keeps track of the location for the data in the shader per stage.
 * The uniform block classes have helper functions to set individual members at
 * the correct offset in the block. The shader performance data is output at the
 * top of the header file in a comment for each stage.
 */
bool IntermediateFileManager::OutputUniformBlockHeaderFiles(const std::string& path)
{
    for(std::unordered_map<std::string, ShaderCompileData>::iterator itr = m_ShaderPrograms.begin(); itr != m_ShaderPrograms.end(); ++itr)
    {
        std::ofstream file;
        file.open(path + (itr->first + "DataHelper.h"), std::fstream::out);

        if(!file.is_open())
        {
            std::cout << "Failed to create output header file for " << itr->first << "\n";
            return false;
        }

        std::cout << "    " << path << itr->first << "DataHelper.h\n";

        file << "/*--------------------------------------------------------------------------------*\n";
        file << "Copyright (C)Nintendo All rights reserved.\n";
        file << "\n";
        file << "These coded instructions, statements, and computer programs contain proprietary\n";
        file << "information of Nintendo and/or its licensed developers and are protected by\n";
        file << "national and international copyright laws. They may not be disclosed to third\n";
        file << "parties or copied or duplicated in any form, in whole or in part, without the\n";
        file << "prior written consent of Nintendo.\n";
        file << "\n";
        file << "The content herein is highly confidential and should be handled accordingly.\n";
        file << "*--------------------------------------------------------------------------------*/\n\n";

        file << "/*\n";
        file << " * " << itr->first << "DataHelper.h\n";
        file << " * -------------------------------------------------\n";
        file << " * This is a generated file from GLSLC shader\n";
        file << " * compile reflection information. Do not edit.\n";
        file << " * -------------------------------------------------\n";
        file << " */\n\n";

        ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
        file << "/*\n";

        file << " * Shader Program: " << itr->first << "\n * \n";

        GLSLCoutput* compileOutput = itr->second.m_CompileObject.lastCompiledResults->glslcOutput;

        for (uint32_t i = 0; i < compileOutput->numSections; ++i)
        {
            GLSLCsectionTypeEnum type = compileOutput->headers[i].genericHeader.common.type;

            if (type == GLSLC_SECTION_TYPE_GPU_CODE)
            {
                const GLSLCgpuCodeHeader * gpuHeader = &(compileOutput->headers[i].gpuCodeHeader);
                int perfStatsNdx = gpuHeader->perfStatsSectionNdx;

                const GLSLCperfStatsHeader * perfStatsHeader = &(compileOutput->headers[perfStatsNdx].perfStatsHeader);

                    /*
                     * GLSLCperfStatsData
                     * ------------------
                     * The GLSLCperfStatsData structure holds the performance data
                     * information for a specific shader stage. This information is
                     * present in the option was set in the GLSLCoptions structure
                     * before the last compile.
                     *
                     * magic                - Indicates the presence of perf stats data for
                     *                        sanity checking. This should be equal to the
                     *                        define GLSLC_PERF_STATS_SECTION_MAGIC_NUMBER.
                     * spillMem             - Spill memory data. (lmem and smem)
                     *
                     * nonSpillLMem         - Non-spill memory.
                     *
                     * throughputLimiter    - Throughput limiter information. Values are in terms
                     *                        of warps per cycle.
                     *
                     * loopData             - Loop unroll information. Contains number of partially
                     *                        unrolled loops and number of not unrolled loops.
                     *
                     * latency              - Cycles per pixel.
                     *
                     * occupancy            - Ratio of active warps to maximum number of warps
                     *                        supported.
                     *
                     * numDivergentBranches - Number of divergent branches.
                     *
                     * attributeMemUsage    - Attribute memory usage in bytes.
                     *
                     * programSize          - Program size in bytes.
                     *
                     */
                const GLSLCperfStatsData * perfStatsData = (const GLSLCperfStatsData *)(((const char*)(compileOutput)) + perfStatsHeader->common.dataOffset);

                std::string stage;

                switch(gpuHeader->stage)
                {
                    case NVNshaderStage::NVN_SHADER_STAGE_VERTEX:
                        stage = "Vertex";
                        break;
                    case NVNshaderStage::NVN_SHADER_STAGE_FRAGMENT:
                        stage = "Fragment";
                        break;
                    case NVNshaderStage::NVN_SHADER_STAGE_GEOMETRY:
                        stage = "Geometry";
                        break;
                    case NVNshaderStage::NVN_SHADER_STAGE_TESS_CONTROL:
                        stage = "Tesselation Control";
                        break;
                    case NVNshaderStage::NVN_SHADER_STAGE_TESS_EVALUATION:
                        stage = "Tesselation Evaluation";
                        break;
                    case NVNshaderStage::NVN_SHADER_STAGE_COMPUTE:
                        stage = "Compute";
                        break;
                    default:
                        assert(false);
                }

                file << " * Perf Statistics - Stage: "           << stage << "\n";
                file << " * -------------------------------------------------------------" << "\n";
                file << " * Latency:                           " << perfStatsData->latency << "\n";
                file << " * Number of lmem spill bytes:        " << perfStatsData->spillMem.numLmemSpillBytes << "\n";
                file << " * Number of lmem refill bytes:       " << perfStatsData->spillMem.numLmemRefillBytes << "\n";
                file << " * Number of smem spill bytes:        " << perfStatsData->spillMem.numSmemSpillBytes << "\n";
                file << " * Number of smem refill bytes:       " << perfStatsData->spillMem.numSmemRefillBytes << "\n";
                file << " * Lmem Spill Size:                   " << perfStatsData->spillMem.size << "\n";
                file << " * Local Memory Non-spill Loads:      " << perfStatsData->nonSpillLMem.loadBytes << "\n";
                file << " * Local Memory Non-spill Stores:     " << perfStatsData->nonSpillLMem.storeBytes << "\n";
                file << " * Non-spill Local Memory Size:       " << perfStatsData->nonSpillLMem.size << "\n";
                file << " * Occupancy:                         " << perfStatsData->occupancy << "\n";
                file << " * Number of Divergent Branches:      " << perfStatsData->numDivergentBranches << "\n";
                file << " * Attribute Memory Usage:            " << perfStatsData->attributeMemUsage << "\n";
                file << " * Program Size:                      " << perfStatsData->programSize << "\n";
                file << " * Issue Ltd Throughput:              " << perfStatsData->throughputLimiter.issue << "\n";
                file << " * FP Ltd Throughput:                 " << perfStatsData->throughputLimiter.fp << "\n";
                file << " * Half Ltd Throughput:               " << perfStatsData->throughputLimiter.half << "\n";
                file << " * Trancedental Ltd Throughput:       " << perfStatsData->throughputLimiter.trancedental << "\n";
                file << " * IPA Ltd Throughput:                " << perfStatsData->throughputLimiter.ipa << "\n";
                file << " * Shared Ltd Throughput:             " << perfStatsData->throughputLimiter.shared << "\n";
                file << " * ControlFlow Ltd Throughput:        " << perfStatsData->throughputLimiter.controlFlow << "\n";
                file << " * Texture/Load/Store Ltd Throughput: " << perfStatsData->throughputLimiter.texLoadStore << "\n";
                file << " * Reg Ltd Throughput:                " << perfStatsData->throughputLimiter.reg << "\n";
                file << " * Warp Ltd Throughput:               " << perfStatsData->throughputLimiter.warp << "\n * \n";
            }
        }

        file << " */\n\n";
        ///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

        file << "#pragma once\n";

        file << "#include <cstdint>\n";
        file << "#include <string>\n\n";

        file << "namespace " << itr->first << "\n";
        file << "{\n\n";

        std::vector<UniformBlockData> uniformBlocks;
        std::vector<UniformData> uniforms;
        std::vector<VertexAttributeData> vertexAttributes;

        GLSLCoutput * output = itr->second.m_CompileObject.lastCompiledResults->glslcOutput;

        for(uint32_t i = 0; i < output->numSections; ++i)
        {
            GLSLCsectionTypeEnum type = output->headers[i].genericHeader.common.type;

            if(type == GLSLC_SECTION_TYPE_REFLECTION)
            {
                void * data = ((char *)output) + (output->headers[i].genericHeader.common.dataOffset);

                const GLSLCprogramReflectionHeader * reflectionHeader = &(output->headers[i].programReflectionHeader);

                const char *stringPool = (const char*)data + reflectionHeader->stringPoolOffset;

                    /*
                     * GLSLCuniformBlockInfo
                     * ---------------------
                     * This structures holds information about a uniform block in
                     * the shader program.
                     *
                     * nameInfo           - Contains the length of the block's name
                     *                      and the offset to the name in the reflection
                     *                      header's string pool.
                     *
                     * size               - Size in bytes of the uniform block.
                     *
                     * numActiveVariables - Number of active variables defined in the block.
                     *
                     * stagesReferencedIn - Mask that represents what stages the block is
                     *                      used in.
                     *
                     * bindings           - Per stage bindings for the uniform block. The
                     *                      array index corresponds to the NVNshaderStage
                     *                      that the binding is for. If the binding value
                     *                      is -1, that means he block is not used in that
                     *                      stage.
                     *
                     */
                const GLSLCuniformBlockInfo * uniformBlock = (GLSLCuniformBlockInfo *)((char *)data + reflectionHeader->uniformBlockOffset);
                for (uint32_t j = 0; j < reflectionHeader->numUniformBlocks; ++j)
                {
                    UniformBlockData uniformBlockData;

                    for(uint32_t k = 0; k < 6; ++k)
                        uniformBlockData.m_Bindings[k] = uniformBlock->bindings[k];

                    uniformBlockData.m_Name               = stringPool + uniformBlock->nameInfo.nameOffset;
                    uniformBlockData.m_Size               = uniformBlock->size;
                    uniformBlockData.m_NumActiveVariables = uniformBlock->numActiveVariables;
                    uniformBlockData.m_StagesReferencedIn = static_cast<uint8_t>(uniformBlock->stagesReferencedIn);

                    uniformBlocks.push_back(uniformBlockData);

                    ++uniformBlock;
                }

                    /*
                     * GLSLCuniformInfo
                     * ---------------------
                     * This structure holds information about individual
                     * uniforms in the sader program. This includes uniforms
                     * that are part of uniform blocks as well sampler/image
                     * type uniforms.
                     *
                     * nameInfo           - Contains the length of the block's name
                     *                      and the offset to the name in the reflection
                     *                      header's string pool.
                     *
                     * type               - Data type of the uniform.
                     *
                     * blockNdx           - Index of the uniform in it's uniform block.
                     *                      (-1 if not a part of a block)
                     *
                     * blockOffset        - Byte offset into the uniform block.
                     *                      (-1 if not a part of a block)
                     *
                     * sizeOfArray        - Number of elements, if an array, 1 if not.
                     *
                     * arrayStride        - Stride between array elements.
                     *
                     * matrixStride       - Specifies stride between columns/rows of matrix
                     *                      depending on isRowMajor.
                     *
                     * isRowMajor         - Specifies row major vs column major.
                     *
                     * stagesReferencedIn - Mask that represents what stages the block is
                     *                      used in.
                     *
                     * bindings           - Per stage bindings for the uniform block. The
                     *                      array index corresponds to the NVNshaderStage
                     *                      that the binding is for. If the binding value
                     *                      is -1, that means he block is not used in that
                     *                      stage.
                     *
                     * kind               - Specifies between image, smapler, or normal uniform.
                     *
                     * isInUBO            - 1 if this uniform is in a uniform block, 0 otherwise.
                     *
                     * isArray            - 1 if this uniform is an array, 0 if otherwise.
                     *
                     */
                GLSLCuniformInfo * uniform = (GLSLCuniformInfo *)((char *)data + reflectionHeader->uniformOffset);
                for (uint32_t j = 0; j < reflectionHeader->numUniforms; ++j)
                {
                    UniformData uniformData;

                    uniformData.m_Name               = stringPool + uniform->nameInfo.nameOffset;
                    uniformData.m_Type               = uniform->type;
                    uniformData.m_BlockNdx           = uniform->blockNdx;
                    uniformData.m_BlockOffset        = uniform->blockOffset;

                    for (uint32_t k = 0; k < 6; ++k)
                    {
                        uniformData.m_Bindings[k] = uniform->bindings[k];
                    }

                    uniformData.m_IsArray            = uniform->isArray;
                    uniformData.m_SizeOfArray        = uniform->sizeOfArray;
                    uniformData.m_ArrayStride        = uniform->arrayStride;
                    uniformData.m_MatrixStride       = uniform->matrixStride;
                    uniformData.m_IsRowMajor         = uniform->isRowMajor;
                    uniformData.m_StagesReferencedIn = static_cast<uint8_t>(uniform->stagesReferencedIn);

                    if(uniformData.m_BlockNdx != -1)
                    {
                        uniformBlocks[uniformData.m_BlockNdx].m_Uniforms.push_back(uniformData);
                    }
                    else
                    {
                        uniforms.push_back(uniformData);
                    }

                    ++uniform;
                }

                    /*
                     * GLSLCprogramInputInfo
                     * ---------------------
                     * This structure holds information about attributes
                     * used by the shader program.
                     *
                     * nameInfo           - Contains the length of the block's name
                     *                      and the offset to the name in the reflection
                     *                      header's string pool.
                     *
                     * type               - Data type of the uniform.
                     *
                     * sizeOfArray        - Number of elements, if an array, 1 if not.
                     *
                     * location           - Location this input is assigned to.
                     *                      (-1 if not a vert/frag program)
                     *
                     * stagesReferencedIn - Per stage bindings for the uniform block. The
                     *                      array index corresponds to the NVNshaderStage
                     *                      that the binding is for. If the binding value
                     *                      is -1, that means he block is not used in that
                     *                      stage.
                     *
                     * isArray            - 1 if this input is an array, 0 if otherwise.
                     *
                     * isPerPatch         - 1 if this input is per patch, 0 if otherwise.
                     *
                     */
                GLSLCprogramInputInfo * programInput = (GLSLCprogramInputInfo *)((char *)data + reflectionHeader->programInputsOffset);
                for(uint32_t j = 0; j < reflectionHeader->numProgramInputs; ++j)
                {
                    VertexAttributeData vertexAttributeData;
                    vertexAttributeData.m_Name               = stringPool + programInput->nameInfo.nameOffset;
                    vertexAttributeData.m_Type               = programInput->type;
                    vertexAttributeData.m_IsArray            = programInput->isArray;
                    vertexAttributeData.m_SizeOfArray        = programInput->sizeOfArray;
                    vertexAttributeData.m_Location           = programInput->location;
                    vertexAttributeData.m_IsPerPatch         = programInput->isPerPatch;
                    vertexAttributeData.m_StagesReferencedIn = static_cast<uint8_t>(programInput->stagesReferencedIn);

                    vertexAttributes.push_back(vertexAttributeData);

                    ++programInput;
                }
            }
        }

        file << "    // Holds a list of attribute data and provides binding information for each attribute\n";
        file << "class Attributes\n";
        file << "{\n";
        file << "    public:\n";
        file << "            // Holds information describing a given attribute\n";
        file << "            // Data fields are filled with data from the GLSLC compile output\n";
        file << "        struct AttributeData\n";
        file << "        {\n";
        file << "            AttributeData(const std::string& name = \"\",\n";
        file << "                          int32_t location = -1,\n";
        file << "                          uint8_t stagesReferencedIn = 0) : m_Name(name),\n";
        file << "                                                            m_Location(location),\n";
        file << "                                                            m_StagesReferencedIn(stagesReferencedIn)\n";
        file << "            {\n";
        file << "            }\n\n";
        file << "            std::string m_Name;              // Name of the attribute in the shader\n";
        file << "            int32_t m_Location;              // Binding location\n";
        file << "            uint8_t m_StagesReferencedIn;    // Bit flag of stages that the attribute is referenced in, of type NVNshaderStageBits\n";
        file << "        };\n\n";
        file << "            // Get the binding location for the attribute with the given name\n";
        file << "        static int32_t GetAttributeLocation(const std::string& name)\n";
        file << "        {\n";
        file << "            for(uint32_t i = 0; i < m_NumAttributes; ++i)\n";
        file << "            {\n";
        file << "                if(m_Attributes[i].m_Name == name)\n";
        file << "                    return m_Attributes[i].m_Location;\n";
        file << "            }\n\n";
        file << "            return -1;\n";
        file << "        }\n\n";
        file << "            // Get the number of vertex attributes in the shader program\n";
        file << "        static uint32_t GetNumAttributes()\n";
        file << "        {\n";
        file << "            return m_NumAttributes;\n";
        file << "        }\n\n";
        file << "            // Get the stages the attribute is referenced in\n";
        file << "        static uint8_t GetStagesReferencedIn(const std::string& name)\n";
        file << "        {\n";
        file << "            for(uint32_t i = 0; i < m_NumAttributes; ++i)\n";
        file << "            {\n";
        file << "                if(m_Attributes[i].m_Name == name)\n";
        file << "                    return m_Attributes[i].m_StagesReferencedIn;\n";
        file << "            }\n\n";
        file << "            return 0;\n";
        file << "        }\n\n";
        file << "        static const AttributeData* GetAttributeList()\n";
        file << "        {\n";
        file << "            return m_Attributes;\n";
        file << "        }\n\n";
        file << "    private:\n";
        file << "        const static uint32_t m_NumAttributes = " << vertexAttributes.size() << ";     // Total number of attributes\n";
        file << "        const static AttributeData m_Attributes[" << vertexAttributes.size() << "];    // Static array of attribute data \n";
        file << "};\n\n";

        for(uint32_t i = 0; i < uniforms.size(); ++i)
        {
            file << "    // Helper class for the " << uniforms[i].m_Name << "uniform\n";
            file << "class " << uniforms[i].m_Name << "UniformData\n";
            file << "{\n";
            file << "    public:\n";
            file << "        static const char name[] = \"" << uniforms[i].m_Name << "\";    // Name of the uniform\n\n";
            file << "            // Binding locations for the uniform\n";
            file << "        static const int32_t m_Bindings[6];\n";
            file << "        static const uint8_t m_StagesReferencedIn = " << (uint32_t)uniforms[i].m_StagesReferencedIn << ";    // Stages that the uniform is referenced in, of type NVNshaderStageBits\n";
            file << "};\n\n";

            file << "const int32_t " << uniforms[i].m_Name << "UniformData::m_Bindings[6] = { " << uniforms[i].m_Bindings[0];

            for(uint32_t j = 1; j < 6; ++j)
            {
                file << ", " << uniforms[i].m_Bindings[j];
            }

            file << " };\n\n";
        }

        for(uint32_t i = 0; i < uniformBlocks.size(); ++i)
        {
            file << "    // Helper class for the " << uniformBlocks[i].m_Name << " uniform block\n";
            file << "class " << uniformBlocks[i].m_Name << "UniformBlockData\n";
            file << "{\n";
            file << "    public:\n";

            for(uint32_t j = 0; j < uniformBlocks[i].m_Uniforms.size(); ++j)
            {
                const UniformData& uniform = uniformBlocks[i].m_Uniforms[j];
                std::string typeName = GetReferenceTypeName(uniform.m_Type, "uniformData");

                file << "            // Sets the data at the appropriate offset for the uniform block member: " << uniform.m_Name << "\n";
                file << "        void SetUniform_" << uniform.m_Name << "(" << typeName << ")\n";
                file << "        {\n";
                file << "            memcpy(&m_BlockData[" << uniform.m_BlockOffset << "], &uniformData, sizeof(uniformData));\n";
                file << "        }\n\n";
            }

            file << "            // Get the binding location for the uniform block\n";
            file << "            // shaderStage is of type NVNshaderStage\n";
            file << "        static uint32_t GetBinding(uint32_t shaderStage)\n";
            file << "        {\n";
            file << "            return m_Bindings[shaderStage];\n";
            file << "        }\n\n";
            file << "            // Get the stages the uniform block is referenced in\n";
            file << "        static uint8_t GetStagesReferencedIn()\n";
            file << "        {\n";
            file << "            return m_StagesReferencedIn;\n";
            file << "        }\n\n";
            file << "    private:\n";
            file << "        char m_BlockData[" << uniformBlocks[i].m_Size << "];    // Block of data representing the uniform block\n\n";
            file << "            // Binding location of the uniform block\n";
            file << "        static const int32_t m_Bindings[6];\n";
            file << "        static const uint8_t m_StagesReferencedIn = " << (uint32_t)uniformBlocks[i].m_StagesReferencedIn << ";    // Bit flag of stages that the attribute is referenced in, of type NVNshaderStageBits\n";
            file << "};\n\n";
        }


        file << "}\n\n";


            // Source file for static member initilization
        std::ofstream sourceFile;
        sourceFile.open(path + (itr->first + "DataHelper.cpp"), std::fstream::out);

        if(!sourceFile.is_open())
        {
            std::cout << "Failed to create output source file for " << itr->first << "\n";
            return false;
        }

        std::cout << "    " << path << itr->first << "DataHelper.cpp\n";

        sourceFile << "/*--------------------------------------------------------------------------------*\n";
        sourceFile << "Copyright (C)Nintendo All rights reserved.\n";
        sourceFile << "\n";
        sourceFile << "These coded instructions, statements, and computer programs contain proprietary\n";
        sourceFile << "information of Nintendo and/or its licensed developers and are protected by\n";
        sourceFile << "national and international copyright laws. They may not be disclosed to third\n";
        sourceFile << "parties or copied or duplicated in any form, in whole or in part, without the\n";
        sourceFile << "prior written consent of Nintendo.\n";
        sourceFile << "\n";
        sourceFile << "The content herein is highly confidential and should be handled accordingly.\n";
        sourceFile << "*--------------------------------------------------------------------------------*/\n\n";

        sourceFile << "/*\n";
        sourceFile << " * " << itr->first << "DataHelper.cpp\n";
        sourceFile << " * -------------------------------------------------\n";
        sourceFile << " * This is a generated file from GLSLC shader\n";
        sourceFile << " * compile reflection information. Do not edit.\n";
        sourceFile << " * -------------------------------------------------\n";
        sourceFile << " */\n\n";

        sourceFile << "#include \"" << itr->first << "DataHelper.h\"\n\n";

        sourceFile <<"    // The AttributeData array is filled with values from the GLSLC shader compile output\n";
        sourceFile << "const " << itr->first << "::Attributes::AttributeData " << itr->first << "::Attributes::m_Attributes[" << vertexAttributes.size() << "] = { ";
        for(uint32_t i = 0; i < vertexAttributes.size(); ++i)
        {
            sourceFile << itr->first << "::Attributes::AttributeData(\"" << vertexAttributes[i].m_Name << "\", " << vertexAttributes[i].m_Location << ", " << (uint32_t)vertexAttributes[i].m_StagesReferencedIn << ")";

            if (i + 1 < vertexAttributes.size())
            {
                sourceFile << ", ";
            }
        }

        sourceFile << " };\n\n";

        for(uint32_t i = 0; i < uniformBlocks.size(); ++i)
        {
            sourceFile << "const int32_t " << itr->first << "::" << uniformBlocks[i].m_Name << "UniformBlockData::m_Bindings[6] = { " << uniformBlocks[i].m_Bindings[0];

            for(uint32_t j = 1; j < 6; ++j)
            {
                sourceFile << ", " << uniformBlocks[i].m_Bindings[j];
            }

            sourceFile << " };\n\n";
        }
    }

    return true;
}//NOLINT(impl/function_size)

/*
 * IntermediateFileManager::GetReferenceTypeName
 * ---------------------------------------------
 * Helper function that converts a given type enum and variable name
 * into a string to be used by the shader header/source helper file
 * output. The list of types here is not complete and is simply the
 * types that are used by the shaders in these tutorials.
 */
std::string IntermediateFileManager::GetReferenceTypeName(GLSLCpiqTypeEnum type, const std::string& variableName)
{
    switch(type)
    {
        case GLSLC_PIQ_TYPE_UINT:
            return "const uint32_t& " + variableName;
        case GLSLC_PIQ_TYPE_UINT64:
            return "const uint64_t& " + variableName;
        case GLSLC_PIQ_TYPE_FLOAT:
            return "const float& " + variableName;
        case GLSLC_PIQ_TYPE_FLOAT_VEC2:
            return "const float (&" + variableName + ")[2]";
        case GLSLC_PIQ_TYPE_FLOAT_VEC3:
            return "const float (&" + variableName + ")[3]";
        case GLSLC_PIQ_TYPE_FLOAT_VEC4:
            return "const float (&" + variableName + ")[4]";
        case GLSLC_PIQ_TYPE_MAT3:
            return "const float (&" + variableName + ")[9]";
        case GLSLC_PIQ_TYPE_MAT4X3:
            return "const float (&" + variableName + ")[12]";
        case GLSLC_PIQ_TYPE_MAT4:
            return "const float (&" + variableName + ")[16]";
        default:
        {
            std::cout << "Unlisted variable type, case needs to be added.\n";
            assert("Unlisted variable type, case needs to be added." && 0);
            return std::string();
        }
    };
}

/*
 * IntermediateFileManager::LoadTextureData
 * ----------------------------------------
 * Uses the NVN image library to load the raw texture data
 * from the files specified by the config text file. The raw
 * texture data is stored in the nvnTool::texpkg::RawImage class.
 */
bool IntermediateFileManager::LoadTextureData(const std::string& path)
{
        /*
         * Texpkg Raw Image
         * ----------------
         * The RawImage structure is a wrapper around a generic pitch-linear
         * image with member functions to get information about the texture.
         * A RawImage can be created with specific settings with the Create
         * function. An image created through the image loader will be setup
         * based on the texture file that is loaded.
         */

        /* Textures */
    for(std::vector<TextureData>::iterator itr = m_RawTextureData.begin(); itr != m_RawTextureData.end(); ++itr)
    {
            /* Create a RawImage struct for the base texture to be loaded. */
        nvnTool::texpkg::RawImage* temp = g_NvnImageLib.createRawImage();

        std::string tempFile = path + itr->m_FileName;
        std::wstring filename(tempFile.begin(), tempFile.end());

            /* Load the texture in from file into the RawImage. */
        TPError err = g_ImageLoader->Load(filename.c_str(), temp);

        if(err != TP_OK)
        {
            std::cout << "Failed to load image: " << tempFile.c_str() << "\n";
            return false;
        }

            /* If the texture has mip maps that need to be loaded in... */
        if(itr->m_MipMaps.size())
        {
            nvnTool::texpkg::RawImage* image = g_NvnImageLib.createRawImage();

                /* Create a new RawImage that has the appropriate number of mip maps. */
            TPError create = image->Create(temp->Target(), temp->Width(), temp->Height(), temp->Depth(), temp->Format(), static_cast<uint32_t>(itr->m_MipMaps.size()) + 1);

            if(create != TP_OK)
            {
                std::cout << "Failed to created image: " << tempFile.c_str() << "\n";
                return false;
            }

                /* Grab a pointer to the base level texture data and copy the loaded texture data into it. */
            nvnTool::texpkg::RawMipMapLevel* base = image->MipMapLevel(0);
            memcpy(base->data, temp->GetData(), base->dataSize);

                /* For each mip level... */
            for(uint32_t i = 0; i < itr->m_MipMaps.size(); ++i)
            {
                    /* Create a RawImage and load the texture file for the mip level. */
                nvnTool::texpkg::RawImage* tempMip = g_NvnImageLib.createRawImage();
                std::string tempFileMip = path + itr->m_MipMaps[i].m_FileName;
                err = g_ImageLoader->Load(std::wstring(tempFileMip.begin(), tempFileMip.end()).c_str(), tempMip);

                if(err != TP_OK)
                {
                    std::cout << "Failed to load image: " << tempFileMip.c_str() << "\n";
                    return false;
                }

                    /* Grab a poiner to the mip level data and copy the loaded data into it. */
                nvnTool::texpkg::RawMipMapLevel* mipMap = image->MipMapLevel(itr->m_MipMaps[i].m_Level);
                memcpy(mipMap->data, tempMip->GetData(), mipMap->dataSize);
            }

            itr->m_pRawImage = image;
        }
        else
        {
            itr->m_pRawImage = temp;
        }

    }

        /* Cube maps */
    for(std::vector<CubeMapData>::iterator itr = m_RawCubeMapData.begin(); itr != m_RawCubeMapData.end(); ++itr)
    {
        std::vector<nvnTool::texpkg::RawImage*> faces;
        faces.resize(itr->m_Faces.size(), NULL);

            /* For each face... */
        for(uint32_t i = 0; i < itr->m_Faces.size(); ++i)
        {
                /* Load in the cube map face from file and create a RawImage for it. */
            faces[i] = g_NvnImageLib.createRawImage();

            std::string tempFile = path + itr->m_Faces[i];
            std::wstring filename(tempFile.begin(), tempFile.end());

            TPError err = g_ImageLoader->Load(filename.c_str(), faces[i]);

            if(err != TP_OK)
            {
                std::cout << "Failed to load image: " << itr->m_Faces[i].c_str() << "\n";
                return false;
            }
        }

            /* Create a RawImage for the full cube map with the appropriate parameters. */
        nvnTool::texpkg::RawImage* cubeImage = g_NvnImageLib.createRawImage();

        TPError create = cubeImage->Create(NVNtextureTarget::NVN_TEXTURE_TARGET_CUBEMAP, faces[0]->Width(), faces[0]->Height(), faces[0]->Depth(), faces[0]->Format(), 1, 6);

        if(create != TP_OK)
        {
            std::cout << "Failed to created image: " << itr->m_Name.c_str() << "\n";
            return false;
        }

            /* Set the cube face mask to all to denote that all cube map faces are being used. */
        cubeImage->SetCubeFaceMask(nvnTool::texpkg::RawTextureCubeFace::CUBE_FACE_All);

            /* Copy the data for each face at the appropriate subimage. */
        for(uint32_t i = 0; i < itr->m_Faces.size(); ++i)
        {
            nvnTool::texpkg::RawMipMapLevel* mipMap = cubeImage->MipMapLevel(0, i);

            memcpy(mipMap->data, faces[i]->GetData(), mipMap->dataSize);
        }

        itr->m_pRawImage = cubeImage;
    }


    return true;
}

/*
 * IntermediateFileManager::ConvertTextures
 * ----------------------------------------
 * Takes the raw texture data that was loaded in and runs
 * it through the texpkg libraries conversion process. This
 * results in the texture being converted to a format that
 * allows the target harware to use it directly.
 */
bool IntermediateFileManager::ConvertTextures()
{
        /*
         * NVNHWTexture
         * ------------
         * The NVNHWTexture hold the texture data from a RawImage after
         * it has been converted to the hardware optimized format. This
         * converted format can also be used on Windows if it is given
         * to the texture builder through the SetPackagedTextureData function.
         */

        /* Convert the list of textures. */
    for (std::vector<TextureData>::iterator itr = m_RawTextureData.begin(); itr != m_RawTextureData.end(); ++itr)
    {
        TPError err = g_HwTextureExporter->Convert(itr->m_pRawImage, &m_ConvertedTextureData[itr->m_FileName]);

        if(err != TP_OK)
        {
            std::cout << "Failed to convert texture: " << itr->m_FileName.c_str() << "\n";
            return false;
        }
    }

        /* Convert the cube maps. */
    for(std::vector<CubeMapData>::iterator itr = m_RawCubeMapData.begin(); itr != m_RawCubeMapData.end(); ++itr)
    {
        TPError err = g_HwTextureExporter->Convert(itr->m_pRawImage, &m_ConvertedTextureData[itr->m_Name]);

        if(err != TP_OK)
        {
            std::cout << "Failed to convert texture: " << itr->m_Name.c_str() << "\n";
            return false;
        }
    }

    return true;
}

/*
 * IntermediateFileManager::LoadShaderConfig
 * -----------------------------------------
 * Load the shader section of the config file.
 */
void IntermediateFileManager::LoadShaderConfig(std::ifstream& configFile)
{
    std::string text, programName;

    configFile >> text;

    if (text == "Name:")
    {
        configFile >> programName;
    }
    else
    {
        std::cout << "Missing Shader Program Name\n";
        assert(0 && "Missing Shader Program Name\n");
    }

    bool loop = true;
    while (loop)
    {
        configFile >> text;

        if (text == "VertexShader:")
        {
            configFile >> text;
            m_ShaderPrograms[programName].m_Vert = text;
        }
        else if (text == "FragmentShader:")
        {
            configFile >> text;
            m_ShaderPrograms[programName].m_Frag = text;
        }
        else if (text == "ComputeShader:")
        {
            configFile >> text;
            m_ShaderPrograms[programName].m_Comp = text;
        }
        else if (text == "TessEvalShader:")
        {
            configFile >> text;
            m_ShaderPrograms[programName].m_TessEval = text;
        }
        else if (text == "TessContShader:")
        {
            configFile >> text;
            m_ShaderPrograms[programName].m_TessCont = text;
        }
        else if (text == "GeometryShader:")
        {
            configFile >> text;
            m_ShaderPrograms[programName].m_Geom = text;
        }
        else if (text == "End")
        {
            loop = false;
            break;
        }
    }
}

/*
 * IntermediateFileManager::LoadTextureConfig
 * ------------------------------------------
 * Load the texture section of the config file.
 */
void IntermediateFileManager::LoadTextureConfig(std::ifstream& configFile)
{
    std::string text, textureName;

    configFile >> text;

    if (text == "Name:")
    {
        configFile >> textureName;
    }
    else
    {
        std::cout << "Missing Texture File Name\n";
        assert(0 && "Missing Texture File Name\n");
    }

    m_RawTextureData.push_back(TextureData());

    m_RawTextureData.back().m_FileName = textureName;
    m_RawTextureData.back().m_pRawImage = NULL;

    bool loop = true;
    while (loop)
    {
        configFile >> text;

        if (text == "End")
        {
            loop = false;
            break;
        }
        else if (text == "MipmapBegin")
        {
            m_RawTextureData.back().m_MipMaps.push_back(MipMapData());

            configFile >> text;

            if (text != "Name:")
            {
                std::cout << "Missing Mipmap File Name\n";
                assert(0 && "Missing Mipmap File Name\n");
            }

            configFile >> m_RawTextureData.back().m_MipMaps.back().m_FileName;

            configFile >> text;

            if (text != "Level:")
            {
                std::cout << "Missing Mipmap Level\n";
                assert(0 && "Missing Mipmap Level\n");
            }

            configFile >> m_RawTextureData.back().m_MipMaps.back().m_Level;

            configFile >> text;

            if (text != "End")
            {
                std::cout << "Missing Mipmap End\n";
                assert(0 && "Missing Mipmap End\n");
            }
        }
    }
}

/*
 * IntermediateFileManager::LoadCubeMapConfig
 * ------------------------------------------
 * Load the cube map section of the config file.
 */
void IntermediateFileManager::LoadCubeMapConfig(std::ifstream& configFile)
{
    std::string text;

    m_RawCubeMapData.push_back(CubeMapData());

    configFile >> text;

    if (text != "Name:")
    {
        std::cout << "Missing Cube Map Name\n";
        assert(0 && "Missing Cube Map Name\n");
    }

    configFile >> m_RawCubeMapData.back().m_Name;
    m_RawCubeMapData.back().m_pRawImage = NULL;

    bool loop = true;
    while (loop)
    {
        configFile >> text;

        if (text == "End")
        {
            loop = false;
            break;
        }
        else if (text == "Face:")
        {
            m_RawCubeMapData.back().m_Faces.push_back(std::string());
            configFile >> m_RawCubeMapData.back().m_Faces.back();
        }
        else
        {
            std::cout << "Unknown Cube Map Element: " << text << "\n";
            assert(0 && "Unknown Cube Map Element\n");
        }
    }

    if (m_RawCubeMapData.back().m_Faces.size() != 6)
    {
        std::cout << "Cube map with less than 6 faces: " << m_RawCubeMapData.back().m_Name.c_str() << "\n";
        assert("Cube map with less than 6 faces" && 0);
    }
}

/*
 * IntermediateFileManager::LoadModelConfig
 * ----------------------------------------
 * Load the model section of the config file.
 */
void IntermediateFileManager::LoadModelConfig(std::ifstream& configFile)
{
    Model model;

    std::string text;

    configFile >> text;

    if (text != "Name:")
    {
        std::cout << "Missing Model Name\n";
        assert(0 && "Missing Model Name\n");
    }

    configFile >> model.m_Name;

    configFile >> text;
    if (text != "NumAttributes:")
    {
        std::cout << "Missing NumAttributes\n";
        assert(0 && "Missing NumAttributes\n");
    }

    configFile >> model.m_NumVertexAttributes;

    configFile >> text;
    if (text != "NumPrimitives:")
    {
        std::cout << "Missing NumPrimitives\n";
        assert(0 && "Missing NumPrimitives\n");
    }

    configFile >> model.m_NumPrimitives;

    configFile >> text;
    if (text != "nvnDrawPrimitiveType:")
    {
        std::cout << "Missing nvnDrawPrimitiveType\n";
        assert(0 && "Missing nvnDrawPrimitiveType\n");
    }

    configFile >> model.m_NvnDrawPrimitiveType;

    bool loop = true;
    while (loop)
    {
        configFile >> text;

        if (text == "VertexAttributeBegin")
        {
            LoadVertexAttributeConfig(configFile, model);
        }
        else if (text == "IndicesBegin")
        {
            LoadIndices(configFile, model);
        }
        else if (text == "End")
        {
            loop = false;
        }
    }

    m_RawModelData.push_back(model);
}

/*
 * IntermediateFileManager::LoadVertexAttributeConfig
 * --------------------------------------------------
 * Load the vertex attribute section of the config file.
 */
void IntermediateFileManager::LoadVertexAttributeConfig(std::ifstream& configFile, Model& model)
{
    std::string text;
    VertexAttribute vertexAttribute;

    configFile >> text;

    if (text != "Name:")
    {
        std::cout << "Missing Attribute Name\n";
        assert(0 && "Missing Attribute Name\n");
    }

    configFile >> vertexAttribute.m_Name;

    configFile >> text;

    if (text != "DataSize:")
    {
        std::cout << "Missing Attribute DataSize\n";
        assert(0 && "Missing Attribute DataSize\n");
    }

    configFile >> vertexAttribute.m_DataSize;

    configFile >> text;

    if (text != "ElementSize:")
    {
        std::cout << "Missing Attribute ElementSize\n";
        assert(0 && "Missing Attribute ElementSize\n");
    }

    configFile >> vertexAttribute.m_ElementSize;

    configFile >> text;

    if (text != "Stride:")
    {
        std::cout << "Missing Attribute Stride\n";
        assert(0 && "Missing Attribute Stride\n");
    }

    configFile >> vertexAttribute.m_Stride;

    configFile >> text;

    if (text != "nvnFormat:")
    {
        std::cout << "Missing Attribute nvnFormat\n";
        assert(0 && "Missing Attribute nvnFormat\n");
    }

    configFile >> vertexAttribute.m_NvnFormat;

    configFile >> text;

    if (text != "DataBegin")
    {
        std::cout << "Missing Attribute Data\n";
        assert(0 && "Missing Attribute Data\n");
    }

    vertexAttribute.m_pData = malloc(vertexAttribute.m_DataSize);
    void* temp = vertexAttribute.m_pData;

    bool loop = true;
    while (loop)
    {
        configFile >> text;

        if (text == "End")
        {
            loop = false;
            break;
        }

        if (vertexAttribute.m_ElementSize == sizeof(float))
        {
            *(float*)temp = (float)std::atof(text.c_str());
            temp = (float*)temp + 1;
        }
        else if (vertexAttribute.m_ElementSize == sizeof(short))
        {
            *(short*)temp = static_cast<short>(std::atoi(text.c_str()));
            temp = (short*)temp + 1;
        }
        else if (vertexAttribute.m_ElementSize == sizeof(char))
        {
            *(char*)temp = static_cast<char>(std::atoi(text.c_str()));
            temp = (char*)temp + 1;
        }
    }

    configFile >> text;
    if (text != "End")
    {
        std::cout << "Missing Attribute End\n";
        assert(0 && "Missing Attribute End\n");
    }

    model.m_VertexAttributes.push_back(vertexAttribute);
}

/*
 * IntermediateFileManager::LoadIndices
 * ------------------------------------
 * Load the index section of the config file.
 */
void IntermediateFileManager::LoadIndices(std::ifstream& configFile, Model& model)
{
    std::string text;
    IndexData& indexData = model.m_IndexData;

    configFile >> text;

    if (text != "nvnType:")
    {
        std::cout << "Missing Indices NVN Type\n";
        assert(0 && "Missing Indices NVN Type\n");
    }

    configFile >> indexData.m_IndexType;

    configFile >> text;

    if (text != "DataSize:")
    {
        std::cout << "Missing Indices DataSize\n";
        assert(0 && "Missing Indices DataSize\n");
    }

    configFile >> indexData.m_DataSize;

    configFile >> text;

    if (text != "Stride:")
    {
        std::cout << "Missing Indices Stride\n";
        assert(0 && "Missing Indices Stride\n");
    }

    configFile >> indexData.m_Stride;

    configFile >> text;

    if (text != "DataBegin")
    {
        std::cout << "Missing Indices Data\n";
        assert(0 && "Missing Indices Data\n");
    }

    indexData.m_pData = malloc(indexData.m_DataSize);

    void* temp = indexData.m_pData;

    bool loop = true;
    while (loop)
    {
        configFile >> text;

        if (text == "End")
        {
            loop = false;
            break;
        }

        if (indexData.m_Stride == sizeof(uint32_t))
        {
            *(uint32_t*)temp = std::atoi(text.c_str());
            temp = (uint32_t*)temp + 1;
        }
        else if (indexData.m_Stride == sizeof(short))
        {
            *(short*)temp = static_cast<short>(std::atoi(text.c_str()));
            temp = (short*)temp + 1;
        }
        else if (indexData.m_Stride == sizeof(char))
        {
            *(char*)temp = static_cast<char>(std::atoi(text.c_str()));
            temp = (char*)temp + 1;
        }
    }

    configFile >> text;
    if (text != "End")
    {
        std::cout << "Missing Indices End\n";
        assert(0 && "Missing Indices End\n");
    }
}

/*
 * IntermediateFileManager::LoadConfigFile
 * ---------------------------------------
 * Load the config file.
 */
bool IntermediateFileManager::LoadConfigFile(std::string fileName)
{
    std::ifstream configFile;
    configFile.open(fileName);

    if (!configFile.is_open())
    {
        return false;
    }

    while (!configFile.eof())
    {
        std::string text;
        configFile >> text;

        if (text == "ShaderProgramBegin")
        {
            LoadShaderConfig(configFile);
        }
        else if (text == "TextureBegin")
        {
            LoadTextureConfig(configFile);
        }
        else if (text == "CubeMapBegin")
        {
            LoadCubeMapConfig(configFile);
        }
        else if (text == "ModelBegin")
        {
            LoadModelConfig(configFile);
        }
    }

    return true;
}
