﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cmath>
#include <nvn/nvn_FuncPtrInline.h>
#include <nvn/nvn_FuncPtrImpl.h>
#include <nvngdSupport/TutorialBaseClass.h>
#include <nvngdSupport/MemoryPool.h>
#include <nvngdSupport/TutorialUtil.h>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

static const size_t g_CommandMemorySize = 512;
static const size_t g_ControlMemorySize = 512;
static const int    g_NumColorBuffers = 2;

class GettingStartedWithNVN : public TutorialBaseClass
{
public:
    GettingStartedWithNVN();
    virtual ~GettingStartedWithNVN();
    virtual void Init(PFNNVNBOOTSTRAPLOADERPROC loader, NVNnativeWindow nativeWindow);
    virtual void Shutdown();

    virtual void Draw(uint64_t millisec);
    virtual void Resize(int width, int height);

private:
    static void NVNAPIENTRY DebugLayerCallback(
        NVNdebugCallbackSource source,
        NVNdebugCallbackType type,
        int id,
        NVNdebugCallbackSeverity severity,
        const char* message,
        void* pUser
    );

    int UpdateRenderTargets();

    NVNdevice         m_Device;
    NVNqueue          m_Queue;
    void*             m_pQueueMemory;

    MemoryPool*       m_pCommandMemoryPool;

    ptrdiff_t         m_CommandPoolOffset;
    void*             m_pControlPool;
    NVNcommandBuffer  m_CommandBuffer;
    NVNcommandHandle  m_CommandHandle;

    ptrdiff_t         m_RenderTargetCommandPoolOffset;
    void*             m_pRenderTargetControlPool;
    NVNcommandBuffer  m_RenderTargetCommandBuffer;
    NVNcommandHandle  m_RenderTargetCommandHandle;

    NVNtextureBuilder m_RenderTargetBuilder;
    NVNtexture*       m_RenderTargets[g_NumColorBuffers];

    NVNsync           m_CommandBufferSync;

    MemoryPool*       m_pRenderTargetMemoryPool;

    NVNwindow*        m_pWindow;
    NVNwindowBuilder  m_WindowBuilder;
    size_t            m_ColorTargetSize;
};

/*
* GettingStartedWithNVN Constructor
* ----------------------------
* Sets up default values for the members of the class.
*/
GettingStartedWithNVN::GettingStartedWithNVN() :
    m_pQueueMemory(NULL),
    m_pCommandMemoryPool(NULL),
    m_pControlPool(NULL),
    m_CommandHandle(NULL),
    m_pRenderTargetControlPool(NULL),
    m_RenderTargetCommandHandle(NULL),
    m_pRenderTargetMemoryPool(NULL),
    m_pWindow(NULL)
{
    for (int i = 0; i < g_NumColorBuffers; ++i)
    {
        m_RenderTargets[i] = NULL;
    }
}

/*
* GettingStartedWithNVN Destructor
* --------------------------------
* Empty destructor.
*/
GettingStartedWithNVN::~GettingStartedWithNVN()
{
}
#ifndef _WIN32
//#define NVN_GRAPHICS_DEBUGGER
#endif

#ifdef NVN_GRAPHICS_DEBUGGER
extern "C" PFNNVNGENERICFUNCPTRPROC nvnDeviceGetToolsProcAddress(NVNdevice *device, const char* name);
#endif

/*
* GettingStartedWithNVN::Init
* ---------------------------
* This method demonstrates basic NVN initialization. The bootstrap loader
* function pointer is obtained in the platform-specific base class.
*/
void GettingStartedWithNVN::Init(PFNNVNBOOTSTRAPLOADERPROC loader, NVNnativeWindow nativeWindow)
{
    NN_ASSERT(loader != NULL, "Bootstrap loader function pointer is NULL\n");

#ifdef NVN_GRAPHICS_DEBUGGER
    /* Load procs from NVNGD instead of from the driver, this must be called once prior to nvnDeviceInitialize. */
    nvnLoadCProcs(nullptr, (PFNNVNDEVICEGETPROCADDRESSPROC)nvnDeviceGetToolsProcAddress);

    NVNdeviceBuilder deviceBuilder;
    nvnDeviceBuilderSetDefaults(&deviceBuilder);
    int deviceFlags = NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_4_BIT;
    nvnDeviceBuilderSetFlags(&deviceBuilder, deviceFlags);

    if (nvnDeviceInitialize(&m_Device, &deviceBuilder) == false)
    {
        NN_ASSERT(0, "nvnDeviceInitialize");
    }
    /* Never call nvnLoadCProcs after nvnDeviceInitialize. */
#else /* NVN_GRAPHICS_DEBUGGER */
    pfnc_nvnDeviceInitialize = reinterpret_cast<PFNNVNDEVICEINITIALIZEPROC>((*loader)("nvnDeviceInitialize"));
    pfnc_nvnDeviceGetProcAddress = reinterpret_cast<PFNNVNDEVICEGETPROCADDRESSPROC>((*loader)("nvnDeviceGetProcAddress"));
    if (!pfnc_nvnDeviceInitialize)
    {
        /* This can happen if an NVN driver is not installed on a Windows PC. */
        NN_ASSERT(0, "BootstrapLoader failed to find nvnDeviceInitialize");
    }

    nvnLoadCProcs(NULL, pfnc_nvnDeviceGetProcAddress);

    int MajorVersion, MinorVersion;
    nvnDeviceGetInteger(NULL, NVN_DEVICE_INFO_API_MAJOR_VERSION, &MajorVersion);
    nvnDeviceGetInteger(NULL, NVN_DEVICE_INFO_API_MINOR_VERSION, &MinorVersion);

    if (MajorVersion != NVN_API_MAJOR_VERSION || MinorVersion < NVN_API_MINOR_VERSION)
    {
        NN_ASSERT(0, "NVN SDK not supported by current driver.");
    }

    /* If debug or develop is enabled, turn on NVN's debug layer. */
    int deviceFlags = 0;
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    deviceFlags = NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_4_BIT;
#endif

    NVNdeviceBuilder deviceBuilder;
    nvnDeviceBuilderSetDefaults(&deviceBuilder);
    nvnDeviceBuilderSetFlags(&deviceBuilder, deviceFlags);

    if (nvnDeviceInitialize(&m_Device, &deviceBuilder) == false)
    {
        /*
        * This can fail for a few reasons; the most likely on Horizon is
        * insufficent device memory.
        */
        NN_ASSERT(0, "nvnDeviceInitialize");
    }

    /*
    * API Loading
    * -----------
    * Once the device has been initialized, it is now possible to load all of the
    * remaining NVN API entry points. The bootstrap loader is used again to
    * locate the "real" loader function. The real loader may be passed along
    * with the device into nvnLoadProcs if using nvn_FuncPtrInline.h. It is also
    * completely valid for applications to load with their own entry points.
    */
    nvnLoadCProcs(&m_Device, pfnc_nvnDeviceGetProcAddress);
#endif /* NVN_GRAPHICS_DEBUGGER */

    if (deviceFlags & NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_4_BIT)
    {
        nvnDeviceInstallDebugCallback(
            &m_Device,
            reinterpret_cast<PFNNVNDEBUGCALLBACKPROC>(&DebugLayerCallback),
            NULL, // For testing purposes; any pointer is OK here.
            NVN_TRUE // NVN_TRUE = Enable the callback.
        );
    }

    NVNqueueBuilder queueBuilder;
    nvnQueueBuilderSetDevice(&queueBuilder, &m_Device);
    nvnQueueBuilderSetDefaults(&queueBuilder);
    nvnQueueBuilderSetComputeMemorySize(&queueBuilder, 0);

    int minQueueCommandMemorySize = 0;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_QUEUE_COMMAND_MEMORY_MIN_SIZE, &minQueueCommandMemorySize);
    nvnQueueBuilderSetCommandMemorySize(&queueBuilder, minQueueCommandMemorySize);

    nvnQueueBuilderSetCommandFlushThreshold(&queueBuilder, minQueueCommandMemorySize);

    size_t neededQueueMemorySize = nvnQueueBuilderGetQueueMemorySize(&queueBuilder);

    if ((neededQueueMemorySize % NVN_MEMORY_POOL_STORAGE_GRANULARITY) != 0)
    {
        NN_ASSERT(0, "Memory size reported for queue is not the proper granularity");
    }

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
    m_pQueueMemory = NULL;
#else
    m_pQueueMemory = AlignedAllocate(neededQueueMemorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
#endif

    nvnQueueBuilderSetQueueMemory(&queueBuilder, m_pQueueMemory, neededQueueMemorySize);

    if (nvnQueueInitialize(&m_Queue, &queueBuilder) == false)
    {
        NN_ASSERT(0, "nvnQueueInitialize failed");
    }

    if (!nvnCommandBufferInitialize(&m_CommandBuffer, &m_Device))
    {
        NN_ASSERT(0, "nvnCommandBufferInitialize");
    }

    int commandBufferCommandAlignment = 0;
    int commandBufferControlAlignment = 0;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_COMMAND_BUFFER_COMMAND_ALIGNMENT, &commandBufferCommandAlignment);
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_COMMAND_BUFFER_CONTROL_ALIGNMENT, &commandBufferControlAlignment);

    /* Setup the command buffer memory pool. */
    m_pCommandMemoryPool = new MemoryPool();
    m_pCommandMemoryPool->Init(
        NULL,
        Align(g_CommandMemorySize, commandBufferCommandAlignment) * 2,
        NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
        &m_Device);

    /* Setup the control memory with the proper alignment. */
    m_pControlPool = AlignedAllocate(g_ControlMemorySize, commandBufferControlAlignment);

    /* Grab the offset from the memory pool. */
    m_CommandPoolOffset = m_pCommandMemoryPool->GetNewMemoryChunkOffset(g_CommandMemorySize, commandBufferCommandAlignment);

    /* Provides the command buffer with the command and control memory blocks. */
    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_CommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, g_ControlMemorySize);

    /* Render Target Setting Command Buffer */
    if (!nvnCommandBufferInitialize(&m_RenderTargetCommandBuffer, &m_Device))
    {
        NN_ASSERT(0, "nvnCommandBufferInitialize");
    }

    /* Setup the control memory with the proper alignment. */
    m_pRenderTargetControlPool = AlignedAllocate(g_ControlMemorySize, commandBufferControlAlignment);

    /* Grab the offset from the memory pool. */
    m_RenderTargetCommandPoolOffset = m_pCommandMemoryPool->GetNewMemoryChunkOffset(g_CommandMemorySize, commandBufferCommandAlignment);

    nvnCommandBufferAddCommandMemory(&m_RenderTargetCommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_RenderTargetCommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_RenderTargetCommandBuffer, m_pRenderTargetControlPool, g_ControlMemorySize);

    if (!nvnSyncInitialize(&m_CommandBufferSync, &m_Device))
    {
        NN_ASSERT(0, "nvnSyncInitialize");
    }

    nvnTextureBuilderSetDevice(&m_RenderTargetBuilder, &m_Device);
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);

    /* Render targets that need to be displayed to the screen need both the display access and compressible bit. */
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT | NVN_TEXTURE_FLAGS_LINEAR_RENDER_TARGET_BIT);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1920, 1080);
    nvnTextureBuilderSetStride(&m_RenderTargetBuilder, 1920 * 4);
    m_ColorTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);

    m_pRenderTargetMemoryPool = new MemoryPool();
    m_pRenderTargetMemoryPool->Init(
        NULL,
        m_ColorTargetSize * g_NumColorBuffers,
        NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
        &m_Device);

    nvnWindowBuilderSetDefaults(&m_WindowBuilder);
    nvnWindowBuilderSetDevice(&m_WindowBuilder, &m_Device);
    nvnWindowBuilderSetNativeWindow(&m_WindowBuilder, nativeWindow);
}//NOLINT(impl/function_size)

 /*
 * GettingStartedWithNVN::Shutdown
 * -------------------------------
 * This method cleans up all nvn objects and dynamically allocated memory.
 */
void GettingStartedWithNVN::Shutdown()
{
    nvnQueueFinish(&m_Queue);

    nvnSyncFinalize(&m_CommandBufferSync);

    nvnCommandBufferFinalize(&m_RenderTargetCommandBuffer);

    if (m_pRenderTargetControlPool)
    {
        AlignedDeallocate(m_pRenderTargetControlPool);
        m_pRenderTargetControlPool = NULL;
    }

    nvnCommandBufferFinalize(&m_CommandBuffer);

    if (m_pCommandMemoryPool)
    {
        m_pCommandMemoryPool->Shutdown();
        delete m_pCommandMemoryPool;
        m_pCommandMemoryPool = NULL;
    }

    if (m_pControlPool)
    {
        AlignedDeallocate(m_pControlPool);
        m_pControlPool = NULL;
    }

    if (m_pWindow)
    {
        nvnWindowFinalize(m_pWindow);
        delete m_pWindow;
        m_pWindow = NULL;
    }

    for (int i = 0; i < g_NumColorBuffers; ++i)
    {
        if (m_RenderTargets[i])
        {
            nvnTextureFinalize(m_RenderTargets[i]);
            delete m_RenderTargets[i];
            m_RenderTargets[i] = NULL;
        }
    }

    if (m_pRenderTargetMemoryPool)
    {
        m_pRenderTargetMemoryPool->Shutdown();
        delete m_pRenderTargetMemoryPool;
        m_pRenderTargetMemoryPool = NULL;
    }

    nvnQueueFinalize(&m_Queue);
    if (m_pQueueMemory)
    {
        AlignedDeallocate(m_pQueueMemory);
        m_pQueueMemory = NULL;
    }
    nvnDeviceFinalize(&m_Device);
}

/*
* GettingStartedWithNVN::Resize
* -----------------------------
* This method is called everytime the window is resized and is passed
* the new size of the window. It frees the old render target and creates a new
* render target with the new screen size. A new set of commands for the command buffer
* is recorded that performs four scissored clears of different colors on the target.
*/
void GettingStartedWithNVN::Resize(int width, int height)
{
    /* Check for the window being minimized or having no visible surface. */
    if (width == 0 || height == 0)
    {
        return;
    }

    nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);

    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

    /* If it's the first time Resize is called, allocate the NVNwindow. */
    if (m_pWindow == NULL)
    {
        m_pWindow = new NVNwindow;
    }
    else
    {
        nvnWindowFinalize(m_pWindow);
    }

    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, width, height);

    for (int i = 0; i < g_NumColorBuffers; ++i)
    {
        /* If it's the first time Resize is called, allocate the texture. */
        if (!m_RenderTargets[i])
        {
            m_RenderTargets[i] = new NVNtexture;
        }
        else
        {
            nvnTextureFinalize(m_RenderTargets[i]);
        }

        /* Set the memory pool the texture is to be allocated from and offset to where it will be located. */
        nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, m_pRenderTargetMemoryPool->GetMemoryPool(), m_ColorTargetSize * i);

        /* Create the texture using the current state of the texture builder. */
        NN_ASSERT(nvnTextureInitialize(m_RenderTargets[i], &m_RenderTargetBuilder));
    }

    /*
    *  Sets the number of render targets that the NVNwindow will have available and
    *  gives a pointer to an array of those targets.
    */
    nvnWindowBuilderSetTextures(&m_WindowBuilder, g_NumColorBuffers, m_RenderTargets);
    NN_ASSERT(nvnWindowInitialize(m_pWindow, &m_WindowBuilder));

    /* Clear Colors */
    float black[4] = { 0.0f, 0.0f, 0.0f, 1.0f };
    float   red[4] = { 1.0f, 0.0f, 0.0f, 1.0f };
    float green[4] = { 0.0f, 1.0f, 0.0f, 1.0f };
    float  blue[4] = { 0.0f, 0.0f, 1.0f, 1.0f };

    /* Temporary variables for size/position of cleared rects. */
    int halfWidth = static_cast<int>(ceil(width / 2.0f));
    int halfHeight = static_cast<int>(ceil(height / 2.0f));

    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_CommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, g_ControlMemorySize);

    /* Starts the recording of a new set of commands for the given command buffer. */
    nvnCommandBufferBeginRecording(&m_CommandBuffer);

    {
        nvnCommandBufferSetScissor(&m_CommandBuffer, 0, 0, halfWidth, halfHeight);

        /*
        * Clears the currently set render target at a given index.
        * Channels to be cleared are set through the clear color mask.
        */
        nvnCommandBufferClearColor(&m_CommandBuffer, 0, black, NVN_CLEAR_COLOR_MASK_RGBA);

        /* Lower right quadrant - red */
        nvnCommandBufferSetScissor(&m_CommandBuffer, halfWidth, 0, width - halfWidth, halfHeight);
        nvnCommandBufferClearColor(&m_CommandBuffer, 0, red, NVN_CLEAR_COLOR_MASK_RGBA);

        /* Upper right quadrant - green */
        nvnCommandBufferSetScissor(&m_CommandBuffer, halfWidth, halfHeight, width - halfWidth, height - halfHeight);
        nvnCommandBufferClearColor(&m_CommandBuffer, 0, green, NVN_CLEAR_COLOR_MASK_RGBA);

        /* Upper left quadrant - blue */
        nvnCommandBufferSetScissor(&m_CommandBuffer, 0, halfHeight, halfWidth, height - halfHeight);
        nvnCommandBufferClearColor(&m_CommandBuffer, 0, blue, NVN_CLEAR_COLOR_MASK_RGBA);
    }

    /*
    * Ends the recording of commands for the command buffer. Returns a
    * command handle to the compiled set of commands that can be submitted
    * to a queue or used as a part of another command buffer.
    */
    m_CommandHandle = nvnCommandBufferEndRecording(&m_CommandBuffer);

#ifdef CHECK_COMMAND_BUFFER_MEMORY_USAGE
    size_t commandMemoryUsed = nvnCommandBufferGetCommandMemoryUsed(&m_CommandBuffer);
    size_t controlMemoryUsed = nvnCommandBufferGetControlMemoryUsed(&m_CommandBuffer);
#endif
}

/*
* GettingStartedWithNVN::Draw
* ---------------------------
* This method submits the commands recorded in the command buffers to the queue
* and presents the current render target to the screen.
*/
void GettingStartedWithNVN::Draw(uint64_t /*millisec*/)
{
    /*
    * Update the command buffer to set the current render target and grab
    * the index to that target.
    */
    int index = UpdateRenderTargets();

    /* Render target setting command buffer. */
    nvnQueueSubmitCommands(&m_Queue, 1, &m_RenderTargetCommandHandle);

    /* Scissored clears command buffer, created/update by the Resize function. */
    nvnQueueSubmitCommands(&m_Queue, 1, &m_CommandHandle);

    /* Displays the render target at the given index to the screen. */
    nvnQueuePresentTexture(&m_Queue, m_pWindow, index);
}

/*
* GettingStartedWithNVN::UpdateRenderTargets
* ------------------------------------------
* Gets the index of the current render target from the NVNwindow
* and records a command buffer that sets it up to be rendered to.
*/
int GettingStartedWithNVN::UpdateRenderTargets()
{
    int bufferIndex;

    nvnQueueAcquireTexture(&m_Queue, m_pWindow, &bufferIndex);

    nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

    nvnCommandBufferAddCommandMemory(&m_RenderTargetCommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_RenderTargetCommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_RenderTargetCommandBuffer, m_pRenderTargetControlPool, g_ControlMemorySize);

    nvnCommandBufferBeginRecording(&m_RenderTargetCommandBuffer);
    nvnCommandBufferSetRenderTargets(&m_RenderTargetCommandBuffer, 1, &m_RenderTargets[bufferIndex], NULL, NULL, NULL);
    m_RenderTargetCommandHandle = nvnCommandBufferEndRecording(&m_RenderTargetCommandBuffer);

    /* Return the index acquired from the NVNwindow so it can be used in the next present call. */
    return bufferIndex;
}

// static
void GettingStartedWithNVN::DebugLayerCallback(
    NVNdebugCallbackSource source,
    NVNdebugCallbackType type,
    int id,
    NVNdebugCallbackSeverity severity,
    const char* message,
    void* pUser
)
{
    NN_ASSERT(pUser == NULL);

    NN_LOG("NVN Debug Layer Callback:\n");
    NN_LOG("  source:       0x%08x\n", source);
    NN_LOG("  type:         0x%08x\n", type);
    NN_LOG("  id:           0x%08x\n", id);
    NN_LOG("  severity:     0x%08x\n", severity);
    NN_LOG("  message:      %s\n", message);

    //NN_ASSERT(0, "NVN Debug layer callback hit");
}

TutorialBaseClass* t()
{
    static GettingStartedWithNVN tut;
    return (&tut);
}

extern "C" void nnMain()
{
    TutorialRun();
}
