﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{TutorialBaseClass.Windows.cpp,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This class is shared among all tutorials in order to abstract away per-operating system
 *  details related to NVN initialization and operating system integration. Each tutorial
 *  derives from the TutorialBaseClass, and provides an implementation of the t()
 *  global singleton accessor function.
 */


#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#include <nn/vi.h>
#include <nn/hid.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/nn_Log.h>

#include <nn/fs.h>
#include <nn/nn_Assert.h>

#include <mmsystem.h>
#include <exception>
#include <stdexcept>
#include <tchar.h>

#include <nvngdSupport/TutorialBaseClass.h>

    // Force the NVidia GPU on Optimus systems
extern "C"
{
    _declspec(dllexport) DWORD NvOptimusEnablement = 0x00000001;
}

namespace {
    void* Allocate(size_t size)
    {
        return std::malloc(size);
    }

    void Deallocate(void* p, size_t size)
    {
        NN_UNUSED(size);
        std::free(p);
    }
} // namespace

class Tutorial
{
    public:
        explicit Tutorial(HINSTANCE hinstance);
        ~Tutorial();

        int Run();
        int TestRun();

    private:

        static int const totalFrameTime = 16; // 60fps.
        // If there is less than this many milliseconds in a frame, just proceed to the next.
        static int const padFrameTime = 2;

        void InitWindow();
        void InitOpenGL();
        void InitNVN();

        void Draw(uint64_t millisec);
        bool TestDraw(unsigned long long millisec);

        HINSTANCE m_Hinstance;
        HWND      m_Hwnd;
        HDC       m_Hdc;
        HGLRC     m_Hglrc;

        nn::vi::Display* m_pDisplay;
        nn::vi::Layer* m_pLayer;
        std::exception_ptr m_exception;
};

Tutorial::Tutorial(HINSTANCE hinstance)
    : m_Hinstance(hinstance)
    , m_Hwnd(0)
    , m_Hdc(0)
    , m_Hglrc(0)
    , m_pDisplay(0)
    , m_pLayer(0)
{
    InitWindow();
    InitOpenGL();
    InitNVN();

    t()->Resize(1280, 720);
}

Tutorial::~Tutorial()
{
    t()->Shutdown();

    if (m_Hglrc != NULL)
    {
        wglMakeCurrent(0, 0);
        wglDeleteContext(m_Hglrc);
        m_Hglrc = 0;
    }

    if (m_Hdc != NULL)
    {
        NN_ASSERT(m_Hwnd);
        ReleaseDC(m_Hwnd, m_Hdc);
        m_Hdc = 0;
    }

    nn::vi::DestroyLayer(m_pLayer);
    nn::vi::CloseDisplay(m_pDisplay);
    nn::vi::Finalize();

    timeEndPeriod(1);
}

int Tutorial::Run()
{
    MSG msg = { 0 };

    uint32_t frameTime = 0;
    uint32_t frameTimePrev = timeGetTime();
    int sleepTime = 0;

    bool runLoop = true;
    bool doFrame = true;
    bool doSleep = true;

        /* Initialize keyboard for user exit */
    nn::hid::InitializeKeyboard();
    nn::hid::KeyboardState keyboardState;

    NN_LOG("--------------------------------------------------------------------------------\n");
    NN_LOG("Hitting Esc on Windows or touching the Touch Screen on NX will exit the program.\n");
    NN_LOG("--------------------------------------------------------------------------------\n");

    while (runLoop)
    {
        if (doFrame)
        {
            frameTime = timeGetTime();
            NN_ASSERT(frameTime >= frameTimePrev);
            Draw(frameTime - frameTimePrev);
            frameTimePrev = frameTime;

            doFrame = false;
        }

        doSleep = runLoop;
        if (doSleep)
        {
                // Figure out how much time to sleep off, if any
            sleepTime = totalFrameTime - static_cast<int>(timeGetTime() - frameTime);

            if (sleepTime <= padFrameTime)
            {
                doSleep = false;
                doFrame = true;
            }
        }

        if (doSleep)
        {
            switch (MsgWaitForMultipleObjects(
                0, 0, // No kernel object array.
                FALSE,
                sleepTime,
                QS_ALLEVENTS
                ))
            {
            case WAIT_OBJECT_0:
                break;

            case WAIT_TIMEOUT:
                doFrame = true;
                break;

            default:
            case WAIT_FAILED:
                throw std::runtime_error("MsgWaitForMultipleObjects");
            }
        }

            /*
             * If the escape key has been hit, break out of the run loop
             * and clean up.
             */
        nn::hid::GetKeyboardState(&keyboardState);
        if (keyboardState.keys.Test<::nn::hid::KeyboardKey::Escape>())
        {
            break;
        }
    }

    return (static_cast<int>(msg.wParam));
}

int Tutorial::TestRun()
{
    MSG msg = { 0 };

    uint32_t frameTime = 0;
    uint32_t frameTimePrev = timeGetTime();
    int sleepTime = 0;

    bool runLoop = true;
    bool doFrame = true;
    bool doSleep = true;

    /* Initialize keyboard for user exit */
    //nn::hid::InitializeKeyboard();
    //nn::hid::KeyboardState keyboardState;

    while (runLoop)
    {
        if (doFrame)
        {
            frameTime = timeGetTime();

            NN_ASSERT(frameTime >= frameTimePrev);
            runLoop = TestDraw(0);
            frameTimePrev = frameTime;

            doFrame = false;
        }

        doSleep = runLoop;
        if (doSleep)
        {
            // Figure out how much time to sleep off, if any
            sleepTime = totalFrameTime - static_cast<int>(timeGetTime() - frameTime);

            if (sleepTime <= padFrameTime)
            {
                doSleep = false;
                doFrame = true;
            }
        }

        if (doSleep)
        {
            switch (MsgWaitForMultipleObjects(
                0, 0, // No kernel object array.
                FALSE,
                sleepTime,
                QS_ALLEVENTS
            ))
            {
            case WAIT_OBJECT_0:
                break;

            case WAIT_TIMEOUT:
                doFrame = true;
                break;

            default:
            case WAIT_FAILED:
                throw std::runtime_error("MsgWaitForMultipleObjects");
            }
        }

        /*
        * If the escape key has been hit, break out of the run loop
        * and clean up.
        */
        //nn::hid::GetKeyboardState(&keyboardState);
        //if (keyboardState.keys.Test<::nn::hid::KeyboardKey::Escape>())
        //{
        //    break;
        //}
        doFrame = true;
    }

    return (static_cast<int>(msg.wParam));
}


void Tutorial::InitWindow()
{
    nn::vi::Initialize();

    if (timeBeginPeriod(1) != TIMERR_NOERROR)
    {
        throw std::runtime_error("timeBeginPeriod");
    }

    nn::vi::OpenDefaultDisplay(&m_pDisplay);

    nn::vi::CreateLayer(&m_pLayer, m_pDisplay);

    nn::vi::GetNativeWindow(reinterpret_cast<nn::vi::NativeWindowHandle*>(&m_Hwnd), m_pLayer);

    if (!SetWindowLongPtr(m_Hwnd, GWLP_USERDATA, reinterpret_cast<LONG_PTR>(this)) &&
        GetLastError())
    {
        throw std::runtime_error("SetWindowLongPtr");
    }
}

void Tutorial::InitOpenGL()
{
    NN_ASSERT(m_Hwnd && !m_Hdc);
    m_Hdc = GetDC(m_Hwnd);

    if (m_Hdc == NULL)
    {
        throw std::runtime_error("GetDC");
    }

    PIXELFORMATDESCRIPTOR pfd = { 0 };
    pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
    pfd.nVersion = 1;
    pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
    pfd.iPixelType = PFD_TYPE_RGBA;
    pfd.cColorBits = 32;
    int selectedPixelFormat = ChoosePixelFormat(m_Hdc, &pfd);

    if (!SetPixelFormat(m_Hdc, selectedPixelFormat, &pfd))
    {
        throw std::runtime_error("SetPixelFormat");
    }

    m_Hglrc = wglCreateContext(m_Hdc);

    if (m_Hglrc == NULL)
    {
        throw std::runtime_error("wglCreateContext");
    }

        // A current OpenGL context is required on the thread that will call
        // nvnQueuePresentTexture in Windows.
    if (!wglMakeCurrent(m_Hdc, m_Hglrc))
    {
        throw std::runtime_error("wglMakeCurrent");
    }
}

bool TutorialBaseClass::TestDraw(uint64_t millisec)
{
    NN_UNUSED(millisec);
    return true;
}

bool Tutorial::TestDraw(unsigned long long millisec)
{
    return t()->TestDraw(millisec);
}

void Tutorial::InitNVN()
{
        // The NVN loader is exposed as a obfuscated OpenGL extension in Windows
    t()->Init(reinterpret_cast<PFNNVNBOOTSTRAPLOADERPROC>(wglGetProcAddress("rq34nd2ffz")), m_Hwnd);
}

void Tutorial::Draw(uint64_t millisec)
{
    t()->Draw(millisec);
}

void TutorialStartup()
{
}

void TutorialRun(bool test)
{
    HINSTANCE hInstance = GetModuleHandle(NULL);
    nn::fs::SetAllocator(Allocate, Deallocate);
    Tutorial tutorial(hInstance);

    if (test)
    {
        tutorial.TestRun();
    }
    else
    {
        tutorial.Run();
    }

    return;
}
